/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.salary.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.util.DateUtil;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TStatisticsTaxSalary;
import com.yifu.cloud.plus.v1.yifu.salary.service.TStatisticsTaxSalaryService;
import com.yifu.cloud.plus.v1.yifu.salary.service.TStatisticsTaxSalaryTwoService;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TStatisticsTaxSalarySearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;


/**
 * 统计-工资薪金
 *
 * @author hgw
 * @date 2022-08-05 11:40:14
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tstatisticstaxsalary")
@Tag(name = "统计-工资薪金管理")
public class TStatisticsTaxSalaryController {

	private final TStatisticsTaxSalaryService tStatisticsTaxSalaryService;
	private final TStatisticsTaxSalaryTwoService tStatisticsTaxSalaryTwoService;


	/**
	 * 简单分页查询
	 *
	 * @param page                 分页对象
	 * @param tStatisticsTaxSalary 统计-工资薪金
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TStatisticsTaxSalary>> getTStatisticsTaxSalaryPage(Page<TStatisticsTaxSalary> page, TStatisticsTaxSalarySearchVo tStatisticsTaxSalary) {
		return new R<>(tStatisticsTaxSalaryService.getTStatisticsTaxSalaryPage(page, tStatisticsTaxSalary));
	}

	/**
	 * 通过id查询统计-工资薪金
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('salary_tstatisticstaxsalary_get')")
	@GetMapping("/{id}")
	public R<TStatisticsTaxSalary> getById(@PathVariable("id") String id) {
		return R.ok(tStatisticsTaxSalaryService.getById(id));
	}

	/**
	 * 通过id删除统计-工资薪金
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除统计-工资薪金", description = "通过id删除统计-工资薪金：hasPermission('salary_tstatisticstaxsalary_del')")
	@SysLog("通过id删除统计-工资薪金")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('salary_tstatisticstaxsalary_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tStatisticsTaxSalaryService.removeById(id));
	}

	/**
	 * 统计-工资薪金 批量导出
	 *
	 * @author hgw
	 * @date 2022-08-05 11:40:14
	 **/
	@Operation(description = "导出统计-工资薪金 hasPermission('salary_tstatisticstaxsalary-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('salary_tstatisticstaxsalary-export')")
	public void export(HttpServletResponse response, @RequestBody TStatisticsTaxSalarySearchVo searchVo) {
		tStatisticsTaxSalaryService.listExport(response, searchVo);
	}

	/**
	 * @Description: 定时任务-年度报账+工资薪金+本期申报
	 * @Author: hgw
	 * @Date: 2022-8-23 16:21:08
	 * @return: com.yifu.cloud.v1.common.core.util.R<java.lang.String>
	 **/
	@Inner
	@Operation(description = "定时任务-年度报账+工资薪金+本期申报")
	@PostMapping("/inner/doRefreshStatisticsTaxAuto")
	public void doRefreshStatisticsTaxAuto() {
		// 2022-8-31 16:04:22同徐梅、简洁改为申报月，实际数据为上个结算月的数据
		String declareMonth = DateUtil.addMonth(0);
		tStatisticsTaxSalaryTwoService.doRefreshStatisticsTaxAuto(declareMonth);
	}

	/**
	 * @param settleMonth 结算月
	 * @Description: 手动刷新-年度报账+工资薪金+本期申报
	 * @Author: hgw
	 * @Date: 2022/2/22 16:23
	 * @return: com.yifu.cloud.v1.common.core.util.R<java.lang.String>
	 **/
	@Operation(description = "手动刷新-年度报账+工资薪金+本期申报（结算月：settleMonth必填）hasPermission('salary_tstatisticstaxsalary_refresh')")
	@PostMapping("/doRefreshStatisticsTaxAll")
	@PreAuthorize("@pms.hasPermission('salary_tstatisticstaxsalary_refresh')")
	public R<String> doRefreshStatisticsTaxAll(@RequestParam String settleMonth) {
		// 2022-8-31 16:04:22同徐梅、简洁改为申报月，实际数据为上个结算月的数据
		return tStatisticsTaxSalaryService.doRefreshStatisticsTaxAll(settleMonth);
	}

}
