/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysUser;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.DateUtil;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.mybatis.base.BaseEntity;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import com.yifu.cloud.plus.v1.yifu.insurances.entity.TInsuranceTypeRate;
import com.yifu.cloud.plus.v1.yifu.social.entity.TSocialDeadlineInfo;
import com.yifu.cloud.plus.v1.yifu.social.service.TSocialDeadlineInfoService;
import com.yifu.cloud.plus.v1.yifu.social.vo.TSocialDeadlineInfoSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.time.LocalDateTime;
import java.util.List;


/**
 * 社保医疗截止日配置
 *
 * @author fxj
 * @date 2024-05-08 11:52:09
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tsocialdeadlineinfo")
@Tag(name = "社保医疗截止日配置管理")
public class TSocialDeadlineInfoController {

	private final TSocialDeadlineInfoService tSocialDeadlineInfoService;


	/**
	 * 简单分页查询
	 *
	 * @param page                分页对象
	 * @param tSocialDeadlineInfo 社保医疗截止日配置
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TSocialDeadlineInfo>> getTSocialDeadlineInfoPage(Page<TSocialDeadlineInfo> page, TSocialDeadlineInfoSearchVo tSocialDeadlineInfo) {
		return new R<>(tSocialDeadlineInfoService.getTSocialDeadlineInfoPage(page, tSocialDeadlineInfo));
	}

	/**
	 * 不分页查询
	 *
	 * @param tSocialDeadlineInfo 社保医疗截止日配置
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	public R<List<TSocialDeadlineInfo>> getTSocialDeadlineInfoNoPage(@RequestBody TSocialDeadlineInfoSearchVo tSocialDeadlineInfo) {
		return R.ok(tSocialDeadlineInfoService.noPageDiy(tSocialDeadlineInfo));
	}

	/**
	 * 通过id查询社保医疗截止日配置
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('demo_tsocialdeadlineinfo_get')")
	@GetMapping("/{id}")
	public R<TSocialDeadlineInfo> getById(@PathVariable("id") String id) {
		return R.ok(tSocialDeadlineInfoService.getById(id));
	}

	/**
	 * 新增社保医疗截止日配置
	 *
	 * @param tSocialDeadlineInfo 社保医疗截止日配置
	 * @return R
	 */
	@Operation(summary = "新增社保医疗截止日配置", description = "新增社保医疗截止日配置：hasPermission('demo_tsocialdeadlineinfo_add')")
	@SysLog("新增社保医疗截止日配置")
	@PostMapping
	public R<Boolean> save(@RequestBody TSocialDeadlineInfo tSocialDeadlineInfo) {

		return tSocialDeadlineInfoService.saveAsso(tSocialDeadlineInfo);
	}



	/**
	 * 修改社保医疗截止日配置
	 *
	 * @param tSocialDeadlineInfo 社保医疗截止日配置
	 * @return R
	 */
	@Operation(summary = "修改社保医疗截止日配置", description = "修改社保医疗截止日配置：hasPermission('demo_tsocialdeadlineinfo_edit')")
	@SysLog("修改社保医疗截止日配置")
	@PutMapping
	public R<Boolean> updateById(@RequestBody TSocialDeadlineInfo tSocialDeadlineInfo) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.UPDATE_DATA_FAIL);
		}
		tSocialDeadlineInfo.setUpdateName(user.getNickname());
		tSocialDeadlineInfo.setUpdateBy(user.getId());
		tSocialDeadlineInfo.setUpdateTime(LocalDateTime.now());
		return tSocialDeadlineInfoService.updateByIdAsso(tSocialDeadlineInfo);
	}

	/**
	 * 通过id删除社保医疗截止日配置
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除社保医疗截止日配置", description = "通过id删除社保医疗截止日配置：hasPermission('demo_tsocialdeadlineinfo_del')")
	@SysLog("通过id删除社保医疗截止日配置")
	@DeleteMapping("/{id}")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tSocialDeadlineInfoService.removeById(id));
	}


	/**
	 * 通过id启用/禁用社保医疗截止日配置
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id启用0/禁用1社保医疗截止日配置", description = "通过id启用0/禁用1社保医疗截止日配置")
	@SysLog("通过id启用/禁用社保医疗截止日配置")
	@PutMapping("/updateStatusById")
	public R<Boolean> updateStatusById(@RequestParam String id,@RequestParam String status) {
		YifuUser user = SecurityUtils.getUser();
		if (!Common.isNotNull(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		if (!CommonConstants.ZERO_STRING.equals(status) && !CommonConstants.ONE_STRING.equals(status)){
			return R.failed(CommonConstants.PARAM_INFO_ERROR);
		}
		LambdaUpdateWrapper<TSocialDeadlineInfo> updateWrapper = new LambdaUpdateWrapper<>();
		updateWrapper.eq(TSocialDeadlineInfo::getId,id)
				.set(TSocialDeadlineInfo::getStatus, status)
				.set(BaseEntity::getUpdateTime, DateUtil.getCurrentDateTime())
				.set(TSocialDeadlineInfo::getUpdateName, user.getNickname())
				.set(BaseEntity::getUpdateBy,user.getId());
		return R.ok(tSocialDeadlineInfoService.update(updateWrapper));
	}

	/**
	 * 社保医疗截止日配置 批量导入
	 *
	 * @author fxj
	 * @date 2024-05-08 11:52:09
	 **/
	@SneakyThrows
	@Operation(description = "批量新增社保医疗截止日配置 hasPermission('demo_tsocialdeadlineinfo-batch-import')")
	@SysLog("批量新增社保医疗截止日配置")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('demo_tsocialdeadlineinfo-batch-import')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file) {
		return tSocialDeadlineInfoService.importDiy(file.getInputStream());
	}

	/**
	 * 社保医疗截止日配置 批量导出
	 *
	 * @author fxj
	 * @date 2024-05-08 11:52:09
	 **/
	@Operation(description = "导出社保医疗截止日配置 hasPermission('demo_tsocialdeadlineinfo-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('demo_tsocialdeadlineinfo-export')")
	public void export(HttpServletResponse response, @RequestBody TSocialDeadlineInfoSearchVo searchVo) {
		tSocialDeadlineInfoService.listExport(response, searchVo);
	}

	/**
	 * @Description: 定时任务生成月度截止日配置信息
	 * @Author: fxj
	 * @Date: 2024/5/8
	 **/
	@Operation(summary = "定时任务生成月度截止日配置信息", description = "定时任务生成月度截止日配置信息")
	@SysLog("定时任务生成月度截止日配置信息")
	@Inner
	@PostMapping("/inner/createDeadLineSetInfo")
	public void createDeadLineSetInfo() {
		tSocialDeadlineInfoService.createDeadLineSetInfo();
	}
}
