/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.archives.service.impl;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.archives.entity.LGuaziOfferRecord;
import com.yifu.cloud.plus.v1.yifu.archives.mapper.LGuaziOfferRecordMapper;
import com.yifu.cloud.plus.v1.yifu.archives.service.LGuaziOfferRecordService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.LGuaziOfferRecordVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.lang.ArrayUtils;
import org.springframework.stereotype.Service;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.text.MessageFormat;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 瓜子offer操作日志
 *
 * @author chenyx
 * @date 2025-06-11 15:59:04
 */
@Log4j2
@Service
@RequiredArgsConstructor
public class LGuaziOfferRecordServiceImpl extends ServiceImpl<LGuaziOfferRecordMapper, LGuaziOfferRecord> implements LGuaziOfferRecordService {
    /**
     * 瓜子offer操作日志简单分页查询
     * @param lGuaziOfferRecord 瓜子offer操作日志
     * @return
     */
    @Override
    public IPage<LGuaziOfferRecord> getLGuaziOfferRecordPage(Page<LGuaziOfferRecord> page, LGuaziOfferRecord lGuaziOfferRecord){
        return baseMapper.getLGuaziOfferRecordPage(page,lGuaziOfferRecord);
    }

	/**
	 * 单个增加 操作日志
	 *
	 * @param lGuaziOfferRecordVo 瓜子offer操作日志包装类
	 */
	@Override
	public void saveRecordOne(LGuaziOfferRecordVo lGuaziOfferRecordVo, String cancelTaskName) {
		if(Common.isNotNull(lGuaziOfferRecordVo.getOfferId())){
			// 生成操作内容
			handleContent(lGuaziOfferRecordVo);
			if (Common.isNotNull(cancelTaskName)) {
				if (Common.isNotNull(lGuaziOfferRecordVo.getContent())) {
					lGuaziOfferRecordVo.setContent(lGuaziOfferRecordVo.getContent() + "；" + cancelTaskName);
				} else {
					lGuaziOfferRecordVo.setContent(cancelTaskName);
				}
			}
			lGuaziOfferRecordVo.setCreateTime(LocalDateTime.now());
			this.save(lGuaziOfferRecordVo);
		}
	}

	/**
	 * 批量增加 操作日志
	 * 3：发送Offer查阅短信；
	 * 4：发送信息收集短信；
	 * 12：发送合同下载短信
	 * @param lGuaziOfferRecordVo 瓜子offer操作日志包装类
	 */
	@Override
	public void saveRecordBatch(LGuaziOfferRecordVo lGuaziOfferRecordVo) {
		LGuaziOfferRecord saveData;
		List<LGuaziOfferRecord> saveDataList = new ArrayList<>();

		if (Common.isNotKong(lGuaziOfferRecordVo.getOfferIds())){
			for(Integer offerId : lGuaziOfferRecordVo.getOfferIds()){
				saveData = new LGuaziOfferRecord();
				saveData.setOfferId(offerId);
				saveData.setRecordFrom(lGuaziOfferRecordVo.getRecordFrom());
				saveData.setCreateTime(LocalDateTime.now());

				// 生成操作内容
				handleContent(lGuaziOfferRecordVo);
				saveData.setContent(lGuaziOfferRecordVo.getContent());

				saveData.setCreateBy(lGuaziOfferRecordVo.getCreateBy());
				saveData.setCreateName(lGuaziOfferRecordVo.getCreateName());
				saveDataList.add(saveData);
			}
			this.saveBatch(saveDataList);
		}
	}

	/**
	 * 生成操作内容
	 *
	 * @param lGuaziOfferRecordVo 瓜子offer操作日志包装类
	 */
	private void handleContent(LGuaziOfferRecordVo lGuaziOfferRecordVo){
		String content = "";
		String statusChangeTemplate = "状态更新：{0}更新为{1}";
		String sendSmsTemplate = "发送{0}，发送结果：{1}";
		// String infoChangeTemplate = "信息变更：{0}：由{1}更新为{2}";

		// 记录来源
		// 1：手动更新状态；2：编辑offer信息；3：发送Offer查阅短信；4：发送信息收集短信；
		// 5：收集短信发送成功更新状态；6：（系统触发）C端采集提交；7：档案审核
		// 9：发起合同申请；10：合同审核通过；11：合同归档；12：发送合同下载短信；13：瓜子系统更新状态）

		// 属于状态更新
		String[] statusChangeArr = {"1", "5", "6", "7", "9", "10", "11", "13"};
		// 属于发送短信
		String[] sendSmsArr = {"3", "4", "12"};

		if(ArrayUtils.contains(statusChangeArr, lGuaziOfferRecordVo.getRecordFrom())){
			// 状态更新
			List<String> changeStatusList = lGuaziOfferRecordVo.getChangeStatusList();
			if(changeStatusList.size() != 2 ){
				content = "状态更新";
			} else {
				String beforeStatus = this.getStatusLabel(changeStatusList.get(0));
				String nowStatus = this.getStatusLabel(changeStatusList.get(1));
				content = MessageFormat.format(statusChangeTemplate, beforeStatus, nowStatus);
			}
		} else if(ArrayUtils.contains(sendSmsArr, lGuaziOfferRecordVo.getRecordFrom())){
			String sendFrom = "";
			String sendSmsRes = "成功";
			if(CommonConstants.ONE_STRING.equals(lGuaziOfferRecordVo.getSendSmsRes())){
				sendSmsRes = "失败";
			}
			// 发送短信
			switch (lGuaziOfferRecordVo.getRecordFrom()) {
				case CommonConstants.THREE_STRING:
					sendFrom = "Offer查阅短信";
					break;
				case CommonConstants.FOUR_STRING:
					sendFrom = "信息收集短信";
					break;
				default:
					sendFrom = "合同下载短信";
					break;
			}
			content = MessageFormat.format(sendSmsTemplate, sendFrom, sendSmsRes);
		} else {
			// 编辑offer信息
			content = "信息变更：";
			if(Common.isNotNull(lGuaziOfferRecordVo.getChangeInfoMap())){
				for (Map.Entry<String,List<String>> entry : lGuaziOfferRecordVo.getChangeInfoMap().entrySet()) {
					String fieldZhName = entry.getKey();
					List<String> changeArr = entry.getValue();
					if(changeArr.size() != 2 ){
						continue;
					}
					String fieldChangeContent = fieldZhName+"：由"+changeArr.get(0)+"更新为"+ changeArr.get(1)+"；";
					content = content + fieldChangeContent;
				}
			}
		}

		// 属于系统更新的，记录创建人为 0：系统触发
		String[] sysCreateArr = {"6"};
		if (ArrayUtils.contains(sysCreateArr, lGuaziOfferRecordVo.getRecordFrom())) {
			lGuaziOfferRecordVo.setCreateBy("0");
			lGuaziOfferRecordVo.setCreateName("系统触发");
		}

		lGuaziOfferRecordVo.setContent(content);
	}

	private String getStatusLabel(String statusCode){
		Map<String, String> statusMap = new HashMap<>();
		// 3：待发送Offer，4：待接受Offer，5：拒绝Offer，7：接受offer，9：候选人填写信息，10：取消offer，
		// 12：候选人信息审核，13：待签署，8：合同审核通过，99：已归档，98：信息待收集，97：候选人审核通过
		statusMap.put("3","待发送Offer");
		statusMap.put("4","待接受Offer");
		statusMap.put("5","拒绝Offer");
		statusMap.put("6","接受offer");
		statusMap.put("9","候选人填写信息");
		statusMap.put("10","取消offer");
		statusMap.put("12","候选人信息审核");
		statusMap.put("13","待签署");
		statusMap.put("8","合同审核通过");
		statusMap.put("99","已归档");
		statusMap.put("98","信息待收集");
		statusMap.put("97","候选人审核通过");

		return statusMap.get(statusCode);
	}
}
