package com.yifu.cloud.plus.v1.ekp.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.yifu.cloud.plus.v1.ekp.config.WxConfig;
import lombok.extern.log4j.Log4j2;
import org.springframework.http.*;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import java.util.*;

/**
 * 企业微信智能表格管理工具类
 * 实现创建智能表格、创建工作表、清空工作表数据和插入数据的功能
 */
@Log4j2
@Service
public class WeComSmartSheetManager {

    /**
     * 企业微信API基础URL
     */
    private static final String WE_COM_API_BASE_URL = "https://qyapi.weixin.qq.com/cgi-bin";

    /**
     * 获取access_token的接口路径
     */
    private static final String GET_TOKEN_PATH = "/gettoken";

    /**
     * 创建文档的接口路径
     */
    private static final String CREATE_DOC_PATH = "/wedoc/create_doc";

    /**
     * 查询子表的接口路径
     */
    private static final String GET_SHEET_PATH = "/wedoc/smartsheet/get_sheet";

    /**
     * 添加记录的接口路径
     */
    private static final String ADD_RECORDS_PATH = "/wedoc/smartsheet/add_records";

    /**
     * 查询记录的接口路径
     */
    private static final String QUERY_RECORDS_PATH = "/wedoc/smartsheet/query_records";

    /**
     * 删除记录的接口路径
     */
    private static final String DELETE_RECORDS_PATH = "/wedoc/smartsheet/delete_records";

    /**
     * 获取字段信息的接口路径
     */
    private static final String GET_FIELDS_PATH = "/wedoc/smartsheet/get_fields";

    /**
     * 添加子表的接口路径
     */
    private static final String ADD_SHEET_PATH = "/wedoc/smartsheet/add_sheet";

	/**
	 * 获取表格行列信息的接口路径
	 */
	private static final String GET_SHEET_PROPERTIES_PATH = "/wedoc/spreadsheet/get_sheet_properties";

	/**
	 * 更新表格内容的接口路径
	 */
	private static final String UPDATE_SPREADSHEET_PATH = "/wedoc/spreadsheet/update";


	/**
	 * 获取记录的接口路径
	 */
	private static final String GET_RECORDS_PATH = "/wedoc/smartsheet/get_records";

    /**
     * 创建智能表格
     *
     * @param docName     文档名称
     * @param adminUsers  管理员用户列表
     * @return 文档ID
     */
    public String createSmartSheet(WxConfig wxConfig, String docName, List<String> adminUsers) {
        try {
			RestTemplate restTemplate = new RestTemplate();
			// 1. 获取access_token
			String accessToken = wxConfig.getAccessToken(restTemplate);
			//getAccessToken(corpId, corpSecret);
            if (accessToken == null || accessToken.isEmpty()) {
                log.error("获取access_token失败");
                return null;
            }

            // 2. 构造请求体
            JSONObject requestBody = new JSONObject();
            requestBody.put("doc_type", 10); // 10: 智能表格
            requestBody.put("doc_name", docName);
            requestBody.put("admin_users", adminUsers);

            // 3. 发送请求创建智能表格
            String docId = createDoc(accessToken, requestBody);
            if (docId != null) {
                System.out.println("智能表格创建成功，文档ID: " + docId);
            } else {
                log.error("智能表格创建失败");
            }
            return docId;
        } catch (Exception e) {
            log.error("创建智能表格时发生异常: " + e.getMessage());
            e.printStackTrace();
            return null;
        }
    }
	/**
	 * 创建智能表格
	 *
	 * @param docName     文档名称
	 * @param adminUsers  管理员用户列表
	 * @return 文档ID
	 */
	public String createSmartSheet(String accessToken, String docName, List<String> adminUsers) {
		try {
			if (accessToken == null || accessToken.isEmpty()) {
				log.error("获取access_token失败");
				return null;
			}

			// 2. 构造请求体
			JSONObject requestBody = new JSONObject();
			requestBody.put("doc_type", 10); // 10: 智能表格
			requestBody.put("doc_name", docName);
			requestBody.put("admin_users", adminUsers);

			// 3. 发送请求创建智能表格
			String docId = createDoc(accessToken, requestBody);
			if (docId != null) {
				System.out.println("智能表格创建成功，文档ID: " + docId);
			} else {
				log.error("智能表格创建失败");
			}
			return docId;
		} catch (Exception e) {
			log.error("创建智能表格时发生异常: " + e.getMessage());
			e.printStackTrace();
			return null;
		}
	}
    /**
     * 获取指定智能表格下的所有工作表ID列表
     * 注意：根据企业微信API文档，创建智能表格时会自动创建一个默认的子表
     * 该方法用于查询这些已存在的子表。
     *
     * @param docId      文档ID
     * @return 工作表ID列表
     */
    public List<String> createWorkSheets(WxConfig wxConfig, String docId) {
        try {
			RestTemplate restTemplate = new RestTemplate();
			// 1. 获取access_token
			String accessToken = wxConfig.getAccessToken(restTemplate);
            if (accessToken == null || accessToken.isEmpty()) {
                log.error("获取access_token失败");
                return new ArrayList<>();
            }

            return getSheetIds(accessToken, docId);
        } catch (Exception e) {
            log.error("获取工作表ID时发生异常: " + e.getMessage());
            e.printStackTrace();
            return new ArrayList<>();
        }
    }
	/**
	 * 获取指定智能表格下的所有工作表ID列表
	 * 注意：根据企业微信API文档，创建智能表格时会自动创建一个默认的子表
	 * 该方法用于查询这些已存在的子表。
	 *
	 * @param docId      文档ID
	 * @return 工作表ID列表
	 */
	public List<String> createWorkSheets(String accessToken, String docId) {
		try {
			if (accessToken == null || accessToken.isEmpty()) {
				log.error("获取access_token失败");
				return new ArrayList<>();
			}

			return getSheetIds(accessToken, docId);
		} catch (Exception e) {
			log.error("获取工作表ID时发生异常: " + e.getMessage());
			e.printStackTrace();
			return new ArrayList<>();
		}
	}
    /**
     * 使用access_token获取智能表格下的工作表ID列表
     *
     * @param accessToken 访问令牌
     * @param docId       文档ID
     * @return 工作表ID列表
     */
    public List<String> getSheetIds(String accessToken, String docId) {
        List<String> sheetIds = new ArrayList<>();
        try {
            // 检查参数是否为空
            if (docId == null || docId.isEmpty()) {
                log.error("docId不能为空");
                return sheetIds;
            }
            
            String url = WE_COM_API_BASE_URL + GET_SHEET_PATH + "?access_token=" + accessToken;

            RestTemplate restTemplate = new RestTemplate();

            // 构造请求体
            JSONObject requestBody = new JSONObject();
            requestBody.put("docid", docId);

            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);

            HttpEntity<String> request = new HttpEntity<>(requestBody.toJSONString(), headers);
            ResponseEntity<String> response = restTemplate.postForEntity(url, request, String.class);

            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSON.parseObject(response.getBody());
                Integer errCode = jsonObject.getInteger("errcode");
                if (errCode == null || errCode == 0) {
                    JSONArray sheets = jsonObject.getJSONArray("sheet_list");
                    if (sheets != null && !sheets.isEmpty()) {
                        for (int i = 0; i < sheets.size(); i++) {
                            JSONObject sheet = sheets.getJSONObject(i);
                            sheetIds.add(sheet.getString("sheet_id"));
                        }
                    } else {
                        log.warn("未找到任何工作表");
                    }
                } else {
                    log.error("查询工作表失败，错误代码：" + errCode
                            + "，错误信息：" + jsonObject.getString("errmsg"));
                    // 特别处理一些常见的错误代码
                    switch (errCode) {
                        case 404:
                            log.error("请求的文档未找到，请检查docId是否正确，并确保是通过API创建的智能表格");
                            break;
                        case 400:
                            log.error("请求参数错误，请检查docId格式是否正确");
                            break;
                        case 48002:
                            log.error("应用没有权限访问该文档，请检查应用权限配置");
                            break;
                        default:
                            log.error("未知错误，请参考企业微信API文档");
                    }
                }
            } else {
                log.error("查询工作表时HTTP状态码错误: " + response.getStatusCode());
            }
        } catch (Exception e) {
            log.error("查询工作表时发生异常: " + e.getMessage(), e);
        }
        return sheetIds;
    }

    /**
     * 通过工作表的ID先清空工作表数据，然后插入新的多条数据
     *
     * @param docId      文档ID
     * @param sheetId    工作表ID
     * @param records    要插入的记录列表
     * @return 是否成功
     */
    public boolean clearAndInsertData(WxConfig wxConfig, String docId, String sheetId,
                                      List<Map<String, Object>> records) {
        try {
			RestTemplate restTemplate = new RestTemplate();
			// 1. 获取access_token
			String accessToken = wxConfig.getAccessToken(restTemplate);
            if (accessToken == null || accessToken.isEmpty()) {
                log.error("获取access_token失败");
                return false;
            }

            // 2. 验证字段是否存在
            if (!validateFields(accessToken, docId, sheetId, records)) {
                log.error("字段验证失败，请检查字段名称是否正确");
                return false;
            }

            // 3. 清空工作表数据
            if (!clearSheetData(accessToken, docId, sheetId)) {
                log.error("清空工作表数据失败");
                return false;
            }

            // 4. 插入新数据（分批处理，避免接口限制）
            if (!insertDataInBatches(accessToken, docId, sheetId, records)) {
                log.error("插入数据失败");
                return false;
            }

            System.out.println("清空并插入数据成功");
            return true;
        } catch (Exception e) {
            log.error("清空并插入数据时发生异常: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    /**
     * 获取企业微信access_token
     *
     * @param corpId     企业ID
     * @param corpSecret 应用密钥
     * @return access_token
     */
    public String getAccessToken(String corpId, String corpSecret) {
        try {
            String url = WE_COM_API_BASE_URL + GET_TOKEN_PATH + "?corpid=" + corpId + "&corpsecret=" + corpSecret;

            RestTemplate restTemplate = new RestTemplate();
            ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);

            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSON.parseObject(response.getBody());
                if (jsonObject.getInteger("errcode") == 0) {
                    return jsonObject.getString("access_token");
                } else {
                    log.error("获取access_token失败，错误代码：" + jsonObject.getInteger("errcode")
                            + "，错误信息：" + jsonObject.getString("errmsg"));
                    return null;
                }
            }
        } catch (Exception e) {
            log.error("获取access_token时发生异常: " + e.getMessage());
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 创建文档
     *
     * @param accessToken 访问令牌
     * @param requestBody 请求体
     * @return 文档ID
     */
    private String createDoc(String accessToken, JSONObject requestBody) {
        try {
            String url = WE_COM_API_BASE_URL + CREATE_DOC_PATH + "?access_token=" + accessToken;

            RestTemplate restTemplate = new RestTemplate();

            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);

            HttpEntity<String> request = new HttpEntity<>(requestBody.toJSONString(), headers);
            ResponseEntity<String> response = restTemplate.postForEntity(url, request, String.class);

            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSON.parseObject(response.getBody());
                if (jsonObject.getInteger("errcode") == 0) {
                    return jsonObject.getString("docid");
                } else {
                    log.error("创建文档失败，错误代码：" + jsonObject.getInteger("errcode")
                            + "，错误信息：" + jsonObject.getString("errmsg"));
                    return null;
                }
            }
        } catch (Exception e) {
            log.error("创建文档时发生异常: " + e.getMessage());
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 验证字段是否存在
     *
     * @param accessToken 访问令牌
     * @param docId       文档ID
     * @param sheetId     工作表ID
     * @param records     记录列表
     * @return 是否验证通过
     */
    private boolean validateFields(String accessToken, String docId, String sheetId, List<Map<String, Object>> records) {
        try {
            // 1. 获取表格字段信息
            List<FieldInfo> fieldInfos = getSheetFields(accessToken, docId, sheetId);
            Set<String> fieldTitles = new HashSet<>();
            for (FieldInfo fieldInfo : fieldInfos) {
                fieldTitles.add(fieldInfo.getTitle());
            }

            // 2. 检查记录中的字段是否都存在
            for (Map<String, Object> record : records) {
                Map<String, Object> values = (Map<String, Object>) record.get("values");
                if (values != null) {
                    for (String fieldName : values.keySet()) {
                        if (!fieldTitles.contains(fieldName)) {
                            log.error("字段 '" + fieldName + "' 在表格中不存在，请检查字段名称是否正确");
                            return false;
                        }
                    }
                }
            }

            return true;
        } catch (Exception e) {
            log.error("验证字段时发生异常: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    /**
     * 获取智能表格字段信息
     *
     * @param accessToken 访问令牌
     * @param docId       文档ID
     * @param sheetId     工作表ID
     * @return 字段信息列表
     */
    private List<FieldInfo> getSheetFields(String accessToken, String docId, String sheetId) {
        List<FieldInfo> fieldInfos = new ArrayList<>();
        try {
            String url = WE_COM_API_BASE_URL + GET_FIELDS_PATH + "?access_token=" + accessToken;

            RestTemplate restTemplate = new RestTemplate();

            // 构造请求体
            JSONObject requestBody = new JSONObject();
            requestBody.put("docid", docId);
            requestBody.put("sheet_id", sheetId);

            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);

            HttpEntity<String> request = new HttpEntity<>(requestBody.toJSONString(), headers);
            ResponseEntity<String> response = restTemplate.postForEntity(url, request, String.class);

            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSON.parseObject(response.getBody());
                if (jsonObject.getInteger("errcode") == 0) {
                    JSONArray fields = jsonObject.getJSONArray("fields");
                    if (fields != null && !fields.isEmpty()) {
						FieldInfo fieldInfo;
                        for (int i = 0; i < fields.size(); i++) {
                            JSONObject field = fields.getJSONObject(i);
                            fieldInfo = new FieldInfo();
                            fieldInfo.setFieldId(field.getString("field_id"));
                            fieldInfo.setTitle(field.getString("field_title"));
                            fieldInfo.setType(field.getString("field_type"));
                            fieldInfos.add(fieldInfo);
                        }
                    }
                } else {
                    log.error("获取字段信息失败，错误代码：" + jsonObject.getInteger("errcode")
                            + "，错误信息：" + jsonObject.getString("errmsg"));
                }
            }
        } catch (Exception e) {
            log.error("获取字段信息时发生异常: " + e.getMessage());
            e.printStackTrace();
        }
        return fieldInfos;
    }

    /**
     * 清空工作表数据
     *
     * @param accessToken 访问令牌
     * @param docId       文档ID
     * @param sheetId     工作表ID
     * @return 是否成功
     */
    public boolean clearSheetData(String accessToken, String docId, String sheetId) {
        try {
            // 检查参数是否为空
            if (docId == null || sheetId == null || docId.isEmpty() || sheetId.isEmpty()) {
                log.error("docId或sheetId不能为空");
                return false;
            }
            
            // 1. 查询所有记录ID
            List<String> recordIds = queryAllRecordIds(accessToken, docId, sheetId);
            if (recordIds.isEmpty()) {
                // 没有记录需要删除
                log.info("没有记录需要删除");
                return true;
            }

            // 2. 分批删除记录（企业微信API限制每次最多删除100条记录）
            int batchSize = 100;
            for (int i = 0; i < recordIds.size(); i += batchSize) {
                int endIndex = Math.min(i + batchSize, recordIds.size());
                List<String> batchRecordIds = recordIds.subList(i, endIndex);

                if (!deleteRecords(accessToken, docId, sheetId, batchRecordIds)) {
                    return false;
                }
            }

            return true;
        } catch (Exception e) {
            log.error("清空工作表数据时发生异常: " + e.getMessage(), e);
            return false;
        }
    }

    /**
     * 查询所有记录ID
     *
     * @param accessToken 访问令牌
     * @param docId       文档ID
     * @param sheetId     工作表ID
     * @return 记录ID列表
     */
    private List<String> queryAllRecordIds(String accessToken, String docId, String sheetId) {
        List<String> recordIds = new ArrayList<>();
        try {
            // 检查参数是否为空
            if (docId == null || sheetId == null || docId.isEmpty() || sheetId.isEmpty()) {
                log.error("docId或sheetId不能为空");
                return recordIds;
            }
            
            String url = WE_COM_API_BASE_URL + QUERY_RECORDS_PATH + "?access_token=" + accessToken;

            RestTemplate restTemplate = new RestTemplate();

            // 构造请求体
            JSONObject requestBody = new JSONObject();
            requestBody.put("docid", docId);
            requestBody.put("sheet_id", sheetId);
            requestBody.put("key_type", "CELL_VALUE_KEY_TYPE_FIELD_TITLE");

            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);

            HttpEntity<String> request = new HttpEntity<>(requestBody.toJSONString(), headers);
            ResponseEntity<String> response = restTemplate.postForEntity(url, request, String.class);

            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSON.parseObject(response.getBody());
                Integer errCode = jsonObject.getInteger("errcode");
                if (errCode == null || errCode == 0) {
                    JSONArray records = jsonObject.getJSONArray("records");
                    if (records != null && !records.isEmpty()) {
                        for (int i = 0; i < records.size(); i++) {
                            JSONObject record = records.getJSONObject(i);
                            recordIds.add(record.getString("record_id"));
                        }
                    }
                } else {
                    log.error("查询记录失败，错误代码：" + errCode
                            + "，错误信息：" + jsonObject.getString("errmsg"));
                    // 特别处理一些常见的错误代码
                    switch (errCode) {
                        case 404:
                            log.error("请求的文档或表格未找到，请检查docId和sheetId是否正确，并确保是通过API创建的智能表格");
                            break;
                        case 400:
                            log.error("请求参数错误，请检查docId和sheetId格式是否正确");
                            break;
                        case 48002:
                            log.error("应用没有权限访问该文档，请检查应用权限配置");
                            break;
                        default:
                            log.error("未知错误，请参考企业微信API文档");
                    }
                }
            } else {
                log.error("查询记录时HTTP状态码错误: " + response.getStatusCode());
            }
        } catch (Exception e) {
            log.error("查询记录时发生异常: " + e.getMessage(), e);
        }
        return recordIds;
    }

    /**
     * 删除记录
     *
     * @param accessToken  访问令牌
     * @param docId        文档ID
     * @param sheetId      工作表ID
     * @param recordIds    记录ID列表
     * @return 是否成功
     */
    private boolean deleteRecords(String accessToken, String docId, String sheetId, List<String> recordIds) {
        try {
            // 检查参数是否为空
            if (docId == null || sheetId == null || docId.isEmpty() || sheetId.isEmpty()) {
                log.error("docId或sheetId不能为空");
                return false;
            }
            
            if (recordIds == null || recordIds.isEmpty()) {
                log.info("没有记录需要删除");
                return true;
            }
            
            String url = WE_COM_API_BASE_URL + DELETE_RECORDS_PATH + "?access_token=" + accessToken;

            RestTemplate restTemplate = new RestTemplate();

            // 构造请求体
            JSONObject requestBody = new JSONObject();
            requestBody.put("docid", docId);
            requestBody.put("sheet_id", sheetId);
            requestBody.put("record_ids", recordIds);

            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);

            HttpEntity<String> request = new HttpEntity<>(requestBody.toJSONString(), headers);
            ResponseEntity<String> response = restTemplate.postForEntity(url, request, String.class);

            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSON.parseObject(response.getBody());
                Integer errCode = jsonObject.getInteger("errcode");
                if (errCode == null || errCode == 0) {
                    return true;
                } else {
                    log.error("删除记录失败，错误代码：" + errCode
                            + "，错误信息：" + jsonObject.getString("errmsg"));
                    // 特别处理一些常见的错误代码
                    switch (errCode) {
                        case 404:
                            log.error("请求的文档或表格未找到，请检查docId和sheetId是否正确");
                            break;
                        case 400:
                            log.error("请求参数错误，请检查参数格式是否正确");
                            break;
                        case 48002:
                            log.error("应用没有权限访问该文档，请检查应用权限配置");
                            break;
                        default:
                            log.error("未知错误，请参考企业微信API文档");
                    }
                    return false;
                }
            } else {
                log.error("删除记录时HTTP状态码错误: " + response.getStatusCode());
                return false;
            }
        } catch (Exception e) {
            log.error("删除记录时发生异常: " + e.getMessage(), e);
            return false;
        }
    }

    /**
     * 分批插入数据
     *
     * @param accessToken 访问令牌
     * @param docId       文档ID
     * @param sheetId     工作表ID
     * @param records     记录列表
     * @return 是否成功
     */
    private boolean insertDataInBatches(String accessToken, String docId, String sheetId, List<Map<String, Object>> records) {
        try {
            // 分批插入数据（企业微信API限制每次最多插入500条记录）
            int batchSize = 500;
            for (int i = 0; i < records.size(); i += batchSize) {
                int endIndex = Math.min(i + batchSize, records.size());
                List<Map<String, Object>> batchRecords = records.subList(i, endIndex);

                if (!addRecords(accessToken, docId, sheetId, batchRecords)) {
                    return false;
                }
            }

            return true;
        } catch (Exception e) {
            log.error("分批插入数据时发生异常: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    /**
     * 添加记录
     *
     * @param accessToken 访问令牌
     * @param docId       文档ID
     * @param sheetId     工作表ID
     * @param records     记录列表
     * @return 是否成功
     */
    private boolean addRecords(String accessToken, String docId, String sheetId, List<Map<String, Object>> records) {
        try {
            String url = WE_COM_API_BASE_URL + ADD_RECORDS_PATH + "?access_token=" + accessToken;

            RestTemplate restTemplate = new RestTemplate();

            // 构造请求体
            JSONObject requestBody = new JSONObject();
            requestBody.put("docid", docId);
            requestBody.put("sheet_id", sheetId);
            requestBody.put("key_type", "CELL_VALUE_KEY_TYPE_FIELD_TITLE");

            // 构造records数组
            JSONArray recordsArray = new JSONArray();
            for (Map<String, Object> record : records) {
                recordsArray.add(record);
            }
            requestBody.put("records", recordsArray);

            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);

            HttpEntity<String> request = new HttpEntity<>(requestBody.toJSONString(), headers);
            ResponseEntity<String> response = restTemplate.postForEntity(url, request, String.class);

            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSON.parseObject(response.getBody());
                if (jsonObject.getInteger("errcode") == 0) {
                    return true;
                } else {
                    log.error("添加记录失败，错误代码：" + jsonObject.getInteger("errcode")
                            + "，错误信息：" + jsonObject.getString("errmsg"));
                    // 特别处理字段不存在的错误
                    if (jsonObject.getInteger("errcode") == 2022004) {
                        log.error("请检查您要插入的字段名称是否与表格中的字段完全匹配");
                    }
                    return false;
                }
            }
        } catch (Exception e) {
            log.error("添加记录时发生异常: " + e.getMessage());
            e.printStackTrace();
        }
        return false;
    }

    /**
     * 使用示例
     *
     */
    public  void test(WxConfig wxConfig) {
        WeComSmartSheetManager manager = new WeComSmartSheetManager();

        // 示例数据
        //String corpId = "your_corp_id";
        //String corpSecret = "your_corp_secret";
        String docName = "测试智能表格";
        List<String> adminUsers = new ArrayList<>();
        adminUsers.add("ZhangQianQian");
        adminUsers.add("FeiTaiLang_Fang");

        // 1. 创建智能表格
        String docId = "dcTSMCj4lWV9MEP7lquZacOY8WS4g-iHxP_jVYA02tdYwb2YYc3BpDR8NrFWkEngHUg8abET4eTIpo9Zv-571NEA";//manager.createSmartSheet(wxConfig, docName, adminUsers);
        if (StringUtils.isEmpty(docId)) {
            System.out.println("创建智能表格失败");
            return;
        }

        // 2. 获取工作表ID列表
        List<String> sheetIds = manager.createWorkSheets(wxConfig, docId);
        if (sheetIds.isEmpty()) {
            System.out.println("获取工作表ID失败");
            return;
        }

        String sheetId = sheetIds.get(0);

        // 3. 准备要插入的数据
        List<Map<String, Object>> records = new ArrayList<>();

        Map<String, Object> record1 = new HashMap<>();
        Map<String, Object> values1 = new HashMap<>();
        values1.put("姓名", "张三");
        values1.put("年龄", "25");
        values1.put("部门", "技术部");
        record1.put("values", values1);

        Map<String, Object> record2 = new HashMap<>();
        Map<String, Object> values2 = new HashMap<>();
        values2.put("姓名", "李四");
        values2.put("年龄", "30");
        values2.put("部门", "市场部");
        record2.put("values", values2);

        records.add(record1);
        records.add(record2);

        // 4. 清空并插入数据
        boolean success = manager.clearAndInsertData(wxConfig, docId, sheetId, records);
        if (success) {
            System.out.println("清空并插入数据成功");
        } else {
            System.out.println("清空并插入数据失败");
        }

    }

	/**
	 * 通过文档ID和标题创建子表
	 *
	 * @param accessToken 访问令牌
	 * @param docId       文档ID
	 * @param title       子表标题
	 * @return 子表ID
	 */
	public String addSheet(String accessToken, String docId, String title) {
		try {
			String url = WE_COM_API_BASE_URL + ADD_SHEET_PATH + "?access_token=" + accessToken;

			RestTemplate restTemplate = new RestTemplate();

			// 构造请求体
			JSONObject requestBody = new JSONObject();
			requestBody.put("docid", docId);

			JSONObject properties = new JSONObject();
			properties.put("title", title);
			requestBody.put("properties", properties);

			HttpHeaders headers = new HttpHeaders();
			headers.setContentType(MediaType.APPLICATION_JSON);

			HttpEntity<String> request = new HttpEntity<>(requestBody.toJSONString(), headers);
			ResponseEntity<String> response = restTemplate.postForEntity(url, request, String.class);

			if (response.getStatusCode() == HttpStatus.OK) {
				JSONObject jsonObject = JSON.parseObject(response.getBody());
				Integer errCode = jsonObject.getInteger("errcode");
				if (errCode == null || errCode == 0) {
					JSONObject sheetProperties = jsonObject.getJSONObject("properties");
					if (sheetProperties != null) {
						String sheetId = sheetProperties.getString("sheet_id");
						log.info("成功创建子表，子表ID: " + sheetId);
						return sheetId;
					}
				} else {
					log.error("创建子表失败，错误代码：" + errCode
							+ "，错误信息：" + jsonObject.getString("errmsg"));
					return null;
				}
			} else {
				log.error("创建子表时HTTP状态码错误: " + response.getStatusCode());
				return null;
			}
		} catch (Exception e) {
			log.error("创建子表时发生异常: " + e.getMessage(), e);
			return null;
		}
		return null;
	}

	/**
	 * 获取表格行数
	 *
	 * @param docId   文档ID
	 * @param sheetId 工作表ID
	 * @return 表格行数
	 */
	public int getRowCount(String  accessToken,String docId, String sheetId) {
		try {
			if (accessToken == null || accessToken.isEmpty()) {
				log.error("获取access_token失败");
				return -1;
			}

			// 2. 获取表格行列信息
			WeComSmartSheetManager.SheetProperties properties = this.getSheetProperties(accessToken, docId, sheetId);
			if (properties == null) {
				log.error("获取表格信息失败，请检查docId和sheetId是否正确，以及应用是否有足够权限访问该文件");
				log.error("注意：根据企业微信API限制，只有通过API创建的智能表格才能通过API进行操作");
				return -1;
			}

			log.info("文档ID: " + docId + ", 工作表ID: " + sheetId + ", 表格信息: " + properties);
			return properties.getRowCount();
		} catch (Exception e) {
			log.error("获取表格行数时发生异常: " + e.getMessage(), e);
			return -1;
		}
	}
	/**
	 * 获取表格行列信息
	 *
	 * @param accessToken 访问令牌
	 * @param docId       文档ID
	 * @param sheetId     工作表ID
	 * @return 表格行列信息
	 */
	public WeComSmartSheetManager.SheetProperties getSheetProperties(String accessToken, String docId, String sheetId) {
		try {
			String url = WE_COM_API_BASE_URL + GET_SHEET_PROPERTIES_PATH + "?access_token=" + accessToken;

			RestTemplate restTemplate = new RestTemplate();

			// 构造请求体
			JSONObject requestBody = new JSONObject();
			requestBody.put("docid", docId);
			//requestBody.put("sheet_id", sheetId);

			HttpHeaders headers = new HttpHeaders();
			headers.setContentType(MediaType.APPLICATION_JSON);

			HttpEntity<String> request = new HttpEntity<>(requestBody.toJSONString(), headers);
			ResponseEntity<String> response = restTemplate.postForEntity(url, request, String.class);

			if (response.getStatusCode() == HttpStatus.OK) {
				JSONObject jsonObject = JSON.parseObject(response.getBody());
				Integer errCode = jsonObject.getInteger("errcode");
				if (errCode == null || errCode == 0) {
					JSONObject properties = jsonObject.getJSONObject("properties");
					if (properties != null) {
						WeComSmartSheetManager.SheetProperties sheetProperties = new WeComSmartSheetManager.SheetProperties();
						sheetProperties.setRowCount(properties.getInteger("row_count"));
						sheetProperties.setColumnCount(properties.getInteger("column_count"));
						return sheetProperties;
					}
				} else {
					log.error("获取表格行列信息失败，错误代码：" + errCode
							+ "，错误信息：" + jsonObject.getString("errmsg"));
				}
			} else {
				log.error("获取表格行列信息时HTTP状态码错误: " + response.getStatusCode());
			}
		} catch (Exception e) {
			log.error("获取表格行列信息时发生异常: " + e.getMessage(), e);
		}
		return null;
	}

	/**
	 * 删除所有行数据
	 *
	 * @param accessToken 访问令牌
	 * @param docId       文档ID
	 * @param sheetId     工作表ID
	 * @return 是否成功
	 */
	public boolean deleteAllRows(String accessToken, String docId, String sheetId) {
		try {
			if (accessToken == null || accessToken.isEmpty()) {
				log.error("获取access_token失败");
				return false;
			}

			// 2. 循环获取并删除记录，直到没有记录为止
			int totalDeleted = 0;
			while (true) {
				// 获取记录
				List<WeComSmartSheetManager.RecordInfo> records = getRecords(accessToken, docId, sheetId, 1000);
				if (records.isEmpty()) {
					log.info("没有更多记录需要删除，已完成清理任务");
					break;
				}

				log.info("获取到 {} 条记录，准备删除", records.size());

				// 提取记录ID
				List<String> recordIds = new ArrayList<>();
				for (WeComSmartSheetManager.RecordInfo record : records) {
					recordIds.add(record.getRecordId());
				}

				// 删除记录
				boolean deleted = deleteRecords(accessToken, docId, sheetId, recordIds);
				if (deleted) {
					totalDeleted += records.size();
					log.info("成功删除 {} 条记录，当前总计已删除 {} 条记录", records.size(), totalDeleted);
				} else {
					log.error("删除记录失败");
					return false;
				}
			}

			log.info("任务完成，总共删除 {} 条记录", totalDeleted);
			return true;
		} catch (Exception e) {
			log.error("获取并删除记录时发生异常: " + e.getMessage(), e);
			return false;
		}
	}
	/**
	 * 获取记录
	 *
	 * @param accessToken 访问令牌
	 * @param docId       文档ID
	 * @param sheetId     工作表ID
	 * @param limit       限制条数
	 * @return 记录列表
	 */
	private List<WeComSmartSheetManager.RecordInfo> getRecords(String accessToken, String docId, String sheetId, int limit) {
		List<WeComSmartSheetManager.RecordInfo> records = new ArrayList<>();
		try {
			String url = WE_COM_API_BASE_URL + GET_RECORDS_PATH + "?access_token=" + accessToken;

			RestTemplate restTemplate = new RestTemplate();

			// 构造请求体
			JSONObject requestBody = new JSONObject();
			requestBody.put("docid", docId);
			requestBody.put("sheet_id", sheetId);
			requestBody.put("limit", limit);

			HttpHeaders headers = new HttpHeaders();
			headers.setContentType(MediaType.APPLICATION_JSON);

			HttpEntity<String> request = new HttpEntity<>(requestBody.toJSONString(), headers);
			ResponseEntity<String> response = restTemplate.postForEntity(url, request, String.class);

			if (response.getStatusCode() == HttpStatus.OK) {
				JSONObject jsonObject = JSON.parseObject(response.getBody());
				Integer errCode = jsonObject.getInteger("errcode");
				if (errCode == null || errCode == 0) {
					JSONArray recordArray = jsonObject.getJSONArray("records");
					if (recordArray != null && !recordArray.isEmpty()) {
						for (int i = 0; i < recordArray.size(); i++) {
							JSONObject recordObj = recordArray.getJSONObject(i);
							WeComSmartSheetManager.RecordInfo recordInfo = new WeComSmartSheetManager.RecordInfo();
							recordInfo.setRecordId(recordObj.getString("record_id"));
							records.add(recordInfo);
						}
					}
				} else {
					log.error("获取记录失败，错误代码：" + errCode
							+ "，错误信息：" + jsonObject.getString("errmsg"));
				}
			} else {
				log.error("获取记录时HTTP状态码错误: " + response.getStatusCode());
			}
		} catch (Exception e) {
			log.error("获取记录时发生异常: " + e.getMessage(), e);
		}
		return records;
	}

	/**
	 * 记录信息内部类
	 */
	public static class RecordInfo {
		private String recordId;

		public String getRecordId() {
			return recordId;
		}

		public void setRecordId(String recordId) {
			this.recordId = recordId;
		}

		@Override
		public String toString() {
			return "RecordInfo{" +
					"recordId='" + recordId + '\'' +
					'}';
		}
	}
	/**
	 * 表格属性信息类
	 */
	public static class SheetProperties {
		/**
		 * 行数
		 */
		private int rowCount;

		/**
		 * 列数
		 */
		private int columnCount;

		public int getRowCount() {
			return rowCount;
		}

		public void setRowCount(int rowCount) {
			this.rowCount = rowCount;
		}

		public int getColumnCount() {
			return columnCount;
		}

		public void setColumnCount(int columnCount) {
			this.columnCount = columnCount;
		}

		@Override
		public String toString() {
			return "SheetProperties{" +
					"rowCount=" + rowCount +
					", columnCount=" + columnCount +
					'}';
		}
	}
}