/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.salary.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.MenuUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TSalaryStandard;
import com.yifu.cloud.plus.v1.yifu.salary.service.TSalaryStandardService;
import com.yifu.cloud.plus.v1.yifu.salary.util.SalaryConstants;
import com.yifu.cloud.plus.v1.yifu.salary.vo.AccountByEkpDiffVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.SalaryRiskVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TSalaryStandardSearchVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TSalaryStandardUnprojectVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.Date;
import java.util.List;


/**
 * 标准薪酬工资表
 *
 * @author hgw
 * @date 2022-08-05 11:40:14
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tsalarystandard")
@Tag(name = "标准薪酬工资表管理")
public class TSalaryStandardController {

	private final TSalaryStandardService tSalaryStandardService;
	private final MenuUtil menuUtil;

	/**
	 * 简单分页查询-申请
	 *
	 * @param page            分页对象
	 * @param tSalaryStandard 标准薪酬工资表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/pageApply")
	public R<IPage<TSalaryStandard>> getTSalaryStandardPageApply(Page<TSalaryStandard> page, TSalaryStandardSearchVo tSalaryStandard) {
		tSalaryStandard.setDeleteFlag(CommonConstants.ZERO_INT);
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, tSalaryStandard);
		return new R<>(tSalaryStandardService.getTSalaryStandardPageApply(page, tSalaryStandard));
	}

	/**
	 * 简单分页查询
	 *
	 * @param page            分页对象
	 * @param tSalaryStandard 标准薪酬工资表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TSalaryStandard>> getTSalaryStandardPage(Page<TSalaryStandard> page, TSalaryStandardSearchVo tSalaryStandard) {
		YifuUser user = SecurityUtils.getUser();
		tSalaryStandard.setDeleteFlag(CommonConstants.ZERO_INT);
		menuUtil.setAuthSql(user, tSalaryStandard);
		return new R<>(tSalaryStandardService.getTSalaryStandardPage(page, tSalaryStandard));
	}

	/**
	 * 通过id查询标准薪酬工资表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('salary_tsalarystandard_get')")
	@GetMapping("/{id}")
	public R<TSalaryStandard> getById(@PathVariable("id") String id) {
		return R.ok(tSalaryStandardService.getById(id));
	}

	/**
	 * 修改标准薪酬工资表
	 *
	 * @param tSalaryStandard 标准薪酬工资表
	 * @return R
	 */
	@Operation(summary = "修改标准薪酬工资表", description = "修改标准薪酬工资表：hasPermission('salary_tsalarystandard_edit')")
	@SysLog("修改标准薪酬工资表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('salary_tsalarystandard_edit')")
	public R<Boolean> updateById(@RequestBody TSalaryStandard tSalaryStandard) {
		return R.ok(tSalaryStandardService.updateById(tSalaryStandard));
	}

	/**
	 * 通过id删除标准薪酬工资表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除标准薪酬工资表", description = "通过id删除标准薪酬工资表：hasPermission('salary_tsalarystandard_del')")
	@SysLog("通过id删除标准薪酬工资表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('salary_tsalarystandard_del')")
	public R<String> removeById(@PathVariable String id) {
		return tSalaryStandardService.deleteSalaryById(id);
	}

	/**
	 * @param id 工资主表ID
	 * @Description: HCM通过id删除标准薪酬工资表
	 * @Author: hgw
	 * @Date: 2025/5/20 19:59
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "HCM通过id删除标准薪酬工资表", description = "通过id删除标准薪酬工资表")
	@SysLog("HCM通过id删除标准薪酬工资表")
	@GetMapping("/deleteById")
	public R<String> deleteById(@RequestParam String id) {
		return tSalaryStandardService.deleteSalaryById(id);
	}

	/**
	 * 标准薪酬工资表 批量导出
	 *
	 * @author fxj
	 * @date 2022-08-05 11:40:14
	 **/
	@SysLog("薪资-导出标准薪酬工资表")
	@Operation(description = "导出标准薪酬工资表 hasPermission('salary_tsalarystandard-export')")
	@PostMapping("/export")
	//@PreAuthorize("@pms.hasPermission('salary_tsalarystandard-export')")
	public void export(HttpServletResponse response, @RequestBody TSalaryStandardSearchVo searchVo) {
		YifuUser user = SecurityUtils.getUser();
		if (user != null && Common.isNotNull(user.getId())) {
			menuUtil.setAuthSql(user, searchVo);
		}
		tSalaryStandardService.listExport(response, searchVo);
	}

	/**
	 * 审核页面简单分页查询
	 *
	 * @param page            分页对象
	 * @param tSalaryStandard 标准薪酬工资表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/auditPage")
	public R<IPage<TSalaryStandard>> getTSalaryStandardAuditPage(Page<TSalaryStandard> page, TSalaryStandardSearchVo tSalaryStandard) {
		return new R<>(tSalaryStandardService.getTSalaryStandardAuditPage(page, tSalaryStandard));
	}

	/**
	 * 审核
	 *
	 * @param auditFlag 是否通过 0 是 1 否
	 * @param auditRemark 审核意见
	 * @param id
	 * @return
	 */
	@SysLog("薪资-薪酬审核")
	@Operation(description = "薪酬审核")
	@PostMapping("/salaryStandardAudit")
	public R<String> salaryStandardAudit(@RequestParam String auditFlag, @RequestParam(required = false) String auditRemark,
								@RequestParam String id) {
		return tSalaryStandardService.salaryStandardAudit(auditFlag, auditRemark, id);
	}

	/**
	 * 保存并提交
	 * @return
	 */
	@SysLog("校验商险")
	@Operation(description = "校验商险")
	@GetMapping("/doCheckRisk")
	public R<List<SalaryRiskVo>> doCheckRisk(@RequestParam String salaryId) {
		TSalaryStandard s = tSalaryStandardService.getById(salaryId);
		if (s == null) {
			return R.failed("未找到工资表");
		} else if (s.getStatus() == SalaryConstants.STATUS[0]
				|| s.getStatus() == SalaryConstants.STATUS[5]
				|| s.getStatus() == SalaryConstants.STATUS[6]
				|| s.getStatus() == SalaryConstants.STATUS[7]) {
			// 已校验过了：
			if (Common.isNotNull(s.getIsCheckRisk()) && CommonConstants.ONE_INT == s.getIsCheckRisk()) {
				return R.ok();
			}
			if (!CommonConstants.ZERO_STRING.equals(s.getFormType())) {
				s.setIsCheckRisk(CommonConstants.ONE_INT);
				s.setCheckRiskRemark("非工资类型，放开限制");
				this.updateById(s);
				return R.ok();
			}
			YifuUser user = SecurityUtils.getUser();
			if (user != null) {
				// 2025-1-14 10:59:37 提交前拦截：
				return tSalaryStandardService.judgeRisk(s, user);
			} else {
				return R.failed("请登录！");
			}
		} else {
			return R.failed("请刷新后检查薪资表状态！");
		}
	}

	/**
	 * 保存并提交
	 * @return
	 */
	@SysLog("薪资保存并提交")
	@Operation(description = "保存并提交")
	@PostMapping("/doSubmit")
	public R<String> salaryStandardAudit(@RequestBody TSalaryStandard tSalaryStandard) {
		if (tSalaryStandard != null && Common.isNotNull(tSalaryStandard.getId())) {
			TSalaryStandard s = tSalaryStandardService.getById(tSalaryStandard.getId());
			if (s == null) {
				return R.failed("未找到工资表");
			} else if (s.getStatus() == SalaryConstants.STATUS[0]
					|| s.getStatus() == SalaryConstants.STATUS[5]
					|| s.getStatus() == SalaryConstants.STATUS[6]
					|| s.getStatus() == SalaryConstants.STATUS[7]) {
				if (CommonConstants.ZERO_STRING.equals(s.getFormType())
						&& (Common.isEmpty(s.getIsCheckRisk()) || CommonConstants.ZERO_INT == s.getIsCheckRisk())) {
					return R.failed("请先校验商险信息！");
				}

				YifuUser user = SecurityUtils.getUser();
				if (user != null) {

					// 添加流程进展明细
					tSalaryStandardService.saveRecordLog(s, user, CommonConstants.TWO_STRING, "提交，并自动生成收入");
					tSalaryStandard.setSubmitTime(new Date());
					tSalaryStandard.setStatus(CommonConstants.dingleDigitIntArray[11]);
					tSalaryStandard.setIncomeStartTime(new Date());
					tSalaryStandardService.updateById(tSalaryStandard);
					tSalaryStandard = tSalaryStandardService.getById(tSalaryStandard.getId());
					// 生成收入
					tSalaryStandardService.doIncome(tSalaryStandard);

					return R.ok("提交成功！");
				} else {
					return R.failed("请登录！");
				}
			} else {
				return R.failed("请刷新后检查薪资表状态！");
			}
		} else {
			return R.failed("请传参！");
		}
	}

	/**
	 * @param id 主键
	 * @Description: 发送至EKP
	 * @Author: hgw
	 * @Date: 2022/8/26 11:40
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R
	 **/
	@SysLog("薪资-发送至EKP")
	@Operation(description = "发送至EKP")
	@PostMapping("/doSend")
	public R<String> doSend(@RequestParam String id) {
		return tSalaryStandardService.doSend(id);
	}

	/**
	 * 退回薪资表，退回中，再调用ekp接口，通知他们删除
	 * @author  hgw
	 * @return R
	 */
	@Operation(summary = "退回薪资表--EKP调用接口", description = "退回薪资表--EKP调用接口")
	@SysLog("退回薪资表--EKP调用接口")
	@GetMapping("/backSalaryByApplyNo")
	public R<String> backSalaryByApplyNo(@RequestParam String applyNo) {
		return tSalaryStandardService.backSalaryByApplyNo(applyNo);
	}

	/**
	 * @param id
	 * @return
	 */
	@SysLog("薪资-重新生成收入")
	@Operation(description = "重新生成收入")
	@GetMapping("/salaryDoIncome")
	public R<String> salaryDoIncome(@RequestParam String id) {
		return tSalaryStandardService.salaryDoIncome(id);
	}


	@Operation(description = "测试推送EKP主表含代发户list")
	@SysLog("测试推送EKP主表含代发户list")
	@PostMapping("/testSendEkpList")
	public String testSendEkpList(@RequestParam String id) {
		return tSalaryStandardService.testSendEkpList(id);
	}

	/**
	 * @Description: 差额单--EKP调用接口
	 * @Author: hgw
	 * @Date: 2023/1/31 17:28
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "差额单--EKP调用接口", description = "差额单--EKP调用接口")
	@SysLog("差额单--EKP调用接口")
	@PostMapping("/saveSalaryDiff")
	public R<String> saveSalaryDiff(@RequestBody List<AccountByEkpDiffVo> voList) {
		return tSalaryStandardService.saveSalaryDiff(voList);
	}

	/**
	 * 获取不在项人员发薪人员列表
	 *
	 * @return
	 */
	@SysLog("获取不在项人员发薪人员列表")
	@Operation(description = "获取不在项人员发薪人员列表")
	@GetMapping("/getUnDeptSalaryList")
	public R<IPage<TSalaryStandardUnprojectVo>> getUnDeptSalaryList(Page<TSalaryStandardUnprojectVo> page, String deptNo) {
		return new R<>(tSalaryStandardService.getUnDeptSalaryList(page, deptNo));
	}
}


