/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.salary.service.impl;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.DateUtil;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ExcelUtil;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TStatisticsDeclarer;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TStatisticsRemuneration;
import com.yifu.cloud.plus.v1.yifu.salary.mapper.TStatisticsDeclarerMapper;
import com.yifu.cloud.plus.v1.yifu.salary.mapper.TStatisticsRemunerationMapper;
import com.yifu.cloud.plus.v1.yifu.salary.service.TStatisticsRemunerationService;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TStatisticsRemunerationSearchVo;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Service;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 本期稿酬申报表
 *
 * @author hgw
 * @date 2022-08-05 11:40:14
 */
@Log4j2
@Service
@RequiredArgsConstructor
public class TStatisticsRemunerationServiceImpl extends ServiceImpl<TStatisticsRemunerationMapper, TStatisticsRemuneration> implements TStatisticsRemunerationService {

	private final TStatisticsDeclarerMapper declarerMapper;

	/**
	 * 本期稿酬申报表简单分页查询
	 *
	 * @param tStatisticsRemuneration 本期稿酬申报表
	 * @return
	 */
	@Override
	public IPage<TStatisticsRemuneration> getTStatisticsRemunerationPage(Page<TStatisticsRemuneration> page, TStatisticsRemunerationSearchVo tStatisticsRemuneration) {
		return baseMapper.getTStatisticsRemunerationPage(page, tStatisticsRemuneration);
	}

	/**
	 * 本期稿酬申报表批量导出
	 *
	 * @return
	 */
	@Override
	public void listExport(HttpServletResponse response, TStatisticsRemunerationSearchVo searchVo) {
		String fileName = "本期稿酬申报表导出" + DateUtil.getThisTime() + CommonConstants.XLSX;
		//获取要导出的列表
		List<TStatisticsRemuneration> list = new ArrayList<>();
		long count = noPageCountDiy(searchVo);
		ServletOutputStream out = null;
		try {
			out = response.getOutputStream();
			response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
			response.setCharacterEncoding("utf-8");
			response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName, "UTF-8"));
			// 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
			ExcelWriter excelWriter = EasyExcel.write(out, TStatisticsRemuneration.class).build();
			int index = 0;
			if (count > CommonConstants.ZERO_INT) {
				WriteSheet writeSheet;
				ExcelUtil<TStatisticsRemuneration> util;
				for (int i = 0; i <= count; ) {
					// 获取实际记录
					searchVo.setLimitStart(i);
					searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
					list = noPageDiy(searchVo);
					if (Common.isNotNull(list)) {
						util = new ExcelUtil<>(TStatisticsRemuneration.class);
						for (TStatisticsRemuneration vo : list) {
							util.convertEntity(vo, null, null, null);
						}
					}
					if (Common.isNotNull(list)) {
						writeSheet = EasyExcel.writerSheet("本期稿酬申报表" + index).build();
						excelWriter.write(list, writeSheet);
						index++;
					}
					i = i + CommonConstants.EXCEL_EXPORT_LIMIT;
					if (Common.isNotNull(list)) {
						list.clear();
					}
				}
			} else {
				WriteSheet writeSheet = EasyExcel.writerSheet("本期稿酬申报表" + index).build();
				excelWriter.write(list, writeSheet);
			}
			if (Common.isNotNull(list)) {
				list.clear();
			}
			out.flush();
			excelWriter.finish();
		} catch (Exception e) {
			log.error("执行异常", e);
		} finally {
			try {
				if (null != out) {
					out.close();
				}
			} catch (IOException e) {
				log.error("执行异常", e);
			}
		}
	}

	@Override
	public R batchDelete(String declareMonth) {
		if (Common.isNotNull(declareMonth)) {
			List<TStatisticsRemuneration> deleteList = baseMapper.selectList(Wrappers.<TStatisticsRemuneration>query().lambda()
					.eq(TStatisticsRemuneration::getDeclareMonth, declareMonth));
			if (Common.isNotNull(deleteList)) {
				List<String> idList = deleteList.stream().map(TStatisticsRemuneration::getId).collect(Collectors.toList());
				baseMapper.deleteBatchIds(idList);
				return R.ok();
			}
			return R.failed(CommonConstants.NO_DATA_TO_HANDLE);
		}
		return null;
	}

	@Override
	public R doStatisticsRemuneration() {
		String nowMonth = DateUtil.addMonth(0);  //本月
		TStatisticsRemuneration std = new TStatisticsRemuneration();
		std.setDeclareMonth(nowMonth);
		QueryWrapper<TStatisticsRemuneration> queryWrapperSs = new QueryWrapper<>();
		queryWrapperSs.setEntity(std);
		long num = this.count(queryWrapperSs);
		if (num > 0) {
			return R.failed("上月数据已生成，不可重复生成！");
		} else {
			String lastMonth =  DateUtil.addMonth(-1);  //上月
			List<TStatisticsRemuneration> stdvoList = baseMapper.doStatisticsRemuneration(nowMonth,lastMonth);
			for (TStatisticsRemuneration remuneration : stdvoList) {
				this.save(remuneration);
			}
			return R.ok();
		}
	}

	@Override
	public R refreshStatisticsRemuneration(String declareMonth) {
		TStatisticsRemuneration std = new TStatisticsRemuneration();
		std.setDeclareMonth(declareMonth);
		QueryWrapper<TStatisticsRemuneration> queryWrapperSs = new QueryWrapper<>();
		queryWrapperSs.setEntity(std);
		long num = this.count(queryWrapperSs);
		if (num > 0) {
			baseMapper.delete(Wrappers.<TStatisticsRemuneration>query().lambda().eq(TStatisticsRemuneration::getDeclareMonth,declareMonth));
		}
		String lastMonth =  DateUtil.addMonthByYearMonth(-1,declareMonth);  //上月
		List<TStatisticsRemuneration> stdvoList = baseMapper.doStatisticsRemuneration(declareMonth,lastMonth);
		for (TStatisticsRemuneration remuneration : stdvoList) {
			TStatisticsDeclarer declarer = declarerMapper.selectOne(Wrappers.<TStatisticsDeclarer>query().lambda()
					.eq(TStatisticsDeclarer::getDeclareMonth,declareMonth)
					.eq(TStatisticsDeclarer::getEmpIdcard,remuneration.getEmpIdcard())
					.eq(TStatisticsDeclarer::getDeclareUnit,remuneration.getDeclareUnit())
					.last(CommonConstants.LAST_ONE_SQL));
			if (Common.isNotNull(declarer) && CommonConstants.ZERO_STRING.equals(declarer.getIsDeclare())) {
				this.save(remuneration);
			}
		}
		return R.ok();
	}

	private List<TStatisticsRemuneration> noPageDiy(TStatisticsRemunerationSearchVo searchVo) {
		LambdaQueryWrapper<TStatisticsRemuneration> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TStatisticsRemuneration::getId, idList);
		}
		if (searchVo.getLimitStart() >= 0 && searchVo.getLimitEnd() > 0) {
			wrapper.last(" limit " + searchVo.getLimitStart() + "," + searchVo.getLimitEnd());
		}
		return baseMapper.selectList(wrapper);
	}

	private Long noPageCountDiy(TStatisticsRemunerationSearchVo searchVo) {
		LambdaQueryWrapper<TStatisticsRemuneration> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TStatisticsRemuneration::getId, idList);
		}
		return baseMapper.selectCount(wrapper);
	}

	private LambdaQueryWrapper buildQueryWrapper(TStatisticsRemunerationSearchVo entity) {
		LambdaQueryWrapper<TStatisticsRemuneration> wrapper = Wrappers.lambdaQuery();
		if (Common.isNotNull(entity.getDeclareMonth())) {
			wrapper.eq(TStatisticsRemuneration::getDeclareMonth,entity.getDeclareMonth());
		}
		if (Common.isNotNull(entity.getEmpName())) {
			wrapper.eq(TStatisticsRemuneration::getEmpName,entity.getEmpName());
		}
		if (Common.isNotNull(entity.getEmpIdcard())) {
			wrapper.eq(TStatisticsRemuneration::getEmpIdcard,entity.getEmpIdcard());
		}
		return wrapper;
	}

}
