/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.social.entity.SysHouseHoldInfo;
import com.yifu.cloud.plus.v1.yifu.social.service.SysHouseHoldInfoService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 社保户公积金户数据
 *
 * @author fxj
 * @date 2022-07-12 08:53:19
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/syshouseholdinfo" )
@Tag(name = "社保户公积金户数据管理")
public class SysHouseHoldInfoController {

    private final SysHouseHoldInfoService sysHouseHoldInfoService;

    /**
     * 简单分页查询
     * @param page 分页对象
     * @param sysHouseHoldInfo 社保户公积金户数据
     * @return
     */
	@Operation(summary = "简单分页查询", description = "简单分页查询")
    @GetMapping("/page")
    public R<IPage<SysHouseHoldInfo>> getSysHouseHoldInfoPage(Page<SysHouseHoldInfo> page, SysHouseHoldInfo sysHouseHoldInfo) {
        return new R<>(sysHouseHoldInfoService.getSysHouseHoldInfoPage(page,sysHouseHoldInfo));
    }


	/**
	 * 获取所有的独立户数据
	 * @param page 分页对象
	 * @return
	 */
	@Operation(description = "获取所有的独立户数据")
	@GetMapping("/getAllSingleSocail" )
	public R<IPage<SysHouseHoldInfo>> getAllSingleSocailPage(Page page) {
		return new R<>(sysHouseHoldInfoService.getAllSingleSocailPage(page));
	}

    /**
     * 不分页查询
     * @param sysHouseHoldInfo 社保户公积金户数据
     * @return
     */
    @Operation(summary = "不分页查询", description = "不分页查询")
    @PostMapping("/noPage" )
    public R<List<SysHouseHoldInfo>> getSysHouseHoldInfoNoPage(@RequestBody SysHouseHoldInfo sysHouseHoldInfo) {
        return R.ok(sysHouseHoldInfoService.list(Wrappers.query(sysHouseHoldInfo)));
    }

    /**
     * 通过id查询社保户公积金户数据
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id查询", description = "通过id查询：hasPermission('demo_syshouseholdinfo_get')")
    @GetMapping("/{id}" )
    public R<SysHouseHoldInfo> getById(@PathVariable("id" ) String id) {
        return sysHouseHoldInfoService.getByIdAtta(id);
    }

    /**
     * 新增社保户公积金户数据
     * @param sysHouseHoldInfo 社保户公积金户数据
     * @return R
     */
    @Operation(summary = "新增社保户公积金户数据", description = "新增社保户公积金户数据：hasPermission('demo_syshouseholdinfo_add')")
    @SysLog("新增社保户公积金户数据" )
    @PostMapping
    //@PreAuthorize("@pms.hasPermission('demo_syshouseholdinfo_add')" )
    public R<Boolean> save(@RequestBody SysHouseHoldInfo sysHouseHoldInfo) {
        return sysHouseHoldInfoService.saveInfo(sysHouseHoldInfo);
    }

    /**
     * 修改社保户公积金户数据
     * @param sysHouseHoldInfo 社保户公积金户数据
     * @return R
     */
    @Operation(summary = "修改社保户公积金户数据", description = "修改社保户公积金户数据：hasPermission('demo_syshouseholdinfo_edit')")
    @SysLog("修改社保户公积金户数据" )
    @PutMapping
    //@PreAuthorize("@pms.hasPermission('demo_syshouseholdinfo_edit')" )
    public R<Boolean> updateById(@RequestBody SysHouseHoldInfo sysHouseHoldInfo) {
        return sysHouseHoldInfoService.updateByIdAsso(sysHouseHoldInfo);
    }

	/**
	 * 修改社保户公积金户自动化配置
	 * @param autoStatus 户自动化配置启用禁用标识 0 启用 1 禁用
	 * @return R
	 */
	@Operation(summary = "修改社保户公积金户自动化配置", description = "修改社保户公积金户自动化配置：hasPermission('demo_syshouseholdinfo_edit')")
	@SysLog("修改社保户公积金户自动化配置" )
	@PostMapping("/updateAutoType")
	@PreAuthorize("@pms.hasPermission('demo_syshouseholdinfo_edit')" )
	public R updateAutoType(@RequestParam String id, @RequestParam(required = false) String autoStatus,@RequestParam(required = false) String autoStatusYsd) {
		if (Common.isEmpty(autoStatus) && Common.isEmpty(autoStatusYsd)){
			return R.failed("请选择启用禁用状态");
		}
		SysHouseHoldInfo sysHouseHoldInfo = sysHouseHoldInfoService.getById(id);
		if (Common.isNotNull(sysHouseHoldInfo)) {
			if (Common.isNotNull(autoStatusYsd)){
				sysHouseHoldInfo.setAutoStatusYsd(autoStatusYsd);
			}
			if (Common.isNotNull(autoStatus)){
				sysHouseHoldInfo.setAutoStatus(autoStatus);
			}
			sysHouseHoldInfoService.updateById(sysHouseHoldInfo);
			return R.ok();
		}
		return R.failed(CommonConstants.NO_DATA_TO_HANDLE);
	}

    /**
     * 通过id删除社保户公积金户数据
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id删除社保户公积金户数据", description = "通过id删除社保户公积金户数据：hasPermission('demo_syshouseholdinfo_del')")
    @SysLog("通过id删除社保户公积金户数据" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_syshouseholdinfo_del')" )
    public R<Boolean> removeById(@PathVariable String id) {
        return R.ok(sysHouseHoldInfoService.removeById(id));
    }

	/**
	 * @param flag 0/1
	 * @param name 户名
	 * @return 上级部门用户列表
	 */
	@PostMapping("/getHouseList")
	@Operation(summary = "获取所有的户名", description = "获取所有的户名")
	public R<List<SysHouseHoldInfo>> getSysUserList(@RequestParam String flag,
										   @RequestParam(required = false) String name) {
		List<SysHouseHoldInfo> houseHoldList;
		LambdaQueryWrapper<SysHouseHoldInfo> wrapper = Wrappers.lambdaQuery();
		wrapper.eq(SysHouseHoldInfo::getDelFlag, CommonConstants.ZERO_STRING);
		if (CommonConstants.ZERO_STRING.equals(flag)) {
			wrapper.eq(SysHouseHoldInfo::getType, CommonConstants.ZERO_STRING);
			if (Common.isNotNull(name)) {
				wrapper.like(SysHouseHoldInfo::getName, name);
			}
		} else {
			wrapper.eq(SysHouseHoldInfo::getType, CommonConstants.ONE_STRING);
			if (Common.isNotNull(name)) {
				wrapper.like(SysHouseHoldInfo::getName, name);
			}
		}
		houseHoldList = sysHouseHoldInfoService.list(wrapper);
		return R.ok(houseHoldList);
	}

	/**
	 * @param flag 0/1
	 * @param name 户名
	 * @return  获取自动化标识
	 */
	@GetMapping("/getAutoFlag")
	@Operation(summary = "获取自动化标识", description = "获取自动化标识")
	public R<Integer> getAutoFlag(@RequestParam String flag, @RequestParam(required = false) String name) {
		SysHouseHoldInfo houseHoldInfo;
		LambdaQueryWrapper<SysHouseHoldInfo> wrapper = Wrappers.lambdaQuery();
		wrapper.eq(SysHouseHoldInfo::getDelFlag, CommonConstants.ZERO_STRING);
		if (CommonConstants.ZERO_STRING.equals(flag)) {
			wrapper.eq(SysHouseHoldInfo::getType, CommonConstants.ZERO_STRING);
			if (Common.isNotNull(name)) {
				wrapper.eq(SysHouseHoldInfo::getName, name);
			}
		} else {
			wrapper.eq(SysHouseHoldInfo::getType, CommonConstants.ONE_STRING);
			if (Common.isNotNull(name)) {
				wrapper.eq(SysHouseHoldInfo::getName, name);
			}
		}
		houseHoldInfo = sysHouseHoldInfoService.getOne(wrapper);
		if (Common.isNotNull(houseHoldInfo) && CommonConstants.ZERO_STRING.equals(houseHoldInfo.getAutoStatus())){
			return R.ok(0);
		}
		return R.ok(1);
	}

	/**
	 * 批量新增社保户配置数据
	 *
	 * @author huyc
	 * @date 2024-12-5 10:18:05
	 **/
	@SneakyThrows
	@Operation(description = "批量新增社保户配置数据 hasPermission('syshouseholdinfo_batch_add')")
	@SysLog("批量新增社保户配置数据")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('syshouseholdinfo_batch_add')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file){
		return sysHouseHoldInfoService.importListAdd(file.getInputStream());
	}

	/**
	 * 批量更新社保户配置数据
	 *
	 * @author huyc
	 * @date 2024-12-5 10:18:05
	 **/
	@SneakyThrows
	@Operation(description = "批量更新社保户配置数据")
	@SysLog("批量更新社保户配置数据")
	@PostMapping("/importListUpdate")
	public R<List<ErrorMessage>> importListUpdate(@RequestBody MultipartFile file){
		return sysHouseHoldInfoService.importListUpdate(file.getInputStream());
	}

	/**
	 * 导出社保户配置数据
	 *
	 * @author huyc
	 * @date 2024-12-5 11:32:08
	 **/
	@Operation(description = "导出社保户配置数据")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('syshouseholdinfo_export')")
	public void export(HttpServletResponse response, @RequestBody SysHouseHoldInfo sysHouseHoldInfo) {
		sysHouseHoldInfoService.listExport(response,sysHouseHoldInfo);
	}

	/**
	 * @param type 0社保/1公积金
	 * @param name 户名
	 * @Description: 根据户名与户类型，返回户信息 type 0社保/1公积金 name 户名
	 * @Author: hgw
	 * @Date: 2025/4/28 10:08
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.social.entity.SysHouseHoldInfo>
	 **/
	@GetMapping("/getHuByTypeAndName")
	@Operation(summary = "获取所有的户名", description = "获取所有的户名")
	public R<SysHouseHoldInfo> getHuByTypeAndName(@RequestParam String type, @RequestParam String name) {
		SysHouseHoldInfo houseHold;
		LambdaQueryWrapper<SysHouseHoldInfo> wrapper = Wrappers.lambdaQuery();
		wrapper.eq(SysHouseHoldInfo::getDelFlag, CommonConstants.ZERO_STRING);
		wrapper.eq(SysHouseHoldInfo::getType, type);
		wrapper.eq(SysHouseHoldInfo::getName, name);
		wrapper.last(CommonConstants.LAST_ONE_SQL);
		houseHold = sysHouseHoldInfoService.getOne(wrapper);
		return R.ok(houseHold);
	}

}
