/*
 * Copyright (c) 2020 yifu4cloud Authors. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.yifu.cloud.plus.v1.yifu.admin.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysDataAuth;
import com.yifu.cloud.plus.v1.yifu.admin.api.vo.SysDataAuthVO;
import com.yifu.cloud.plus.v1.yifu.admin.service.SysDataAuthService;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @Description: 数据权限控制器
 * @Author: hgw
 * @Date: 2022/6/7 10:44
 * @return:
 **/
@RestController
@RequiredArgsConstructor
@RequestMapping("/dataAuth")
@Tag(name = "数据权限控制器")
public class DataAuthController {

	private final SysDataAuthService sysDataAuthService;

	@Inner
	@PostMapping("/refreshAuth")
	public void refreshAuth(@RequestBody SysDataAuth sysDataAuth) {
		sysDataAuthService.refreshAuth(sysDataAuth);
	}

	/**
	 * @Description: 分页查询
	 * @Author: hgw
	 * @Date: 2022/6/7 14:59
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.baomidou.mybatisplus.core.metadata.IPage < com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysDataAuth>>
	 **/
	@Operation(summary = "分页查询")
	@GetMapping("/getSysDataAuthPage")
	public R<IPage<SysDataAuth>> getSysDataAuthPage(Page<SysDataAuth> page, SysDataAuth sysDataAuth) {
		return R.ok(sysDataAuthService.getSysDataAuthPage(page, sysDataAuth));
	}

	/**
	 * @Description: 分页查询
	 * @Author: hgw
	 * @Date: 2022/6/7 14:59
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.baomidou.mybatisplus.core.metadata.IPage < com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysDataAuth>>
	 **/
	@Operation(summary = "获取list")
	@GetMapping("/getSysDataAuthList")
	public R<List<SysDataAuth>> getSysDataAuthList(SysDataAuth sysDataAuth) {
		return R.ok(sysDataAuthService.getSysDataAuthList(sysDataAuth));
	}

	/**
	 * @Description: 详情/编辑获取原数据
	 * @Author: hgw
	 * @Date: 2022/6/7 14:59
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysDataAuth>
	 **/
	@Operation(summary = "详情/编辑获取原数据")
	@GetMapping("/{id:\\d+}")
	public R<SysDataAuthVO> getById(@PathVariable String id) {
		return sysDataAuthService.getByMainId(id);
	}

	/**
	 * @Description: 添加数据权限配置
	 * @Author: hgw
	 * @Date: 2022/6/7 14:57
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.Boolean>
	 **/
	@Operation(summary = "新增/编辑数据权限配置")
	@SysLog("系统管理-数据权限-新增/编辑数据权限配置")
	@PostMapping("/saveOrUpdate")
	@PreAuthorize("@pms.hasPermission('sys_permission_add')")
	public R<String> saveOrUpdate(@RequestBody SysDataAuthVO sysDataAuthVO) {
		return sysDataAuthService.saveDataAuth(sysDataAuthVO);
	}

	/**
	 * @Description: 删除数据权限配置
	 * @Author: hgw
	 * @Date: 2022/6/7 14:58
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.Boolean>
	 **/
	@Operation(summary = "删除数据权限配置")
	@SysLog("系统管理-数据权限-删除数据权限配置")
	@DeleteMapping("/{id:\\d+}")
	@PreAuthorize("@pms.hasPermission('sys_permission_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(sysDataAuthService.removeDataAuthById(id));
	}

}
