/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.social.entity.THolidayInfo;
import com.yifu.cloud.plus.v1.yifu.social.service.THolidayInfoService;
import com.yifu.cloud.plus.v1.yifu.social.vo.TEmployeeInsuranceWorkDayVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.THolidayInfoSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * @author fxj
 * @date 2024-05-08 11:52:08
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tholidayinfo")
@Tag(name = "节假日管理")
public class THolidayInfoController {

	private final THolidayInfoService tHolidayInfoService;


	/**
	 * 简单分页查询
	 *
	 * @param page         分页对象
	 * @param tHolidayInfo
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<THolidayInfo>> getTHolidayInfoPage(Page<THolidayInfo> page, THolidayInfoSearchVo tHolidayInfo) {
		return new R<>(tHolidayInfoService.getTHolidayInfoPage(page, tHolidayInfo));
	}

	/**
	 * 不分页查询
	 *
	 * @param tHolidayInfo
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	//@PreAuthorize("@pms.hasPermission('demo_tholidayinfo_get')" )
	public R<List<THolidayInfo>> getTHolidayInfoNoPage(@RequestBody THolidayInfoSearchVo tHolidayInfo) {
		return R.ok(tHolidayInfoService.noPageDiy(tHolidayInfo));
	}

	/**
	 * 通过id查询
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('demo_tholidayinfo_get')")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('demo_tholidayinfo_get')")
	public R<THolidayInfo> getById(@PathVariable("id") String id) {
		return R.ok(tHolidayInfoService.getById(id));
	}

	/**
	 * 新增
	 *
	 * @param tHolidayInfo
	 * @return R
	 */
	@Operation(summary = "新增", description = "新增：hasPermission('demo_tholidayinfo_add')")
	@SysLog("新增")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('demo_tholidayinfo_add')")
	public R<Boolean> save(@RequestBody THolidayInfo tHolidayInfo) {
		return R.ok(tHolidayInfoService.save(tHolidayInfo));
	}

	/**
	 * 修改
	 *
	 * @param tHolidayInfo
	 * @return R
	 */
	@Operation(summary = "修改", description = "修改：hasPermission('demo_tholidayinfo_edit')")
	@SysLog("修改")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('demo_tholidayinfo_edit')")
	public R<Boolean> updateById(@RequestBody THolidayInfo tHolidayInfo) {
		return R.ok(tHolidayInfoService.updateById(tHolidayInfo));
	}

	/**
	 * 通过id删除
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除", description = "通过id删除：hasPermission('demo_tholidayinfo_del')")
	@SysLog("通过id删除")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('demo_tholidayinfo_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tHolidayInfoService.removeById(id));
	}

	/**
	 * 批量导入
	 *
	 * @author fxj
	 * @date 2024-05-08 11:52:08
	 **/
	@SneakyThrows
	@Operation(description = "批量新增 hasPermission('demo_tholidayinfo-batch-import')")
	@SysLog("批量新增")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('demo_tholidayinfo-batch-import')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file) {
		return tHolidayInfoService.importDiy(file.getInputStream());
	}

	/**
	 * 批量导出
	 *
	 * @author fxj
	 * @date 2024-05-08 11:52:08
	 **/
	@Operation(description = "导出 hasPermission('demo_tholidayinfo-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('demo_tholidayinfo-export')")
	public void export(HttpServletResponse response, @RequestBody THolidayInfoSearchVo searchVo) {
		tHolidayInfoService.listExport(response, searchVo);
	}

	/**
	 * @Description: 定时任务1月1号生成年度假期表
	 * @Author: fxj
	 * @Date: 2024/5/8
	 **/
	@Operation(summary = "定时任务1月1号生成年度假期表", description = "定时任务1月1号生成年度假期表")
	@SysLog("定时任务1月1号生成年度假期表")
	@Inner
	@PostMapping("/inner/createHolidayInfo")
	public void createHolidayInfo() {
		tHolidayInfoService.createHolidayInfo();
	}

	/**
	 * @Author huyc
	 * @Description 查询日期的前一个/后一个工作日
	 * @Date 10:27 2025/4/7
	 **/
	@PostMapping("/selectBeforeOrAfterWorkDay")
	public R selectBeforeOrAfterWorkDay(@RequestBody TEmployeeInsuranceWorkDayVo vo) {
		return R.ok(tHolidayInfoService.selectBeforeOrAfterWorkDay(vo));
	}

	/**
	 * @Author huyc
	 * @Description 查询日期的前一个/后一个工作日
	 * @Date 10:27 2025/4/7
	 **/
	@Inner
	@PostMapping("/inner/getBeforeOrAfterWorkDay")
	public TEmployeeInsuranceWorkDayVo getBeforeOrAfterWorkDay(@RequestBody TEmployeeInsuranceWorkDayVo vo) {
		return tHolidayInfoService.selectBeforeOrAfterWorkDay(vo);
	}

	/**
	 * @Author huyc
	 * @Description 判断日期是否为节假日
	 * @Date 17:55 2025/4/7
	 **/
	@Inner
	@PostMapping("/inner/checkIsWorkDay")
	public Boolean checkIsWorkDay(@RequestBody TEmployeeInsuranceWorkDayVo vo) {
		return tHolidayInfoService.checkIsWorkDay(vo);
	}
}
