/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.csp.service.impl;

import cn.hutool.core.util.ArrayUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.csp.entity.TOperationLog;
import com.yifu.cloud.plus.v1.csp.mapper.TOperationLogMapper;
import com.yifu.cloud.plus.v1.csp.service.TOperationLogService;
import com.yifu.cloud.plus.v1.csp.vo.TOperationLogSearchVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;

/**
 * 入离职等的操作日志
 *
 * @author hgw
 * @date 2025-02-28 15:18:43
 */
@Log4j2
@Service
public class TOperationLogServiceImpl extends ServiceImpl<TOperationLogMapper, TOperationLog> implements TOperationLogService {
	/**
	 * 入离职等的操作日志简单分页查询
	 *
	 * @param tOperationLog 入离职等的操作日志
	 * @return
	 */
	@Override
	public IPage<TOperationLog> getTOperationLogPage(Page<TOperationLog> page, TOperationLogSearchVo tOperationLog) {
		return baseMapper.getTOperationLogPage(page, tOperationLog);
	}


	@Override
	public List<TOperationLog> noPageDiy(TOperationLogSearchVo searchVo) {
		LambdaQueryWrapper<TOperationLog> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TOperationLog::getId, idList);
		}
		if (searchVo.getLimitStart() >= 0 && searchVo.getLimitEnd() > 0) {
			wrapper.last(" limit " + searchVo.getLimitStart() + "," + searchVo.getLimitEnd());
		}
		wrapper.orderByAsc(TOperationLog::getCreateTime);
		return baseMapper.selectList(wrapper);
	}

	private LambdaQueryWrapper<TOperationLog> buildQueryWrapper(TOperationLogSearchVo entity) {
		LambdaQueryWrapper<TOperationLog> wrapper = Wrappers.lambdaQuery();
		if (ArrayUtil.isNotEmpty(entity.getCreateTimes())) {
			wrapper.ge(TOperationLog::getCreateTime, entity.getCreateTimes()[0]).le(TOperationLog::getCreateTime, entity.getCreateTimes()[1]);
		}
		if (Common.isNotNull(entity.getCreateName())) {
			wrapper.eq(TOperationLog::getCreateName, entity.getCreateName());
		}
		if (Common.isNotNull(entity.getMainId())) {
			wrapper.eq(TOperationLog::getMainId, entity.getMainId());
			wrapper.eq(TOperationLog::getType, CommonConstants.ZERO_STRING);
		}
		return wrapper;
	}

	/**
	 * @param mainId 关联ID
	 * @param type 传0，就是入离职申请（可以为空）
	 * @param title 标题
	 * @param operationTime 操作时间
	 * @param operationName 操作人
	 * @param remark 备注（可以为空）
	 * @Description:  新增日志
	 * @Author: hgw
	 * @Date: 2025/2/28 16:13
	 * @return: com.yifu.cloud.plus.v1.csp.entity.TOperationLog
	 **/
	@Override
	public TOperationLog saveLog(String mainId, String type, String title, LocalDateTime operationTime, String operationName, String remark) {
		YifuUser user = SecurityUtils.getUser();
		if (user == null || Common.isEmpty(user.getId())
				|| Common.isEmpty(mainId) || Common.isEmpty(title)
				|| Common.isEmpty(operationTime) || Common.isEmpty(operationName)) {
			return null;
		}
		TOperationLog log = new TOperationLog();
		log.setMainId(mainId);
		if (Common.isEmpty(type)) {
			type = CommonConstants.ZERO_STRING;
		}
		log.setType(type);
		log.setTitle(title);
		log.setOperationTime(operationTime);
		log.setOperationName(operationName);
		log.setRemark(remark);
		log.setCreateBy(user.getId());
		log.setCreateName(user.getNickname());
		log.setCreateTime(LocalDateTime.now());
		this.save(log);
		return log;
	}

	/**
	 * @param mainId 关联ID
	 * @param type 传0，就是入离职申请（可以为空）
	 * @param title 标题
	 * @param operationTime 操作时间
	 * @param operationName 操作人
	 * @param remark 备注（可以为空）
	 * @Description:  新增日志
	 * @Author: hgw
	 * @Date: 2025/2/28 16:13
	 * @return: com.yifu.cloud.plus.v1.csp.entity.TOperationLog
	 **/
	@Override
	public TOperationLog saveInnerLog(String mainId, String type, String title, LocalDateTime operationTime,
									  String operationName, String remark, String userId) {
		if (Common.isEmpty(mainId) || Common.isEmpty(title)
				|| Common.isEmpty(operationTime) || Common.isEmpty(operationName)) {
			return null;
		}
		TOperationLog log = new TOperationLog();
		log.setMainId(mainId);
		if (Common.isEmpty(type)) {
			type = CommonConstants.ZERO_STRING;
		}
		log.setType(type);
		log.setTitle(title);
		log.setOperationTime(operationTime);
		log.setOperationName(operationName);
		log.setRemark(remark);
		log.setCreateBy(userId);
		log.setCreateName(operationName);
		log.setCreateTime(LocalDateTime.now());
		this.save(log);
		return log;
	}

}
