package com.yifu.cloud.plus.v1.yifu.common.core.util;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

/**
 * 月份计算工具类
 * 提供根据当前时间向前推N个月的开始时间和结束时间计算功能
 *
 * @author fxj
 */
public class MonthCalculationUtils {

    /**
     * 私有构造函数，防止实例化
     */
    private MonthCalculationUtils() {
        throw new IllegalStateException("MonthCalculationUtils class");
    }

    /**
     * 根据给定的月份数，计算从当前时间往前推n个月的开始时间和结束时间
     *
     * @param months 往前推的月份数
     * @return 包含开始时间和结束时间的字符串数组，索引0为开始时间，索引1为结束时间
     */
    public static String[] calculateMonthRange(int months) {
        // 获取当前时间
        LocalDateTime now = LocalDateTime.now();
        
        // 计算目标月份的第一天 00:00:00
        LocalDateTime start = now.minusMonths(months)
                .withDayOfMonth(1)
                .withHour(0)
                .withMinute(0)
                .withSecond(0)
                .withNano(0);
        
        // 计算目标月份的最后一天 23:59:59
        LocalDateTime end = start.withDayOfMonth(start.toLocalDate().lengthOfMonth())
                .withHour(23)
                .withMinute(59)
                .withSecond(59)
                .withNano(0);
        
        // 格式化时间
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        
        return new String[]{
                start.format(formatter),
                end.format(formatter)
        };
    }
    
    /**
     * 根据给定的基准时间和月份数，计算从基准时间往前推n个月的开始时间和结束时间
     *
     * @param baseTime 基准时间
     * @param months 往前推的月份数
     * @return 包含开始时间和结束时间的字符串数组，索引0为开始时间，索引1为结束时间
     */
    public static String[] calculateMonthRangeFromBase(LocalDateTime baseTime, int months) {
        // 计算目标月份的第一天 00:00:00
        LocalDateTime start = baseTime.minusMonths(months)
                .withDayOfMonth(1)
                .withHour(0)
                .withMinute(0)
                .withSecond(0)
                .withNano(0);
        
        // 计算目标月份的最后一天 23:59:59
        LocalDateTime end = start.withDayOfMonth(start.toLocalDate().lengthOfMonth())
                .withHour(23)
                .withMinute(59)
                .withSecond(59)
                .withNano(0);
        
        // 格式化时间
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        
        return new String[]{
                start.format(formatter),
                end.format(formatter)
        };
    }
}