/*
 * Copyright (c) 2020 yifu4cloud Authors. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.yifu.cloud.plus.v1.yifu.admin.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.tree.Tree;
import cn.hutool.core.lang.tree.TreeNode;
import cn.hutool.core.lang.tree.TreeUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysDict;
import com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysDictItem;
import com.yifu.cloud.plus.v1.yifu.admin.api.vo.DictRedisVo;
import com.yifu.cloud.plus.v1.yifu.admin.mapper.SysDictItemMapper;
import com.yifu.cloud.plus.v1.yifu.admin.service.SysDictItemService;
import com.yifu.cloud.plus.v1.yifu.admin.service.SysDictService;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CacheConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.enums.DictTypeEnum;
import com.yifu.cloud.plus.v1.yifu.common.core.exception.ErrorCodes;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.MsgUtils;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.util.RedisUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.validation.constraints.NotNull;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 字典项
 *
 * @author lengleng
 * @date 2019/03/19
 */
@Log4j2
@Service
@RequiredArgsConstructor
public class SysDictItemServiceImpl extends ServiceImpl<SysDictItemMapper, SysDictItem> implements SysDictItemService {

	private final SysDictService dictService;

	@Autowired
	private RedisUtil redisUtil;

	/**
	 * 删除字典项
	 * @param id 字典项ID
	 * @return
	 */
	@Override
	@CacheEvict(value = CacheConstants.DICT_DETAILS, allEntries = true)
	public Boolean removeDictItem(Long id) {
		// 查询父节点为当前节点的节点
		List<SysDictItem> dictList = this.list(Wrappers.<SysDictItem>query().lambda().eq(SysDictItem::getParentId, id));
		Assert.isTrue(CollUtil.isEmpty(dictList), MsgUtils.getMessage(ErrorCodes.SYS_MENU_DELETE_EXISTING));
		// 根据ID查询字典ID
		SysDictItem dictItem = this.getById(id);
		SysDict dict = dictService.getById(dictItem.getDictId());
		// 系统内置
		Assert.state(!DictTypeEnum.SYSTEM.getType().equals(dict.getSystemFlag()),
				MsgUtils.getMessage(ErrorCodes.SYS_DICT_DELETE_SYSTEM));
		return this.removeById(id);
	}

	/**
	 * 更新字典项
	 * @param item 字典项
	 * @return
	 */
	@Override
	@CacheEvict(value = CacheConstants.DICT_DETAILS, key = "#item.type")
	public Boolean updateDictItem(SysDictItem item) {
		// 查询字典
		SysDict dict = dictService.getById(item.getDictId());
		// 系统内置
		Assert.state(!DictTypeEnum.SYSTEM.getType().equals(dict.getSystemFlag()),
				MsgUtils.getMessage(ErrorCodes.SYS_DICT_UPDATE_SYSTEM));
		if (this.updateById(item)){
			// 更新字典
			extractedRedis(item);
			return true;
		}
		return false;
	}

	private void extractedRedis(SysDictItem item) {
		Map<String, Object> resultMap = new HashMap<>();
		updateDictItem(CacheConstants.DICT_DETAILS+CommonConstants.COLON_STRING+item.getType(),resultMap);
		redisUtil.set(item.getType(),resultMap);
	}

	@CacheEvict(value = CacheConstants.DICT_DETAILS, allEntries = true)
	@Override
	public R<Boolean> saveDiy(SysDictItem sysDictItem) {
		int res = baseMapper.insert(sysDictItem);
		if (res < 0){
			return R.failed();
		}
		// 更新字典
		extractedRedis(sysDictItem);
		return R.ok();
	}

	@Override
	public R<String> getLableFronValue(String value, String type) {
		SysDictItem sysDictItem = baseMapper.selectOne(Wrappers.<SysDictItem>query().lambda()
				.eq(SysDictItem::getType,type).eq(SysDictItem::getValue,value)
				.eq(SysDictItem::getDelFlag,CommonConstants.ZERO_STRING)
				.eq(SysDictItem::getDisable,CommonConstants.ZERO_STRING));
		if (Common.isNotNull(sysDictItem)) {
			return R.ok(sysDictItem.getLabel());
		}
		return R.ok(CommonConstants.EMPTY_STRING);
	}

	@Override
	public List<DictRedisVo> getDictRedisVo() {
		return dictService.getDictRedisVo();
	}

	private void  updateDictItem(String type, Map<String,Object> resultMap){
		List<SysDictItem>  allList = baseMapper.selectList(Wrappers.<SysDictItem>query().lambda()
				.in(SysDictItem::getType,type)
				.eq(SysDictItem::getDelFlag,CommonConstants.ZERO_STRING)
				.eq(SysDictItem::getDisable,CommonConstants.ZERO_STRING));
		extractedCache(allList, resultMap);
	}

	/**
	 * 构建树查询 1. 不是懒加载情况，查询全部 2. 是懒加载，根据parentId 查询 2.1 父节点为空，则查询ID -1
	 * @param lazy 是否是懒加载
	 * @param parentId 父节点ID
	 * @return
	 */
	@Override
	public List<Tree<Long>> treeMenu(boolean lazy, Long parentId) {
		if (!lazy) {
			List<TreeNode<Long>> collect = baseMapper
					.selectList(Wrappers.<SysDictItem>lambdaQuery().orderByAsc(SysDictItem::getSortOrder)).stream()
					.map(getNodeFunction()).collect(Collectors.toList());

			return TreeUtil.build(collect, CommonConstants.MENU_TREE_ROOT_ID);
		}

		Long parent = parentId == null ? CommonConstants.MENU_TREE_ROOT_ID : parentId;

		List<TreeNode<Long>> collect = baseMapper
				.selectList(Wrappers.<SysDictItem>lambdaQuery().eq(SysDictItem::getParentId, parent)
						.orderByAsc(SysDictItem::getSortOrder))
				.stream().map(getNodeFunction()).collect(Collectors.toList());

		return TreeUtil.build(collect, parent);
	}

	@Override
	public IPage<SysDictItem> pageItem(Page page, SysDictItem sysDictItem) {
		IPage<SysDictItem> dictItemIPage = baseMapper.selectPage(page, buildQueryWrapper(sysDictItem));
		List<SysDictItem> pageRecords = dictItemIPage.getRecords();
		if (Common.isNotNull(pageRecords)){
			for (SysDictItem item:pageRecords){
				item.setParentLabelOfValue(dictService.getParentDicLabelById(item.getType(),item.getValue()));
			}
		}
		dictItemIPage.setRecords(pageRecords);
		return dictItemIPage;
	}

	@Override
	public R<Boolean> removeDictItemDiy(Long id) {
		SysDictItem entity = baseMapper.selectById(id);
		int res = dictService.selectChildDictItemCount(entity.getType(),entity.getValue());
		if (Common.isNotNull(entity) && dictService.selectChildDictItemCount(entity.getType(),entity.getValue()) > 0){
			return R.failed(MsgUtils.getMessage(ErrorCodes.SYS_DICT_ITEM_DELETE_EXIST_CHILD));
		}
		return R.ok(baseMapper.deleteById(id) > 0);
	}

	@Override
	public R<Map<String, Object>> getParentDictItemByTypes(String itemTypes) {
		if (Common.isEmpty(itemTypes)){
			return R.failed(CommonConstants.RESULT_EMPTY);
		}
		Map<String, Object> resultMap = new HashMap<>();
		List<SysDictItem>  allList = baseMapper.selectList(Wrappers.<SysDictItem>query().lambda()
				.in(SysDictItem::getType,Common.initStrToList(itemTypes,CommonConstants.COMMA_STRING))
				.eq(SysDictItem::getDelFlag,CommonConstants.ZERO_STRING));
		extracted(allList, resultMap);
		return R.ok(resultMap);
	}
	/**
	 * @Author fxj
	 * @Description 获取所有字典项数据
	 * @Date 17:31 2022/6/22
	 * @Param
	 * @return
	**/
	@CacheEvict(value = CacheConstants.DICT_ITEM_DETAILS, key = CacheConstants.DICT_ITEM_DETAILS)
	@Override
	public R<Map<String, Object>> getAllDictItem() {
		return R.ok(getAllDictItemSub());
	}

	@Override
	public Map<String, Object> getAllDictItemSub(){
		List<SysDictItem> allList = baseMapper.selectList(Wrappers.<SysDictItem>query().lambda()
				.eq(SysDictItem::getDelFlag,CommonConstants.ZERO_STRING)
				.eq(SysDictItem::getDisable,CommonConstants.ZERO_STRING));
		Map<String, Object> resultMap = new HashMap<>();
		extracted(allList, resultMap);
		return  resultMap;
	}
	/**
	 * @Author fxj
	 * @Description  清理字典项数据
	 * @Date 17:39 2022/6/22
	**/
	@CacheEvict(value = CacheConstants.DICT_ITEM_DETAILS, allEntries = true)
	@Override
	public void clearDictItemCache() {
		log.info("字典数据加入缓存初始化开始...");
		Map<String, Object> dictItem = this.getAllDictItemSub();
		if (Common.isNotNull(dictItem)){
			for (Map.Entry<String,Object> entry:dictItem.entrySet()){
				redisUtil.set(entry.getKey(),entry.getValue());
			}
		}
		log.info("字典数据加入缓存初始化结束...");
	}
	private void extractedCache(List<SysDictItem> allList, Map<String, Object> resultMap) {
		if (Common.isNotNull(allList)){
			if (Common.isEmpty(resultMap)){
				resultMap = new HashMap<>();
			}
			for (SysDictItem item : allList) {
				resultMap.put(item.getValue(), item.getLabel());
			}
		}
	}
	private void extracted(List<SysDictItem> allList, Map<String, Object> resultMap) {
		if (Common.isNotNull(allList)){
			for (SysDictItem item : allList) {
				String dictCode = item.getType();
				if (resultMap.get(dictCode) == null) {
					Map<String, Object> dictMap = new HashMap<String, Object>();
					dictMap.put(item.getValue(), item.getLabel());
					resultMap.put(dictCode, dictMap);
				} else {
					Map dictMap = (Map) resultMap.get(dictCode);
					dictMap.put(item.getValue(), item.getLabel());
				}
			}
		}
	}
	@Override
	public R<Map<String, Map<String, String>>> getDictToLable() {
		Map<String, Map<String, String>> resultMap = this.getStringMapMap(false);
		return R.ok(resultMap);
	}

	@Override
	public R<Map<String, Map<String, String>>> getDictToValue() {
		Map<String, Map<String, String>> resultMap = this.getStringMapMap(true);
		return R.ok(resultMap);
	}

	private Map<String, Map<String, String>> getStringMapMap(boolean type) {
		List<SysDictItem> allList = baseMapper.selectList(Wrappers.<SysDictItem>query().lambda()
		.eq(SysDictItem::getDelFlag,CommonConstants.STATUS_NORMAL).eq(SysDictItem::getDisable,CommonConstants.ZERO_STRING));
		Map<String, Map<String, String>> resultMap = new HashMap<>();
		doAssemble(allList, resultMap, type);
		return resultMap;
	}

	/**
	 * @Description: 组装
	 * @Author: hgw
	 * @Date: 2022/6/22 18:12
	 * @return: void
	 **/
	private void doAssemble(List<SysDictItem> allList, Map<String, Map<String, String>> resultMap, boolean type) {
		if (Common.isNotNull(allList)){
			String dictCode;
			Map<String, String> dictMap;
			for (SysDictItem item : allList) {
				dictCode = item.getType();
				dictMap = resultMap.get(dictCode);
				if (dictMap == null) {
					dictMap = new HashMap<>();
				}
				if (type) {
					dictMap.put(item.getValue(), item.getLabel());
				} else {
					dictMap.put(item.getLabel(), item.getValue());
				}
				resultMap.put(dictCode, dictMap);
			}
		}
	}

	/**
	 * 构建查询的 wrapper
	 * @param sysDictItem 查询条件
	 * @return LambdaQueryWrapper
	 */
	private LambdaQueryWrapper buildQueryWrapper(SysDictItem sysDictItem) {
		LambdaQueryWrapper<SysDictItem> wrapper = Wrappers.lambdaQuery();
		if (StrUtil.isNotBlank(sysDictItem.getType())){
			wrapper.eq(SysDictItem::getType,sysDictItem.getType());
		}
		wrapper.orderByAsc(SysDictItem::getSortOrder);
		return wrapper;
	}

	@NotNull
	private Function<SysDictItem, TreeNode<Long>> getNodeFunction() {
		return sysDictItem -> {
			TreeNode<Long> node = new TreeNode<>();
			node.setId(sysDictItem.getId());
			node.setName(sysDictItem.getLabel());
			node.setParentId(sysDictItem.getParentId());
			node.setWeight(sysDictItem.getSortOrder());
			// 扩展属性
			Map<String, Object> extra = new HashMap<>();
			extra.put("value", sysDictItem.getValue());
			extra.put("dictId", sysDictItem.getDictId());
			extra.put("type", sysDictItem.getType());
			extra.put("label", sysDictItem.getLabel());
			extra.put("sortOrder", sysDictItem.getSortOrder());
			node.setExtra(extra);
			return node;
		};
	}
	@Override
	public R<Map<String, List>> getDictItemsByTypes(String itemTypes) {
		if (Common.isEmpty(itemTypes)){
			return R.failed(CommonConstants.RESULT_EMPTY);
		}
		Map<String, List> resultMap = new HashMap<>();
		List<SysDictItem>  allList = baseMapper.selectList(Wrappers.<SysDictItem>query().lambda()
				.in(SysDictItem::getType,Common.initStrToList(itemTypes,CommonConstants.COMMA_STRING))
				.eq(SysDictItem::getDelFlag,CommonConstants.ZERO_STRING));
		if (Common.isNotNull(allList)){
			List temp;
			for (SysDictItem item : allList) {
				String dictCode = item.getType();
				if (resultMap.get(dictCode) == null) {
					List list = new ArrayList<>();
					list.add(item);
					resultMap.put(dictCode, list);
				} else {
					temp =  resultMap.get(dictCode);
					temp.add(item);
					resultMap.put(dictCode, temp);
				}
			}
		}
		return R.ok(resultMap);
	}
}
