/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.admin.api.vo.SysCspDeptVo;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeProjectBelongDept;
import com.yifu.cloud.plus.v1.yifu.archives.service.TEmployeeProjectBelongDeptService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.EmployeeProjectScpVO;
import com.yifu.cloud.plus.v1.yifu.archives.vo.TEmployeeProjectBelongDeptSearchVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.UserPermissionVo;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.Map;


/**
 * 项目档案所属部门表
 *
 * @author hgw
 * @date 2025-02-28 15:03:17
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/temployeeprojectbelongdept")
@Tag(name = "项目档案所属部门表管理")
public class TEmployeeProjectBelongDeptController {

	private final TEmployeeProjectBelongDeptService tEmployeeProjectBelongDeptService;


	/**
	 * 简单分页查询
	 *
	 * @param page                       分页对象
	 * @param tEmployeeProjectBelongDept 项目档案所属部门表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@PostMapping("/page")
	public R<IPage<EmployeeProjectScpVO>> getTEmployeeProjectBelongDeptPage(Page<EmployeeProjectScpVO> page
			, @RequestBody TEmployeeProjectBelongDeptSearchVo tEmployeeProjectBelongDept) {
		return new R<>(tEmployeeProjectBelongDeptService.getTEmployeeProjectBelongDeptPage(page, tEmployeeProjectBelongDept));
	}

	/**
	 * @param id 项目id
	 * @Description: 项目档案详情，获取所属部门，需要额外请求接口：
	 * @Author: hgw
	 * @Date: 2025/3/3 9:38
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeProjectBelongDept>
	 **/
	@Operation(summary = "通过id查询", description = "通过id查询")
	@GetMapping("/{id}")
	public R<TEmployeeProjectBelongDept> getById(@PathVariable("id") String id) {
		return R.ok(tEmployeeProjectBelongDeptService.getById(id));
	}

	/**
	 * @param tEmployeeProjectBelongDept
	 * @Description: 项目组织维护（新增、修改、清空 项目档案所属部门表）
	 * @Author: hgw
	 * @Date: 2025/3/3 9:37
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "项目组织维护", description = "项目组织维护")
	@SysLog("项目组织维护")
	@PostMapping("/saveOrUpdate")
	public R<String> saveOrUpdate(@RequestBody TEmployeeProjectBelongDept tEmployeeProjectBelongDept) {
		if (Common.isEmpty(tEmployeeProjectBelongDept.getId())) {
			return R.failed("项目ID必填");
		}
		if (Common.isEmpty(tEmployeeProjectBelongDept.getEmpDeptid())) {
			tEmployeeProjectBelongDeptService.removeById(tEmployeeProjectBelongDept.getId());
			return R.ok(CommonConstants.SAVE_SUCCESS);
		} else if (Common.isEmpty(tEmployeeProjectBelongDept.getEmpDeptname())) {
			return R.failed("所属部门名称必填");
		}
		tEmployeeProjectBelongDeptService.saveOrUpdate(tEmployeeProjectBelongDept);
		return R.ok(CommonConstants.SAVE_SUCCESS);
	}

	/**
	 * CSP导出花名册
	 * @author hgw
	 * @date 2025-02-28 15:03:17
	 **/
	@Operation(description = "CSP导出花名册")
	@PostMapping("/export")
	public void export(HttpServletResponse response, @RequestBody TEmployeeProjectBelongDeptSearchVo searchVo) {
		tEmployeeProjectBelongDeptService.listExport(response,searchVo);
	}

	/**
	 * 获取指定部门下花名册总数
	 * @param tEmployeeProjectBelongDept 花名册查询
	 * @return R<Integer> 花名册总数
	 * @author chenyuxi
	 * @since 1.9.7
	 */
	@GetMapping(value = "/getDeptCount")
	public R<Integer> getDeptCount(TEmployeeProjectBelongDeptSearchVo tEmployeeProjectBelongDept) {
		if(Common.isEmpty(tEmployeeProjectBelongDept.getEmpDeptId())){
			R.failed("所属部门id不能为空");
		}
		if(Common.isEmpty(tEmployeeProjectBelongDept.getDeptId())){
			R.failed("项目ID不能为空");
		}
		return R.ok(tEmployeeProjectBelongDeptService.getDeptCount(tEmployeeProjectBelongDept));
	}

	/**
	 * 根据权限获取对应部门ID下的花名册总数Map
	 * @param tEmployeeProjectBelongDept 花名册查询
	 * @return R<Integer> 花名册总数
	 * @author chenyuxi
	 * @since 1.9.7
	 */
	@GetMapping(value = "/getDeptCountMap")
	public R<Map<String,Integer>> getDeptCountMap(TEmployeeProjectBelongDeptSearchVo tEmployeeProjectBelongDept) {
		if(Common.isEmpty(tEmployeeProjectBelongDept.getDeptId())){
			R.failed("项目ID不能为空");
		}
		YifuUser user = SecurityUtils.getUser();
		if (user == null || Common.isEmpty(user.getId())) {
			return R.failed(CommonConstants.PLEASE_LOG_IN);
		}
		List<UserPermissionVo> cspPermList = user.getCspPermMap();
		if(Common.isEmpty(cspPermList)){
			return R.failed("获取用户权限失败！");
		}
		// 获取用户在当前项目下的权限
		UserPermissionVo authority = null;
		for (UserPermissionVo userPermissionVo : cspPermList) {
			if(tEmployeeProjectBelongDept.getDeptId().equals(userPermissionVo.getProjectId())){
				authority = userPermissionVo;
			}
		}
		if(Common.isEmpty(authority)){
			return R.failed("获取用户在当前项目下的权限失败！");
		}
		return R.ok(tEmployeeProjectBelongDeptService.getDeptCountMap(tEmployeeProjectBelongDept,authority));
	}

	/**
	 * 没有关联部门的花名册总数
	 * @param tEmployeeProjectBelongDept 花名册查询
	 * @return R<Integer> 花名册总数
	 * @author chenyuxi
	 * @since 1.9.7
	 */
	@GetMapping(value = "/getNoneDeptCount")
	public R<Integer> getNoneDeptCount(TEmployeeProjectBelongDeptSearchVo tEmployeeProjectBelongDept) {
		if(Common.isEmpty(tEmployeeProjectBelongDept.getDeptId())){
			R.failed("项目ID不能为空");
		}
		return R.ok(tEmployeeProjectBelongDeptService.getNoneDeptCount(tEmployeeProjectBelongDept));
	}


	/**
	 * @Author huych
	 * @Description 根据项目id和用户id查询csp部门信息
	 * @Date 14:27 2025/3/10
	 * @Param
	 * @return
	 **/
	@Inner
	@PostMapping("/inner/saveOrDeleteDeptRelation")
	public Boolean saveOrDeleteDeptRelation(@RequestBody SysCspDeptVo vo) {
		return tEmployeeProjectBelongDeptService.saveOrDeleteDeptRelation(vo);
	}
}
