/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.social.entity.TSocialSoldierLog;
import com.yifu.cloud.plus.v1.yifu.social.service.TSocialSoldierLogService;
import com.yifu.cloud.plus.v1.yifu.social.vo.TSocialSoldierLogSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 自动化日志
 *
 * @author hgw
 * @date 2024-08-30 10:27:16
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tsocialsoldierlog")
@Tag(name = "自动化日志管理")
public class TSocialSoldierLogController {

	private final TSocialSoldierLogService tSocialSoldierLogService;


	/**
	 * 简单分页查询
	 *
	 * @param page              分页对象
	 * @param tSocialSoldierLog 自动化日志
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TSocialSoldierLog>> getTSocialSoldierLogPage(Page<TSocialSoldierLog> page, TSocialSoldierLogSearchVo tSocialSoldierLog) {
		return new R<>(tSocialSoldierLogService.getTSocialSoldierLogPage(page, tSocialSoldierLog));
	}

	/**
	 * 不分页查询
	 *
	 * @param tSocialSoldierLog 自动化日志
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	public R<List<TSocialSoldierLog>> getTSocialSoldierLogNoPage(@RequestBody TSocialSoldierLogSearchVo tSocialSoldierLog) {
		return R.ok(tSocialSoldierLogService.noPageDiy(tSocialSoldierLog));
	}

	/**
	 * 通过id查询自动化日志
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('social_tsocialsoldierlog_get')")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tsocialsoldierlog_get')")
	public R<TSocialSoldierLog> getById(@PathVariable("id") String id) {
		return R.ok(tSocialSoldierLogService.getById(id));
	}

	/**
	 * 新增自动化日志
	 *
	 * @param tSocialSoldierLog 自动化日志
	 * @return R
	 */
	@Operation(summary = "新增自动化日志", description = "新增自动化日志：hasPermission('social_tsocialsoldierlog_add')")
	@SysLog("新增自动化日志")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('social_tsocialsoldierlog_add')")
	public R<Boolean> save(@RequestBody TSocialSoldierLog tSocialSoldierLog) {
		return R.ok(tSocialSoldierLogService.save(tSocialSoldierLog));
	}

	/**
	 * 修改自动化日志
	 *
	 * @param tSocialSoldierLog 自动化日志
	 * @return R
	 */
	@Operation(summary = "修改自动化日志", description = "修改自动化日志：hasPermission('social_tsocialsoldierlog_edit')")
	@SysLog("修改自动化日志")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('social_tsocialsoldierlog_edit')")
	public R<Boolean> updateById(@RequestBody TSocialSoldierLog tSocialSoldierLog) {
		return R.ok(tSocialSoldierLogService.updateById(tSocialSoldierLog));
	}

	/**
	 * 通过id删除自动化日志
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除自动化日志", description = "通过id删除自动化日志：hasPermission('social_tsocialsoldierlog_del')")
	@SysLog("通过id删除自动化日志")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tsocialsoldierlog_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tSocialSoldierLogService.removeById(id));
	}

	/**
	 * 自动化日志 批量导出
	 *
	 * @author hgw
	 * @date 2024-08-30 10:27:16
	 **/
	@Operation(description = "导出自动化日志 hasPermission('social_tsocialsoldierlog_export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('social_tsocialsoldierlog_export')")
	public void export(HttpServletResponse response, @RequestBody TSocialSoldierLogSearchVo searchVo) {
		tSocialSoldierLogService.listExport(response, searchVo);
	}
}
