/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import com.yifu.cloud.plus.v1.yifu.social.entity.TSocialAlertInfo;
import com.yifu.cloud.plus.v1.yifu.social.service.TSocialAlertInfoService;
import com.yifu.cloud.plus.v1.yifu.social.vo.TSocialAlertInfoSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * @author fxj
 * @date 2025-10-15 14:50:53
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tsocialalertinfo")
@Tag(name = "社保办理失败通知信息")
public class TSocialAlertInfoController {

	private final TSocialAlertInfoService tSocialAlertInfoService;


	/**
	 * 简单分页查询
	 *
	 * @param page             分页对象
	 * @param tSocialAlertInfo
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TSocialAlertInfo>> getTSocialAlertInfoPage(Page<TSocialAlertInfo> page, TSocialAlertInfoSearchVo tSocialAlertInfo) {
		return new R<>(tSocialAlertInfoService.getTSocialAlertInfoPage(page, tSocialAlertInfo));
	}

	/**
	 * 不分页查询
	 *
	 * @param tSocialAlertInfo
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	public R<List<TSocialAlertInfo>> getTSocialAlertInfoNoPage(@RequestBody TSocialAlertInfoSearchVo tSocialAlertInfo) {
		YifuUser user = SecurityUtils.getUser();
		if (null == user || null == user.getId()){
			return R.failed(CommonConstants.USER_FAIL);
		}
		tSocialAlertInfo.setReceiveUser(user.getId());
		return R.ok(tSocialAlertInfoService.noPageDiy(tSocialAlertInfo));
	}

	/**
	 * 通过id查询
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('social_tsocialalertinfo_get')")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tsocialalertinfo_get')")
	public R<TSocialAlertInfo> getById(@PathVariable("id") String id) {
		return R.ok(tSocialAlertInfoService.getById(id));
	}

	/**
	 * 新增
	 *
	 * @param tSocialAlertInfo
	 * @return R
	 */
	@Operation(summary = "新增", description = "新增：hasPermission('social_tsocialalertinfo_add')")
	@SysLog("新增")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('social_tsocialalertinfo_add')")
	public R<Boolean> save(@RequestBody TSocialAlertInfo tSocialAlertInfo) {
		return R.ok(tSocialAlertInfoService.save(tSocialAlertInfo));
	}

	/**
	 * 修改
	 *
	 * @param tSocialAlertInfo
	 * @return R
	 */
	@Operation(summary = "修改", description = "修改：hasPermission('social_tsocialalertinfo_edit')")
	@SysLog("修改")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('social_tsocialalertinfo_edit')")
	public R<Boolean> updateById(@RequestBody TSocialAlertInfo tSocialAlertInfo) {
		return R.ok(tSocialAlertInfoService.updateById(tSocialAlertInfo));
	}

	/**
	 * 通过id删除
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除", description = "通过id删除：hasPermission('social_tsocialalertinfo_del')")
	@SysLog("通过id删除")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tsocialalertinfo_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tSocialAlertInfoService.removeById(id));
	}

	/**
	 * 批量导入
	 *
	 * @author fxj
	 * @date 2025-10-15 14:50:53
	 **/
	@SneakyThrows
	@Operation(description = "批量新增 hasPermission('social_tsocialalertinfo-batch-import')")
	@SysLog("批量新增")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('social_tsocialalertinfo-batch-import')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file) {
		return tSocialAlertInfoService.importDiy(file.getInputStream());
	}

	/**
	 * 批量导出
	 *
	 * @author fxj
	 * @date 2025-10-15 14:50:53
	 **/
	@Operation(description = "导出 hasPermission('social_tsocialalertinfo-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('social_tsocialalertinfo-export')")
	public void export(HttpServletResponse response, @RequestBody TSocialAlertInfoSearchVo searchVo) {
		tSocialAlertInfoService.listExport(response, searchVo);
	}

	/**
	 * @Author fxj
	 * @Description 每天刷新办理失败的数据  并发送给申请人，如果申请人异常发给地市办理人，如果地市办理人异常 不处理
	 * @Date 15:14 2025/10/15
	 * @Param
	 * @return
	**/
	@Operation(description = "每天刷新办理失败的数据")
	@Inner
	@PostMapping("/inner/createAndSendSocialAlert")
	public void createAndSendSocialAlert() {
		tSocialAlertInfoService.createAndSendSocialAlert();
	}

	@Operation(description = "每天刷新办理失败的数据")
	@PostMapping("/createAndSendSocialAlert2")
	public void createAndSendSocialAlert2() {
		tSocialAlertInfoService.createAndSendSocialAlert();
	}
}
