package com.yifu.cloud.plus.v1.business.service.vacation.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.bean.copier.CopyOptions;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.business.dto.vacation.AddVacationMonitorClearLogDTO;
import com.yifu.cloud.plus.v1.business.entity.vacation.TVacationMonitor;
import com.yifu.cloud.plus.v1.business.entity.vacation.TVacationMonitorClearLog;
import com.yifu.cloud.plus.v1.business.mapper.vacation.TVacationMonitorClearLogMapper;
import com.yifu.cloud.plus.v1.business.service.vacation.TVacationMonitorClearLogService;
import com.yifu.cloud.plus.v1.business.service.vacation.TVacationMonitorService;
import com.yifu.cloud.plus.v1.business.vo.vacation.TVacationMonitorClearLogVO;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import lombok.AllArgsConstructor;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

import static com.yifu.cloud.plus.v1.business.util.BusinessResultErrorMsg.*;


/**
 * @description: 假期监控清零日志表 service实现层
 * @author: huyc
 * @date: 2023/1/13
*/
@Service
@AllArgsConstructor
public class TVacationMonitorClearLogServiceImpl extends ServiceImpl<TVacationMonitorClearLogMapper, TVacationMonitorClearLog>
implements TVacationMonitorClearLogService {

    private final TVacationMonitorService vacationMonitorService;

    /**
     * @description: 查询-列表: 查询当前年假所有清零记录
     * @param vacationMonitorId 年假监控id
     * @return: R<List<TVacationMonitorClearLogVO>>
	 * @Author: huyc
	 * @Date: 2023/1/6 11:14
     */
    @Override
    public R<List<TVacationMonitorClearLogVO>> getList(String vacationMonitorId) {
        if (vacationMonitorId == null) {
            return R.ok(null,VACATION_MONITOR_ID_IS_NOT_NULL);
        }
        List<TVacationMonitorClearLog> list = this.lambdaQuery()
                .eq(TVacationMonitorClearLog::getVacationMonitorId, vacationMonitorId)
                .orderByDesc(TVacationMonitorClearLog::getClearTime)
                .list();

        List<TVacationMonitorClearLogVO> voList = list.stream().map(v -> {
            TVacationMonitorClearLogVO vo = new TVacationMonitorClearLogVO();
            BeanUtil.copyProperties(v, vo, CopyOptions.create().setIgnoreCase(true));
            return vo;
        }).collect(Collectors.toList());

        return R.ok(voList);
    }

    /**
     * @description: 清零说明: 记录年假清零说明
     * @param dto
     * @return: R<Boolean>
	 * @Author: huyc
	 * @Date: 2023/1/6 11:14
     */
    @Override
    public R<Boolean> clearNote(AddVacationMonitorClearLogDTO dto) {
        YifuUser user = SecurityUtils.getUser();
        if (user == null) {
            return R.failed(USER_NOT_LOGIN);
        }

        if (dto.getVacationMonitorId() == null) {
            return R.ok(null,VACATION_MONITOR_ID_IS_NOT_NULL);
        }
        TVacationMonitor vacationMonitor = this.vacationMonitorService.getById(dto.getVacationMonitorId());

        if (vacationMonitor == null) {
            return R.failed(VACATION_MONITOR_DATA_IS_NOT_EXIST);
        }

        TVacationMonitorClearLog log = new TVacationMonitorClearLog();
        log.setClearNote(dto.getClearNote());
        log.setVacationMonitorId(dto.getVacationMonitorId());
        log.setVacationYear(vacationMonitor.getVacationYear());
        log.setClearDuration(vacationMonitor.getNotUsedVacationDuration());
        log.setClearUser(user.getNickname());
        log.setClearUserId(user.getId());
        log.setClearTime(LocalDateTime.now());

        boolean isSuccess = this.save(log);
        return R.ok(isSuccess,SUCCESS);
    }

}




