/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.vo.SocialFundVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.MenuUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import com.yifu.cloud.plus.v1.yifu.salary.vo.ListStringVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TInsuranceUnpurchasePersonCardVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.UpdateSocialFoundVo;
import com.yifu.cloud.plus.v1.yifu.social.entity.TSocialFundInfo;
import com.yifu.cloud.plus.v1.yifu.social.service.TSocialFundInfoService;
import com.yifu.cloud.plus.v1.yifu.social.vo.DispatchOneDetailVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.TSocialFundInfoDetailVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.TSocialFundInfoOneVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.TSocialFundInfoSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 社保公积金查询表
 *
 * @author fxj
 * @date 2022-07-15 11:38:05
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tsocialfundinfo" )
@Tag(name = "社保公积金查询表管理")
public class TSocialFundInfoController {

    private final TSocialFundInfoService tSocialFundInfoService;

	private final MenuUtil menuUtil;


    /**
     * 简单分页查询
     * @param page 分页对象
     * @param tSocialFundInfo 社保公积金查询表
     * @return
     */
    @Operation(description = "简单分页查询")
    @GetMapping("/page")
    public R<IPage<TSocialFundInfo>> getTSocialFundInfoPage(Page<TSocialFundInfo> page, TSocialFundInfoSearchVo tSocialFundInfo) {
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, tSocialFundInfo);
        return new R<>(tSocialFundInfoService.getTSocialFundInfoPage(page,tSocialFundInfo));
    }
    /**
     * 不分页查询
     * @param tSocialFundInfo 社保公积金查询表
     * @return
     */
    @Operation(summary = "不分页查询", description = "不分页查询")
    @PostMapping("/noPage" )
    public R<List<TSocialFundInfo>> getTSocialFundInfoNoPage(@RequestBody TSocialFundInfoSearchVo tSocialFundInfo) {
        return R.ok(tSocialFundInfoService.noPageDiy(tSocialFundInfo));
    }

	/**
	 * 重新派单明细查询
	 * @param tSocialFundInfo 社保公积金查询表
	 * @return
	 */
	@Operation(summary = "重新派单明细查询", description = "重新派单明细查询")
	@PostMapping("/dispatchAgainInfo" )
	public R<DispatchOneDetailVo> getTSocialFundInfoOne(@RequestBody TSocialFundInfoOneVo tSocialFundInfo) {
		return R.ok(tSocialFundInfoService.getTSocialFundInfoOne(tSocialFundInfo));
	}

    /**
     * 通过id查询社保公积金查询表
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id查询", description = "通过id查询：hasPermission('tsocialfundinfo_get')")
    @GetMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('tsocialfundinfo_get')" )
    public R<TSocialFundInfo> getById(@PathVariable("id" ) String id) {
        return R.ok(tSocialFundInfoService.getById(id));
    }

    /**
	 * 社保公积金查询表 批量导入
	 *
     * @author fxj
     * @date 2022-07-15 11:38:05
	 **/
    @SneakyThrows
    @Operation(description = "批量调基 hasPermission('tsocialfundinfo-batch-import')")
    @SysLog("批量调基")
    @PostMapping("/importListAdd")
    @PreAuthorize("@pms.hasPermission('tsocialfundinfo-batch-import')")
    public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file, @RequestParam String doMonth){
        return tSocialFundInfoService.importDiy(file.getInputStream(), doMonth);
    }

    /**
     * 社保公积金查询表 批量导出
     * @author fxj
     * @date 2022-07-15 11:38:05
     **/
    @Operation(description = "导出社保公积金查询表 hasPermission('tsocialfundinfo-export')")
    @PostMapping("/export")
    @PreAuthorize("@pms.hasPermission('tsocialfundinfo-export')")
    public void export(HttpServletResponse response, @RequestBody TSocialFundInfoSearchVo searchVo) {
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, searchVo);
		if (Common.isNotNull(searchVo.getAuthSql()) && searchVo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
			searchVo.setAuthSql(searchVo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "a.SETTLE_DOMAIN"));
		}
		tSocialFundInfoService.listExport(response,searchVo);
    }

	/**
	 * @param id 社保公积金查询表的主键
	 * @Description: 根据id查询详情
	 * @Author: hgw
	 * @Date: 2022/7/21 16:08
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.social.vo.TSocialFundInfoDetailVo>
	 **/
	@Operation(description = "前端使用：根据id查询详情")
    @PostMapping("/getDetailById")
    public R<TSocialFundInfoDetailVo> getDetailById(@RequestParam String id) {
		return R.ok(tSocialFundInfoService.getDetailById(id));
    }

	/**
	 * @param list
	 * @Description: 更新项目单位信息
	 * @Author: huyc
	 * @Date: 2022/8/25
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updateSocialAndFoundInfo")
	public void updateSocialAndFoundInfo(@RequestBody UpdateSocialFoundVo list) {
		tSocialFundInfoService.updateSocialAndFoundInfo(list);
	}

	/**
	 * @Description: 查找社保公积金，是否都派减或为空，允许直接审核通过合同——盛宇2022年11月28日09:18:38
	 * @Author: hgw
	 * @Date: 2022/11/28 9:18
	 * @return: void
	 **/
	@Inner
	@PostMapping("/inner/selectSocialFoundByContract")
	public int selectSocialFoundByContract(@RequestBody TSocialFundInfo sf) {
		return tSocialFundInfoService.selectSocialFoundByContract(sf);
	}

	/**
	 * @param cardVo
	 * @Description: 判断员工是否存在工伤社保数据
	 * @Author: huyc
	 * @Date: 2025/1/13
	 * @return:
	 **/
	@Inner
	@PostMapping("/inner/selectExitSocialInjury")
	public Boolean selectExitSocialInjury(@RequestBody TInsuranceUnpurchasePersonCardVo cardVo) {
		return tSocialFundInfoService.selectExitSocialInjury(cardVo);
	}

	/**
	 * @Description: 获取存在的工伤社保数据
	 * @Author: hgw
	 * @Date: 2025-1-15 16:17:31
	 * @return:
	 **/
	@Inner
	@PostMapping("/inner/selectWorkInjuryHandleAll")
	public ListStringVo selectWorkInjuryHandleAll() {
		return tSocialFundInfoService.selectWorkInjuryHandleAll();
	}

	/**
	 * 获取社保和公积金列表信息通过身份证集合
	 * @Author hgw
	 * @Date 2020-08-25
	 * @return
	 **/
	@SysLog("获取社保和公积金列表信息通过身份证集合")
	@PostMapping(value = "/inner/getSocialAndFundReduceListByEmpIdCardSet")
	@Inner
	public SocialFundVo getSocialAndFundReduceListByEmpIdCardSet(@RequestBody SocialFundVo vo) {
		if (Common.isNotNull(vo.getEmpIdCardSet())) {
			return tSocialFundInfoService.getSocialAndFundReduceListByEmpIdCardSet(vo.getEmpIdCardSet());
		}
		return new SocialFundVo();
	}

}
