/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.OSSUtil;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.social.entity.TAutoPaymentInfo;
import com.yifu.cloud.plus.v1.yifu.social.service.TAutoPaymentInfoService;
import com.yifu.cloud.plus.v1.yifu.social.vo.TAutoPaymentInfoSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.net.URL;
import java.util.List;


/**
 * 自动化实缴记录表
 *
 * @author fxj
 * @date 2024-05-24 10:56:42
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tautopaymentinfo")
@Tag(name = "自动化实缴记录表管理")
public class TAutoPaymentInfoController {

	private final TAutoPaymentInfoService tAutoPaymentInfoService;

	private final OSSUtil ossUtil;

	/**
	 * 简单分页查询
	 *
	 * @param page             分页对象
	 * @param tAutoPaymentInfo 自动化实缴记录表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TAutoPaymentInfo>> getTAutoPaymentInfoPage(Page<TAutoPaymentInfo> page, TAutoPaymentInfoSearchVo tAutoPaymentInfo) {
		return new R<>(tAutoPaymentInfoService.getTAutoPaymentInfoPage(page, tAutoPaymentInfo));
	}

	/**
	 * 获取附件下载地址
	 * @Author hgw
	 * @Date 2024-6-6 09:42:47
	 **/
	@Schema(description = "附件预览下载地址")
	@SysLog("附件预览下载地址")
	@PostMapping("/getOssFileUrl")
	public R<String> ossFileUrl(@RequestParam String attaSrc) {
		URL url = ossUtil.getObjectUrl(null, attaSrc);
		String urlStr = url.toString().replace("http","https");
		return new R<>(urlStr);
	}

	/**
	 * 不分页查询
	 *
	 * @param tAutoPaymentInfo 自动化实缴记录表
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	public R<List<TAutoPaymentInfo>> getTAutoPaymentInfoNoPage(@RequestBody TAutoPaymentInfoSearchVo tAutoPaymentInfo) {
		return R.ok(tAutoPaymentInfoService.noPageDiy(tAutoPaymentInfo));
	}

	/**
	 * 通过id查询自动化实缴记录表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('social_tautopaymentinfo_get')")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tautopaymentinfo_get')")
	public R<TAutoPaymentInfo> getById(@PathVariable("id") String id) {
		return R.ok(tAutoPaymentInfoService.getById(id));
	}

	/**
	 * 新增自动化实缴记录表
	 *
	 * @param tAutoPaymentInfo 自动化实缴记录表
	 * @return R
	 */
	@Operation(summary = "新增自动化实缴记录表", description = "新增自动化实缴记录表：hasPermission('social_tautopaymentinfo_add')")
	@SysLog("新增自动化实缴记录表")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('social_tautopaymentinfo_add')")
	public R<Boolean> save(@RequestBody TAutoPaymentInfo tAutoPaymentInfo) {
		return R.ok(tAutoPaymentInfoService.save(tAutoPaymentInfo));
	}

	/**
	 * 修改自动化实缴记录表
	 *
	 * @param tAutoPaymentInfo 自动化实缴记录表
	 * @return R
	 */
	@Operation(summary = "修改自动化实缴记录表", description = "修改自动化实缴记录表：hasPermission('social_tautopaymentinfo_edit')")
	@SysLog("修改自动化实缴记录表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('social_tautopaymentinfo_edit')")
	public R<Boolean> updateById(@RequestBody TAutoPaymentInfo tAutoPaymentInfo) {
		return R.ok(tAutoPaymentInfoService.updateById(tAutoPaymentInfo));
	}

	/**
	 * 通过id删除自动化实缴记录表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除自动化实缴记录表", description = "通过id删除自动化实缴记录表：hasPermission('social_tautopaymentinfo_del')")
	@SysLog("通过id删除自动化实缴记录表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tautopaymentinfo_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tAutoPaymentInfoService.removeById(id));
	}

	/**
	 * 自动化实缴记录表 批量导入
	 *
	 * @author fxj
	 * @date 2024-05-24 10:56:42
	 **/
	@SneakyThrows
	@Operation(description = "批量新增自动化实缴记录表 hasPermission('social_tautopaymentinfo-batch-import')")
	@SysLog("批量新增自动化实缴记录表")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('social_tautopaymentinfo-batch-import')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file) {
		return tAutoPaymentInfoService.importDiy(file.getInputStream());
	}

	/**
	 * 自动化实缴记录表 批量导出
	 *
	 * @author fxj
	 * @date 2024-05-24 10:56:42
	 **/
	@Operation(description = "导出自动化实缴记录表 hasPermission('social_tautopaymentinfo-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('social_tautopaymentinfo-export')")
	public void export(HttpServletResponse response, @RequestBody TAutoPaymentInfoSearchVo searchVo) {
		tAutoPaymentInfoService.listExport(response, searchVo);
	}
}
