/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.salary.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TStatisticsLabor;
import com.yifu.cloud.plus.v1.yifu.salary.service.TStatisticsLaborService;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TStatisticsLaborSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;


/**
 * 本期劳务费申报表
 *
 * @author huyc
 * @date 2022-08-05 11:40:14
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tstatisticslabor")
@Tag(name = "本期劳务费申报表管理")
public class TStatisticsLaborController {

	private final TStatisticsLaborService tStatisticsLaborService;


	/**
	 * 简单分页查询
	 *
	 * @param page             分页对象
	 * @param tStatisticsLabor 本期劳务费申报表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TStatisticsLabor>> getTStatisticsLaborPage(Page<TStatisticsLabor> page, TStatisticsLaborSearchVo tStatisticsLabor) {
		return new R<>(tStatisticsLaborService.getTStatisticsLaborPage(page, tStatisticsLabor));
	}

	/**
	 * 通过id查询本期劳务费申报表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('salary_tstatisticslabor_get')")
	@GetMapping("/{id}")
	public R<TStatisticsLabor> getById(@PathVariable("id") String id) {
		return R.ok(tStatisticsLaborService.getById(id));
	}

	/**
	 * 通过id删除本期劳务费申报表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除本期劳务费申报表", description = "通过id删除本期劳务费申报表：hasPermission('salary_tstatisticslabor_del')")
	@SysLog("通过id删除本期劳务费申报表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('salary_tstatisticslabor_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tStatisticsLaborService.removeById(id));
	}

	/**
	 * 本期劳务费申报表 批量导出
	 *
	 * @author huyc
	 * @date 2022-08-05 11:40:14
	 **/
	@Operation(description = "导出本期劳务费申报表 hasPermission('salary_tstatisticslabor-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('salary_tstatisticslabor-export')")
	public void export(HttpServletResponse response, @RequestBody TStatisticsLaborSearchVo searchVo) {
		tStatisticsLaborService.listExport(response, searchVo);
	}

	/**
	 * 本期劳务费申报表 批量删除
	 *
	 * @author huyc
	 * @date 2022-08-14 21:31:15
	 **/
	@Operation(description = "批量删除本期劳务费申报表 hasPermission('salary_tstatisticslabor_del')")
	@SysLog("批量删除本期稿酬申报表")
	@PostMapping("/batchDelete")
	@PreAuthorize("@pms.hasPermission('salary_tstatisticslabor_del')")
	public R batchDelete(@RequestParam String declareMonth){
		return tStatisticsLaborService.batchDelete(declareMonth);
	}

	/**
	 * @param
	 * @Description: 生成本期劳务费
	 * @Author: huyc
	 * @Date: 2022-08-17 18:17:48
	 * @return: com.yifu.cloud.v1.common.core.util.R
	 **/
	@Inner
	@Operation(description = "生成本期劳务费")
	@PostMapping("/inner/doStatisticsLabor")
	public R doStatisticsLabor() {
		return tStatisticsLaborService.doStatisticsLabor();
	}

	/**
	 * @param
	 * @Description: 手动刷新
	 * @Author: huyc
	 * @Date: 2022-08-31 18:17:48
	 * @return: com.yifu.cloud.v1.common.core.util.R
	 **/
	@Operation(description = "手动刷新")
	@PostMapping("/refreshStatisticsLabor")
	public R refreshStatisticsLabor(@RequestParam String declareMonth) {
		return tStatisticsLaborService.refreshStatisticsLabor(declareMonth);
	}
}
