/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TElecEmployeeInfo;
import com.yifu.cloud.plus.v1.yifu.archives.service.TElecEmployeeInfoService;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.Valid;


/**
 * 电子档案信息表
 *
 * @author fxj
 * @date 2022-06-18 09:39:46
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/telecemployeeinfo" )
@Tag(name = "电子档案信息表管理")
public class TElecEmployeeInfoController {

    private final TElecEmployeeInfoService tElecEmployeeInfoService;

    /**
     * 分页查询
     * @param page 分页对象
     * @param tElecEmployeeInfo 电子档案信息表
     * @return
     */
    @Operation(summary = "分页查询", description = "分页查询")
    @GetMapping("/page" )
    public R getTElecEmployeeInfoPage(Page page, TElecEmployeeInfo tElecEmployeeInfo) {
        return R.ok(tElecEmployeeInfoService.getTElecEmployeeInfoPage(page,tElecEmployeeInfo));
    }


    /**
     * 通过id查询电子档案信息表
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id查询", description = "通过id查询")
    @GetMapping("/{id}" )
    public R getById(@PathVariable("id" ) Long id) {
        return R.ok(tElecEmployeeInfoService.getById(id));
    }

    /**
     * 新增电子档案信息表
     * @param tElecEmployeeInfo 电子档案信息表
     * @return R
     */
    @Operation(summary = "新增电子档案信息表", description = "新增电子档案信息表")
    @SysLog("新增电子档案信息表" )
    @PostMapping
    @PreAuthorize("@pms.hasPermission('archives_telecemployeeinfo_add')" )
    public R save(@Valid @RequestBody TElecEmployeeInfo tElecEmployeeInfo) {
        return tElecEmployeeInfoService.saveElecInfo(tElecEmployeeInfo);
    }

    /**
     * 修改电子档案信息表
     * @param tElecEmployeeInfo 电子档案信息表
     * @return R
     */
    @Operation(summary = "修改电子档案信息表", description = "修改电子档案信息表")
    @SysLog("修改电子档案信息表" )
    @PutMapping
    @PreAuthorize("@pms.hasPermission('archives_telecemployeeinfo_edit')" )
    public R updateById(@RequestBody TElecEmployeeInfo tElecEmployeeInfo) {
        return R.ok(tElecEmployeeInfoService.updateById(tElecEmployeeInfo));
    }

    /**
     * 通过id删除电子档案信息表
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id删除电子档案信息表", description = "通过id删除电子档案信息表")
    @SysLog("通过id删除电子档案信息表" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('archives_telecemployeeinfo_del')" )
    public R removeById(@PathVariable Long id) {
        return R.ok(tElecEmployeeInfoService.removeById(id));
    }

	/**
	 * 通过身份证号查询人员档案
	 * @param idCard
	 * @return R
	 */
	@Operation(summary = "通过身份证号查询人员档案", description = "通过身份证号查询人员档案")
	@SysLog("通过身份证号查询人员档案" )
	@PostMapping("/getEmpInfo" )
	public R getEmpInfoByCard(@RequestParam String idCard) {
		return tElecEmployeeInfoService.getEmpInfoByCard(idCard);
	}

	/**
	 * 批量导入电子档案
	 * @param file
	 * @return R
	 */
	@PostMapping("/importZip")
	@Operation(summary = "批量导入电子档案", description = "批量导入电子档案")
	@SysLog("批量导入电子档案" )
	public R importZip(@RequestBody MultipartFile file) {
		return this.tElecEmployeeInfoService.importZip(file);
	}
}
