/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TPreEmpMain;
import com.yifu.cloud.plus.v1.yifu.archives.service.TPreEmpMainService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.*;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.MenuUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.apache.ibatis.annotations.Param;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 预入职-主表
 *
 * @author hgw
 * @date 2024-06-17 11:31:08
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tpreempmain")
@Tag(name = "预入职-主表管理")
public class TPreEmpMainController {

	private final TPreEmpMainService tPreEmpMainService;

	private final MenuUtil menuUtil;

	/**
	 * 简单分页查询
	 *
	 * @param page        分页对象
	 * @param tPreEmpMain 预入职-主表
	 * @return
	 */
	@Operation(summary = "简单分页查询", description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TPreEmpMain>> getTPreEmpMainPage(Page<TPreEmpMain> page, TPreEmpMainSearchVo tPreEmpMain) {
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, tPreEmpMain);
		return new R<>(tPreEmpMainService.getTPreEmpMainPage(page, tPreEmpMain));
	}

	/**
	 * 不分页查询
	 *
	 * @param tPreEmpMain 预入职-主表
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	//@PreAuthorize("@pms.hasPermission('archives_tpreempmain_get')" )
	public R<List<TPreEmpMain>> getTPreEmpMainNoPage(@RequestBody TPreEmpMainSearchVo tPreEmpMain) {
		return R.ok(tPreEmpMainService.noPageDiy(tPreEmpMain));
	}

	/**
	 * @Description: 根据身份证与项目id获取信息
	 * @Author: hgw
	 * @Date: 2024/6/20 17:23
	 * @return:
	 **/
	@Operation(summary = "根据身份证与项目id获取信息", description = "根据身份证与项目id获取信息")
	@GetMapping("/getByCardAndDeptId")
	public R<PreEmpMainDetailVo> getByCardAndDeptId(@Param("empIdCard") String empIdCard, @Param("deptId") String deptId) {
		return tPreEmpMainService.getByCardAndDeptId(empIdCard, deptId);
	}

	/**
	 * 通过id查询预入职-主表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('archives_tpreempmain_get')")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('archives_tpreempmain_get')")
	public R<PreEmpMainDetailVo> getById(@PathVariable("id") String id) {
		return tPreEmpMainService.getInfoById(id);
	}

	/**
	 * 预入职单个完善信息
	 * hgw
	 * 2024-6-21 11:49:35
	 */
	@Operation(summary = "预入职单个完善信息", description = "预入职单个完善信息")
	@SysLog("预入职单个完善信息")
	@PostMapping("/doImproveInformation")
	public R<String> doImproveInformation(@RequestBody PreEmpMainDetailVo vo) {
		return tPreEmpMainService.doImproveInformation(vo);
	}

	/**
	 * 预入职批量完善信息
	 *
	 * @author hgw
	 * @date 2024-6-21 15:25:20
	 **/
	@SneakyThrows
	@Operation(summary = "预入职批量完善信息", description = "预入职批量完善信息hasPermission('archives_tpreempmain-batch-import')")
	@SysLog("预入职批量完善信息")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('archives_tpreempmain-batch-import')")
	public R<List<ErrorMessage<String>>> importListAdd(@RequestBody MultipartFile file) {
		return tPreEmpMainService.importDiy(file.getInputStream());
	}

	/**
	 * 预入职单个审核
	 * hgw
	 * 2024-6-24 11:52:14
	 */
	@Operation(summary = "预入职单个审核", description = "预入职单个审核status:3审核不通过；4审核通过")
	@SysLog("预入职单个审核")
	@PostMapping("/doAudit")
	public R<String> doAudit(@RequestParam String id, @RequestParam String status, @RequestParam(required = false) String auditRemark) {
		return tPreEmpMainService.doAudit(id, status, auditRemark);
	}

	/**
	 * 预入职批量审核
	 *
	 * @author hgw
	 * @date 2024-6-24 12:00:53
	 **/
	@SneakyThrows
	@Operation(summary = "预入职批量审核", description = "预入职批量审核hasPermission('archives_tpreempmain-batch-audit')；status:3审核不通过；4审核通过")
	@SysLog("预入职批量审核")
	@PostMapping("/doBatchAudit")
	@PreAuthorize("@pms.hasPermission('archives_tpreempmain-batch-audit')")
	public R<List<ErrorVO>> doBatchAudit(@RequestBody List<String> idList, @RequestParam String status, @RequestParam(required = false) String auditRemark) {
		return tPreEmpMainService.doBatchAudit(idList, status, auditRemark);
	}

	/**
	 * 草稿预入职
	 * hgw
	 * 2024-6-17 17:39:15
	 */
	@Operation(summary = "草稿预入职", description = "草稿预入职：hasPermission('archives_tpreempmain_add')")
	@SysLog("草稿预入职")
	@PostMapping("/saveDraftPreEmpMain")
	//@PreAuthorize("@pms.hasPermission('archives_tpreempmain_add')")
	public R<String> saveDraftPreEmpMain(@RequestBody PreEmpMainDetailVo vo) {
		return tPreEmpMainService.saveDraftPreEmpMain(vo);
	}

	/**
	 * 保存预入职
	 * hgw
	 * 2024-6-17 17:39:15
	 */
	@Operation(summary = "保存预入职", description = "保存预入职：hasPermission('archives_tpreempmain_add')")
	@SysLog("保存预入职")
	@PostMapping("/savePreEmpMain")
	//@PreAuthorize("@pms.hasPermission('archives_tpreempmain_add')")
	public R<String> savePreEmpMain(@RequestBody PreEmpMainDetailVo vo) {
		return tPreEmpMainService.savePreEmpMain(vo);
	}

	/**
	 * 修改预入职-主表
	 *
	 * @param tPreEmpMain 预入职-主表
	 * @return R
	 */
	@Operation(summary = "修改预入职-主表", description = "修改预入职-主表：hasPermission('archives_tpreempmain_edit')")
	@SysLog("修改预入职-主表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('archives_tpreempmain_edit')")
	public R<Boolean> updateById(@RequestBody TPreEmpMain tPreEmpMain) {
		return R.ok(tPreEmpMainService.updateById(tPreEmpMain));
	}

	/**
	 * 通过id删除预入职-主表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除预入职-主表", description = "通过id删除预入职-主表：hasPermission('archives_tpreempmain_del')")
	@SysLog("通过id删除预入职-主表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('archives_tpreempmain_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tPreEmpMainService.removeById(id));
	}

	/**
	 * 预入职-主表 批量导出
	 *
	 * @author hgw
	 * @date 2024-6-24 16:01:50
	 **/
	@Operation(summary = "导出预入职-主表", description = "导出预入职-主表 hasPermission('archives_tpreempmain-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('archives_tpreempmain-export')")
	public void export(HttpServletResponse response, @RequestBody TPreEmpMainSearchVo searchVo) {
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, searchVo);
		tPreEmpMainService.listExport(response, searchVo);
	}

	/**
	 * 预入职-全量 批量导出
	 *
	 * @author hgw
	 * @date 2024-6-24 16:01:47
	 **/
	@Operation(summary = "导出预入职-全量", description = "导出预入职-全量 hasPermission('archives_tpreempmain-export-all')")
	@PostMapping("/exportAll")
	@SysLog("导出预入职-全量")
	//@PreAuthorize("@pms.hasPermission('archives_tpreempmain-export-all')")
	public void exportAll(HttpServletResponse response, @RequestBody TPreEmpMainSearchVo searchVo) {
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, searchVo);
		tPreEmpMainService.exportAll(response, searchVo);
	}

	/**
	 * @param file
	 * @Description: 预入职批量导入
	 * @Author: hgw
	 * @Date: 2024/10/12 9:24
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.salary.vo.SalaryUploadParamVo>
	 **/
	@SysLog("预入职批量导入")
	@Operation(summary = "预入职批量导入", description = "预入职批量导入 hasPermission('pre_emp_batch_upload')")
	@PostMapping("/uploadPreEmp")
	// @PreAuthorize("@pms.hasPermission('pre_emp_batch_upload')")
	public R<List<PreUploadEmpImportVo>> uploadPreEmp(MultipartFile file, PreUploadImportParamVo vo) throws Exception {
		return tPreEmpMainService.uploadPreEmp(file, vo);
	}

	/**
	 * 根据身份证与姓名查询用户手机号和来源
	 *
	 * @param empIdCard 身份证
	 * @param empName 姓名
	 * @return R<EmpMainToCVo>
	 *
	 * @author chenyuxi
	 * @date 2024-10-24 16:01:47
	 * @since mvp1.7.0
	 *
	 **/
	@Operation(summary = "根据身份证与姓名查询用户手机号和来源", description = "根据身份证与姓名查询用户手机号和来源")
	@GetMapping("/getPhoneByCardAndName")
	public R<EmpMainToCVo> getPhoneByCardAndName(@Param("empIdCard") String empIdCard, @Param("empName") String empName) {
		return tPreEmpMainService.getPhoneByCardAndName(empIdCard, empName);
	}

	/**
	 * 发送手机验证码
	 *
	 * @param empPhone 手机号
	 * @return R<String>
	 *
	 * @author chenyuxi
	 * @date 2024-10-24 16:01:47
	 * @since mvp1.7.0
	 */
	@PostMapping("/sendCode")
	public R<String> sendCode(@Param("empPhone") String empPhone) {
		if (Common.isEmpty(empPhone)) {
			return R.failed("手机号不能为空");
		}
		return tPreEmpMainService.sendCode(empPhone);
	}

	/**
	 * 校验手机验证码
	 *
	 * @param empPhone 手机号
	 * @param verifyCode 验证码
	 * @return R<String>
	 *
	 * @author chenyuxi
	 * @date 2024-10-24 16:01:47
	 * @since mvp1.7.0
	 */
	@PostMapping("/checkCode")
	public R<String> checkCode(@Param("empPhone") String empPhone,@Param("verifyCode") String verifyCode) {
		if (Common.isEmpty(empPhone)) {
			return R.failed("手机号不能为空");
		}
		if (Common.isEmpty(verifyCode)) {
			return R.failed("验证码不能为空");
		}
		return tPreEmpMainService.checkCode(empPhone, verifyCode);
	}
	/**
	 * @Author fxj
	 * @Description 预入职C端签名图盘打包下载
	 * @Date 14:07 2025/1/10
	 * @Param
	 * @return
	**/
	@Operation(summary = "预入职C端签名图盘打包下载ZIP包", description = "预入职C端签名图盘打包下载ZIP包")
	@SysLog("预入职C端签名图盘打包下载ZIP包")
	@GetMapping("/preEmpExport")
	public void preEmpExport(@RequestParam(value = "departNo",required = true) String departNo,
							 @RequestParam(value ="departName",required = true) String departName,
							   @RequestParam(value ="startDate",required = true) String startDate,
							   @RequestParam(value ="endDate",required = true) String endDate,
							   HttpServletResponse response) throws Exception{
		tPreEmpMainService.exportEmpsFile(departName,departNo,startDate,endDate, response);
	}

	/**
	 * @Author fxj
	 * @Description 预入职C端签名图盘打包下载 count
	 * @Date 14:07 2025/1/10
	 * @Param
	 * @return
	 **/
	@Operation(summary = "预入职C端签名图盘打包下载ZIP包count", description = "预入职C端签名图盘打包下载ZIP包count")
	@SysLog("预入职C端签名图盘打包下载ZIP包count")
	@GetMapping("/preEmpCount")
	public R preEmpCount(@RequestParam(value ="departNo",required = true) String departNo,
							   @RequestParam(value ="startDate",required = true) String startDate,
							   @RequestParam(value ="endDate",required = true) String endDate,
							   HttpServletResponse response) throws Exception{
		return tPreEmpMainService.preEmpCount(departNo,startDate,endDate);
	}
}
