/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.insurances.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.insurances.entity.TInsuranceAreaRes;
import com.yifu.cloud.plus.v1.yifu.insurances.service.TInsuranceAreaResService;
import com.yifu.cloud.plus.v1.yifu.insurances.vo.TInsuranceAreaExportVo;
import com.yifu.cloud.plus.v1.yifu.insurances.vo.TInsuranceAreaResSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.apache.ibatis.annotations.Param;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 商险地市权限配置表
 *
 * @author fxj
 * @date 2022-11-01 10:32:05
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tinsuranceareares")
@Tag(name = "商险地市权限配置表管理")
public class TInsuranceAreaResController {

	private final TInsuranceAreaResService tInsuranceAreaResService;


	/**
	 * 简单分页查询
	 *
	 * @param page              分页对象
	 * @param tInsuranceAreaRes 商险地市权限配置表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TInsuranceAreaRes>> getTInsuranceAreaResPage(Page<TInsuranceAreaRes> page, TInsuranceAreaResSearchVo tInsuranceAreaRes) {
		return new R<>(tInsuranceAreaResService.getTInsuranceAreaResPage(page, tInsuranceAreaRes));
	}

	/**
	 * 不分页查询
	 *
	 * @param searchVo 商险地市权限配置表
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	//@PreAuthorize("@pms.hasPermission('tinsuranceareares_get')" )
	public R<List<TInsuranceAreaExportVo>> getTInsuranceAreaResNoPage(@RequestBody TInsuranceAreaResSearchVo searchVo) {
		return R.ok(tInsuranceAreaResService.noPageDiy(searchVo,null, Common.getList(searchVo.getIds()),null));
	}

	/**
	 * 通过id查询商险地市权限配置表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('tinsuranceareares_get')")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('tinsuranceareares_get')")
	public R<TInsuranceAreaRes> getById(@PathVariable("id") String id) {
		return R.ok(tInsuranceAreaResService.getById(id));
	}

	/**
	 * 新增商险地市权限配置表
	 *
	 * @param areaRes 商险地市权限配置表
	 * @return R
	 */
	@Operation(summary = "新增商险地市权限配置表", description = "新增商险地市权限配置表：hasPermission('tinsuranceareares_add')")
	@SysLog("新增商险地市权限配置表")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('tinsuranceareares_add')")
	public R<Boolean> save(@RequestBody TInsuranceAreaRes areaRes) {
		TInsuranceAreaRes res = null;
		if (Common.isEmpty(areaRes.getCity())){
			res = tInsuranceAreaResService.getOne(Wrappers.<TInsuranceAreaRes>query().lambda()
					.eq(TInsuranceAreaRes::getProvince,areaRes.getProvince())
					.isNull(TInsuranceAreaRes::getCity)
					.eq(TInsuranceAreaRes::getUserId,areaRes.getUserId())
					.last(CommonConstants.LAST_ONE_SQL));
		}else {
			res = tInsuranceAreaResService.getOne(Wrappers.<TInsuranceAreaRes>query().lambda()
					.eq(TInsuranceAreaRes::getProvince,areaRes.getProvince())
					.eq(TInsuranceAreaRes::getCity,areaRes.getCity())
					.eq(TInsuranceAreaRes::getUserId,areaRes.getUserId())
					.last(CommonConstants.LAST_ONE_SQL));
		}
		if (Common.isNotNull(res)){
			return R.failed("已存在对应办理人及地市配置！");
		}
		// 清理缓存
		tInsuranceAreaResService.removeAuthCache();
		return R.ok(tInsuranceAreaResService.save(areaRes));
	}

	/**
	 * 修改商险地市权限配置表
	 *
	 * @param tInsuranceAreaRes 商险地市权限配置表
	 * @return R
	 */
	@Operation(summary = "修改商险地市权限配置表", description = "修改商险地市权限配置表：hasPermission('tinsuranceareares_edit')")
	@SysLog("修改商险地市权限配置表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('tinsuranceareares_edit')")
	public R<Boolean> updateById(@RequestBody TInsuranceAreaRes tInsuranceAreaRes) {
		return R.ok(tInsuranceAreaResService.updateById(tInsuranceAreaRes));
	}

	/**
	 * 通过id删除商险地市权限配置表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除商险地市权限配置表", description = "通过id删除商险地市权限配置表：hasPermission('tinsuranceareares_del')")
	@SysLog("通过id删除商险地市权限配置表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('tinsuranceareares_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		// 清理缓存
		tInsuranceAreaResService.removeAuthCache();
		return R.ok(tInsuranceAreaResService.removeById(id));
	}

	/**
	 * 通过id启用或禁用商险地市权限配置表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id启用或禁用商险地市权限配置表", description = "通过id启用或禁用(0 启用，1禁用)商险地市权限配置表：hasPermission('tinsuranceareares_edit')")
	@SysLog("通过id启用或禁用商险地市权限配置表")
	@PutMapping("/updateStatus")
	@PreAuthorize("@pms.hasPermission('tinsuranceareares_edit')")
	public R<Boolean> removeById(@RequestParam(name = "id",required = true) String id,
								 @RequestParam(name = "status",required = true) String status) {
		TInsuranceAreaRes areaRes = tInsuranceAreaResService.getById(id);
		if (Common.isEmpty(areaRes)
				|| !(CommonConstants.ONE_STRING).equals(status)
				&& !CommonConstants.ZERO_STRING.equals(status)){
			return R.failed(CommonConstants.PARAM_INFO_ERROR);
		}
		areaRes.setStatus(status);
		// 清理缓存
		tInsuranceAreaResService.removeAuthCache();
		return R.ok(tInsuranceAreaResService.updateById(areaRes));
	}

	/**
	 * 商险地市权限配置表 批量导入
	 *
	 * @author fxj
	 * @date 2022-11-01 10:32:05
	 **/
	@SneakyThrows
	@Operation(description = "批量新增商险地市权限配置表 hasPermission('tinsuranceareares-batch-import')")
	@SysLog("批量新增商险地市权限配置表")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('tinsuranceareares-batch-import')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file) {
		return tInsuranceAreaResService.importDiy(file.getInputStream());
	}

	/**
	 * 商险地市权限配置表 批量导出
	 *
	 * @author fxj
	 * @date 2022-11-01 10:32:05
	 **/
	@Operation(description = "导出商险地市权限配置表 hasPermission('tinsuranceareares-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('tinsuranceareares-export')")
	public void export(HttpServletResponse response, @RequestBody TInsuranceAreaResSearchVo searchVo) {
		tInsuranceAreaResService.listExport(response, searchVo);
	}
}
