/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.salary.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TStatisticsCurrentReportMarket;
import com.yifu.cloud.plus.v1.yifu.salary.service.TStatisticsCurrentReportMarketService;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TStatisticsCurrentReportMarketSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;


/**
 * 统计-本期申报-回盘数据
 *
 * @author hgw
 * @date 2022-08-05 11:40:14
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tstatisticscurrentreportmarket")
@Tag(name = "统计-本期申报-回盘数据管理")
public class TStatisticsCurrentReportMarketController {

	private final TStatisticsCurrentReportMarketService tStatisticsCurrentReportMarketService;


	/**
	 * 简单分页查询
	 *
	 * @param page                           分页对象
	 * @param tStatisticsCurrentReportMarket 统计-本期申报-回盘数据
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TStatisticsCurrentReportMarket>> getTStatisticsCurrentReportMarketPage(Page<TStatisticsCurrentReportMarket> page, TStatisticsCurrentReportMarketSearchVo tStatisticsCurrentReportMarket) {
		return new R<>(tStatisticsCurrentReportMarketService.getTStatisticsCurrentReportMarketPage(page, tStatisticsCurrentReportMarket));
	}

	/**
	 * 通过id查询统计-本期申报-回盘数据
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('salary_tstatisticscurrentreportmarket_get')")
	@GetMapping("/{id}")
	public R<TStatisticsCurrentReportMarket> getById(@PathVariable("id") String id) {
		return R.ok(tStatisticsCurrentReportMarketService.getById(id));
	}

	/**
	 * 通过id删除统计-本期申报-回盘数据
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除统计-本期申报-回盘数据", description = "通过id删除统计-本期申报-回盘数据：hasPermission('salary_report_market_del')")
	@SysLog("通过id删除统计-本期申报-回盘数据")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('salary_report_market_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tStatisticsCurrentReportMarketService.removeById(id));
	}

	/**
	 * @param yearMonth 年月
	 * @return R
	 */
	@Operation(summary = "删除-按申报月份", description = "通过id删除统计-本期申报-回盘数据：hasPermission('salary_report_market_doDeleteByDeclareMonth')")
	@SysLog("删除-按申报月份")
	@PostMapping("/doDeleteByDeclareMonth")
	@PreAuthorize("@pms.hasPermission('salary_report_market_doDeleteByDeclareMonth')")
	public R<String> doDeleteByDeclareMonth(@RequestParam String yearMonth) {
		int num = tStatisticsCurrentReportMarketService.deleteByYearMonth(yearMonth);
		if (num == 0) {
			return R.failed("当前条件查询无数据，未删除");
		} else {
			return R.ok(null,"已删除" + num + "条数据");
		}
	}

	/**
	 * 统计-本期申报-回盘数据 批量导入
	 *
	 * @author hgw
	 * @date 2022-08-05 11:40:14
	 **/
	@SneakyThrows
	@Operation(description = "批量导入-回盘数据 hasPermission('salary_report_market-batch-import')")
	@SysLog("批量导入-回盘数据")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('salary_report_market-batch-import')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file) {
		return tStatisticsCurrentReportMarketService.importDiy(file.getInputStream());
	}

}
