/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TCustomerInfo;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TDepartSettlementInfo;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TSettleDomain;
import com.yifu.cloud.plus.v1.yifu.archives.service.TDepartSettlementInfoService;
import com.yifu.cloud.plus.v1.yifu.archives.service.TSettleDomainService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.IdNameNoVo;
import com.yifu.cloud.plus.v1.yifu.archives.vo.SetInfoVo;
import com.yifu.cloud.plus.v1.yifu.archives.vo.TSettleDomainListVo;
import com.yifu.cloud.plus.v1.yifu.archives.vo.TSettleDomainSelectVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.BeanUtils;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;


/**
 * 项目表
 *
 * @author hgw
 * @date 2022-06-20 10:02:28
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tsettledomain")
@Tag(name = "项目表管理")
public class TSettleDomainController {

	private final TSettleDomainService tSettleDomainService;
	private final TDepartSettlementInfoService tDepartSettlementInfoService;

	/**
	 * 分页查询
	 *
	 * @param page          分页对象
	 * @param tSettleDomain 项目表
	 * @return
	 */
	// 	2022-9-27 16:36:56hgw备注：前端用这个接口模糊搜索项目信息：/yifu-archives/method/tempchangeinfo/getAllDeptPage
	@Operation(summary = "分页查询", description = "分页查询")
	@GetMapping("/page")
	public R<IPage<TSettleDomain>> getTSettleDomainPage(Page<TSettleDomain> page, TSettleDomain tSettleDomain) {
		return R.ok(tSettleDomainService.page(page, Wrappers.query(tSettleDomain)));
	}


	/**
	 * 通过id查询项目表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询")
	@GetMapping("/{id}")
	public R<TSettleDomain> getById(@PathVariable("id") String id) {
		return R.ok(tSettleDomainService.getById(id));
	}

	/**
	 * 新增项目表
	 *
	 * @param tSettleDomain 项目表
	 * @return R
	 */
	@Operation(summary = "新增项目表", description = "新增项目表")
	@SysLog("新增项目表")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('tsettledomain_add')")
	public R<Boolean> save(@RequestBody TSettleDomain tSettleDomain) {
		return R.ok(tSettleDomainService.save(tSettleDomain));
	}

	/**
	 * 修改项目表
	 *
	 * @param tSettleDomain 项目表
	 * @return R
	 */
	@Operation(summary = "修改项目表", description = "修改项目表")
	@SysLog("修改项目表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('tsettledomain_edit')")
	public R<Boolean> updateById(@RequestBody TSettleDomain tSettleDomain) {
		return R.ok(tSettleDomainService.updateById(tSettleDomain));
	}

	/**
	 * 通过id删除项目表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除项目表", description = "通过id删除项目表")
	@SysLog("通过id删除项目表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('tsettledomain_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tSettleDomainService.removeById(id));
	}

	/**
	 * 获取登录用户拥有的项目信息及单位部分信息
	 *
	 * @param
	 * @return
	 * @Author fxj
	 * @Date 2019-08-28
	 **/
	@Operation(description = "获取登录用户拥有的项目数据）")
	@Inner
	@PostMapping("/getSettleDomainIdsByUserId")
	public TSettleDomainListVo getSettleDomainIdsByUserId(@RequestBody String userId) {
		TSettleDomainListVo vo = new TSettleDomainListVo();
		vo.setDeptIds(tSettleDomainService.getSettleDomainIdsByUserId(userId));
		return vo;
	}

	/**
	 * @param
	 * @Author: wangan
	 * @Date: 2019/10/18
	 * @Description:通过id获取项目及单位部分信息
	 **/
	@Inner
	@PostMapping("/getSettleDomainSelectVoById")
	public TSettleDomainSelectVo getInnerSettleDomainSelectVoByUserId(@RequestBody TSettleDomainSelectVo settleDomainSelectVo) {
		return tSettleDomainService.selectSettleDomainSelectVosById(settleDomainSelectVo);
	}

	/**
	 * @param
	 * @Author: wangan
	 * @Date: 2019/10/18
	 * @Description:获取所有客户单位的项目信息
	 **/
	@Inner
	@PostMapping("/selectAllSettleDomainSelectVos")
	public TSettleDomainListVo selectAllSettleDomainSelectVos() {
		List<TSettleDomainSelectVo> list = tSettleDomainService.selectAllSettleDomainSelectVos();
		TSettleDomainListVo listVo = new TSettleDomainListVo();
		listVo.setListSelectVO(list);
		return listVo;
	}

	/**
	 * @param
	 * @Author: wangan
	 * @Date: 2019/10/18
	 * @Description:获取所有客户单位的项目信息
	 **/
	@Inner
	@PostMapping("/selectSettleDomainSelectVoById")
	public TSettleDomainListVo selectSettleDomainSelectVoById(@RequestBody TSettleDomainSelectVo tSettleDomainSelectVo) {
		List<TSettleDomainSelectVo> list = new ArrayList<>();
		TSettleDomain tSettleDomain = tSettleDomainService.getById(tSettleDomainSelectVo.getId());
		BeanUtils.copyProperties(tSettleDomain,tSettleDomainSelectVo);
		list.add(tSettleDomainSelectVo);
		TSettleDomainListVo listVo = new TSettleDomainListVo();
		listVo.setListSelectVO(list);
		return listVo;
	}

	/**
	 * @param
	 * @Author: wangan
	 * @Date: 2019/10/18
	 * @Description:获取所有客户单位的项目信息
	 **/
	@Inner
	@PostMapping("/selectSettleDomainSelectVoByNo")
	public TSettleDomainListVo selectSettleDomainSelectVoByNo(@RequestBody TSettleDomainSelectVo tSettleDomainSelectVo) {
		List<TSettleDomainSelectVo> list = new ArrayList<>();
		TSettleDomain tSettleDomain = tSettleDomainService.getOne(Wrappers.<TSettleDomain>query().lambda()
				.eq(TSettleDomain::getDepartNo, tSettleDomainSelectVo.getDepartNo())
				.last(CommonConstants.LAST_ONE_SQL));
		BeanUtils.copyProperties(tSettleDomain,tSettleDomainSelectVo);
		list.add(tSettleDomainSelectVo);
		TSettleDomainListVo listVo = new TSettleDomainListVo();
		listVo.setListSelectVO(list);
		return listVo;
	}

	/**
	 * @Author fxj
	 * @Description  通过项目编码获取项目及客户信息
	 * @Date 10:49 2022/7/20
	 * @Param
	 * @return
	**/
	@Operation(description = "通过项目编码获取项目及客户信息")
	@SysLog("通过项目编码获取项目及客户信息")
	@Inner
	@PostMapping("/getSetInfoByCodes")
	public SetInfoVo getSetInfoByCodes(@RequestBody(required = true) List<String> codes) {
		return  tSettleDomainService.getSetInfoByCodes(codes);
	}

	/**
	 * @param id 项目主键
	 * @Description: 通过id获取结算主体及单位部分信息
	 * @Author: hgw
	 * @Date: 2022/8/10 17:01
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.archives.vo.TSettleDomainSelectVo>
	 **/
	@Inner
	@PostMapping("/inner/getSettleDomainVoById")
	public TSettleDomainSelectVo getSettleDomainVoById(@RequestBody String id) {
		return tSettleDomainService.getSettleDomainVoById(id);
	}

	/**
	 * @param deptId 结算主体id
	 * @Description: 根据结算主体查询结算配置
	 * @Author: hgw
	 * @Date: 2022-8-10 18:14:31
	 * @return: com.yifu.cloud.v1.common.core.util.R<com.yifu.cloud.v1.hrms.api.entity.TDepartSettlementInfo>
	 **/
	@Inner
	@PostMapping("/inner/getInnerBySettlementId")
	public TDepartSettlementInfo getInnerBySettlementId(@RequestBody String deptId) {
		TDepartSettlementInfo departSettlementInfo = new TDepartSettlementInfo();
		departSettlementInfo.setSettleId(deptId);
		QueryWrapper<TDepartSettlementInfo> queryWrapperDs = new QueryWrapper<>();
		queryWrapperDs.setEntity(departSettlementInfo);
		List<TDepartSettlementInfo> dsList = tDepartSettlementInfoService.list(queryWrapperDs);
		TDepartSettlementInfo ds = null;
		if (dsList != null && !dsList.isEmpty()) {
			ds = dsList.get(CommonConstants.ZERO_INT);
		}
		return ds;
	}

	/**
	 * 修改项目表--EKP调用接口
	 * @author  hyc
	 * @param list 项目表
	 * @return R
	 */
	@Operation(summary = "修改项目表--EKP调用接口", description = "修改项目表--EKP调用接口")
	@SysLog("修改项目表--EKP调用接口")
	@PostMapping("/updateProjectInfo")
	public R updateProjectInfo(@RequestBody List<TSettleDomain> list) {
		return tSettleDomainService.updateProjectInfo(list);
	}


	/**
	 * 分页查询 客户信息
	 * @return
	 */
	@Operation(summary = "分页查询客户信息", description = "分页查询客户信息")
	@GetMapping("/getUnitPage")
	public R<IPage<IdNameNoVo>> getUnitPage(Page<IdNameNoVo> page, IdNameNoVo idNameNo) {
		return R.ok(tSettleDomainService.getUnitPage(page, idNameNo));
	}

}
