/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.salary.controller;

import com.yifu.cloud.plus.v1.yifu.common.core.util.OSSUtil;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TSalaryAtta;
import com.yifu.cloud.plus.v1.yifu.salary.service.TSalaryAttaService;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.net.URL;
import java.util.List;


/**
 * 薪资结算附件表
 *
 * @author hgw
 * @date 2022-08-05 11:40:14
 */
@Log4j2
@RestController
@RequiredArgsConstructor
@RequestMapping("/tsalaryatta")
@Tag(name = "薪资结算附件表")
public class TSalaryAttaController {

	private final TSalaryAttaService tSalaryAttaService;

	private OSSUtil ossUtil;

	/**
	 * @param recordId 工程薪资暂停发发放记录的id
	 * @Description:
	 * @Author: hgw
	 * @Date: 2021/10/27 17:31
	 * @return: com.yifu.cloud.v1.common.core.util.R<java.util.List < com.yifu.cloud.v1.hrms.api.entity.TSalaryAtta>>
	 **/
	@Schema(description = "获取list（recordId 工程薪资暂停发发放记录的id）")
	@GetMapping("/getTSalaryAttaList")
	public R<List<TSalaryAtta>> getTSalaryAttaList(@RequestParam String recordId) {
		return new R<>(tSalaryAttaService.getAttaListByRecordId(recordId));
	}

	/**
	 * @param linkId id
	 * @Description: 获取附件
	 * @Author: hgw
	 * @Date: 2021-12-8 15:36:58
	 * @return: com.yifu.cloud.v1.common.core.util.R<java.util.List < com.yifu.cloud.v1.hrms.api.entity.TSalaryAtta>>
	 **/
	@Schema(description = "获取list(linkId, linkType链接类型：0工资；1工程工资；2暂停发；3自定义项暂停发；4打印记录；5核准表打印记录6：非扣税 7发放失败 8自有员工附件 9财务回执附件 10换人换卡附件；11薪资原表；12劳务费或稿酬已有发薪记录；13薪资特殊值（3500、5000）；14本月重复金额")
	@GetMapping("/getTSalaryAttaListByIdType")
	public R<List<TSalaryAtta>> getTSalaryAttaListByIdType(@RequestParam String linkId, @RequestParam Integer linkType) {
		return new R<>(tSalaryAttaService.getAttaListHaveSrc(linkId, linkType));
	}

	/**
	 * 阿里云处理文件上传
	 *
	 * @param file     文件
	 * @param filePath 如： import   或  import/test/test..../test
	 * @return
	 * @throws IOException
	 */
	@SysLog("薪资附件上传")
	@Schema(description = "OSS文件上传接口。linkType类型：0工资；1工程工资；2暂停发；3自定义项暂停发；4薪资打印记录；5核准表打印记录;6：非扣税项；8自有员工附件；9:财务回执附件；10换人换卡附件；11薪资原表；12劳务费或稿酬已有发薪记录；13薪资特殊值（3500、5000）；14本月重复金额")
	/*@ApiImplicitParams({
			@ApiImplicitParam(name = "file", value = "Form文件上传", required = true, dataType = "__file", paramType = "form"),
			@ApiImplicitParam(name = "filePath", value = "文件上传路径", dataType = "String", paramType = "form"),
			@ApiImplicitParam(name = "linkId", value = "链接ID（工资主表id、工程主表id、自定义项表id等）", required = true, dataType = "String", paramType = "form"),
			@ApiImplicitParam(name = "linkType", value = "类型：0工资；1工程工资；2暂停发；3自定义项暂停发；4薪资打印记录；5核准表打印记录;6：非扣税项；8自有员工附件；9:财务回执附件；10换人换卡附件；11薪资原表", required = true, dataType = "int", paramType = "form"),
			@ApiImplicitParam(name = "engineerType", value = "工程文件类型：0三方协议；1委托付款函；2劳务费;3其他", required = true, dataType = "int", paramType = "form")
	})*/
	@PostMapping(value = "/ossUploadFile")
	public R uploadImg(@RequestBody MultipartFile file, String filePath, String linkId
			, Integer linkType, Integer engineerType, String printRemark, String recordId) throws IOException {
		return tSalaryAttaService.uploadAtta(file, filePath, linkId, linkType, engineerType, printRemark, recordId);
	}

	/**
	 * 获取附件下载地址
	 *
	 * @param id
	 * @return
	 * @Author fxj
	 * @Date 2019-08-16
	 **/
	@Schema(description = "附件预览下载地址")
	@SysLog("附件预览下载地址")
	@GetMapping("/ossFileUrl/{id}")
	public R ossFileUrl(@PathVariable String id) {
		TSalaryAtta salaryAtta = tSalaryAttaService.getById(id);
		if (null == salaryAtta) {
			return R.failed("无对应附件信息");
		}
		URL url = ossUtil.getObjectUrl(null, salaryAtta.getAttaSrc());
		salaryAtta.setAttaUrl(url.toString().replace("http","https"));
		return new R<>(salaryAtta);
	}

	/**
	 * 删除数据库记录和OSS文件
	 *
	 * @param id
	 * @return
	 * @Author fxj
	 * @Date 2019-09-09
	 **/
	@Schema(description = "客户/员工表单附件删除")
	@SysLog("客户/员工表单附件删除")
	@GetMapping("/ossFileDelete/{id}")
	public R ossFileDelete(@PathVariable String id) {
		try {
			TSalaryAtta salaryAtta = tSalaryAttaService.getById(id);
			if (null == salaryAtta) {
				return R.failed("无对应附件信息");
			}
			if (tSalaryAttaService.removeById(id)) {
				ossUtil.deleteObject(null, salaryAtta.getAttaSrc());
				return R.ok("删除成功！");
			}
		} catch (Exception e) {
			log.error("附件删除异常：" + e.getMessage());
			return R.failed("附件删除异常！");
		}
		return R.ok("删除失败！");
	}

}
