/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.salary.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ArrayUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.read.listener.ReadListener;
import com.alibaba.excel.read.metadata.holder.ReadRowHolder;
import com.alibaba.excel.util.ListUtils;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.check.entity.TCheckBankNo;
import com.yifu.cloud.plus.v1.check.entity.TCheckMobile;
import com.yifu.cloud.plus.v1.check.vo.CheckBankNoVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CacheConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.SecurityConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.*;
import com.yifu.cloud.plus.v1.yifu.common.dapr.config.DaprCheckProperties;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.HttpDaprUtil;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TSalaryEmployee;
import com.yifu.cloud.plus.v1.yifu.salary.mapper.TSalaryEmployeeMapper;
import com.yifu.cloud.plus.v1.yifu.salary.service.TSalaryEmployeeService;
import com.yifu.cloud.plus.v1.yifu.salary.util.SalaryConstants;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TSalaryEmployeeExportVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TSalaryEmployeeSearchVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TSalaryEmployeeUpdateVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TSalaryEmployeeVo;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 薪酬人员表
 *
 * @author hgw
 * @date 2022-08-05 11:40:14
 */
@Log4j2
@Service
@EnableConfigurationProperties({DaprCheckProperties.class})
public class TSalaryEmployeeServiceImpl extends ServiceImpl<TSalaryEmployeeMapper, TSalaryEmployee> implements TSalaryEmployeeService {

	@Autowired
	private DaprCheckProperties checkProperties;

	/**
	 * 薪酬人员表简单分页查询
	 *
	 * @param tSalaryEmployee 薪酬人员表
	 * @return
	 */
	@Override
	public IPage<TSalaryEmployee> getTSalaryEmployeePage(Page<TSalaryEmployee> page, TSalaryEmployeeSearchVo tSalaryEmployee) {
		return baseMapper.getTSalaryEmployeePage(page, tSalaryEmployee);
	}

	/**
	 * 薪酬人员表批量导出
	 *
	 * @return
	 */
	@Override
	public void listExport(HttpServletResponse response, TSalaryEmployeeSearchVo searchVo) {
		String fileName = "薪酬人员查询批量导出" + DateUtil.getThisTime() + CommonConstants.XLSX;
		//获取要导出的列表
		List<TSalaryEmployeeExportVo> list = new ArrayList<>();
		long count = noPageCountDiy(searchVo);
		ServletOutputStream out = null;
		try {
			out = response.getOutputStream();
			response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
			response.setCharacterEncoding("utf-8");
			response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName, "UTF-8"));
			// 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
			//EasyExcel.write(out, TEmpBadRecord.class).sheet("不良记录").doWrite(list)
			ExcelWriter excelWriter = EasyExcel.write(out, TSalaryEmployeeExportVo.class).build();
			int index = 0;
			if (count > CommonConstants.ZERO_INT) {
				for (int i = 0; i <= count; ) {
					// 获取实际记录
					searchVo.setLimitStart(i);
					searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
					list = noPageDiy(searchVo);
					if (Common.isNotNull(list)) {
						ExcelUtil<TSalaryEmployeeExportVo> util = new ExcelUtil<>(TSalaryEmployeeExportVo.class);
						for (TSalaryEmployeeExportVo vo : list) {
							util.convertEntity(vo, null, null, null);
						}
					}
					if (Common.isNotNull(list)) {
						WriteSheet writeSheet = EasyExcel.writerSheet("薪酬人员表" + index).build();
						excelWriter.write(list, writeSheet);
						index++;
					}
					i = i + CommonConstants.EXCEL_EXPORT_LIMIT;
					if (Common.isNotNull(list)) {
						list.clear();
					}
				}
			} else {
				WriteSheet writeSheet = EasyExcel.writerSheet("薪酬人员表" + index).build();
				excelWriter.write(list, writeSheet);
			}
			if (Common.isNotNull(list)) {
				list.clear();
			}
			out.flush();
			excelWriter.finish();
		} catch (Exception e) {
			log.error("执行异常", e);
		} finally {
			try {
				if (null != out) {
					out.close();
				}
			} catch (IOException e) {
				log.error("执行异常", e);
			}
		}
	}

	@Override
	public List<TSalaryEmployeeExportVo> noPageDiy(TSalaryEmployeeSearchVo searchVo) {
		return baseMapper.noPageDiy(searchVo,Common.getList(searchVo.getIds()));
	}

	private Long noPageCountDiy(TSalaryEmployeeSearchVo searchVo) {
		LambdaQueryWrapper<TSalaryEmployee> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TSalaryEmployee::getId, idList);
		}
		if (Common.isNotNull(searchVo.getAuthSql())) {
			wrapper.last(searchVo.getAuthSql());
		}
		return baseMapper.selectCount(wrapper);
	}

	private LambdaQueryWrapper buildQueryWrapper(TSalaryEmployeeSearchVo entity) {
		LambdaQueryWrapper<TSalaryEmployee> wrapper = Wrappers.lambdaQuery();
		wrapper.ne(TSalaryEmployee::getId,CommonConstants.ZERO_STRING);
		if (ArrayUtil.isNotEmpty(entity.getCreateTimes())) {
			wrapper.ge(TSalaryEmployee::getCreateTime, entity.getCreateTimes()[0])
					.le(TSalaryEmployee::getCreateTime,
							entity.getCreateTimes()[1]);
		}
		if (Common.isNotNull(entity.getCreateName())) {
			wrapper.eq(TSalaryEmployee::getCreateName, entity.getCreateName());
		}
		if (Common.isNotNull(entity.getEmpName())) {
			wrapper.eq(TSalaryEmployee::getEmpName, entity.getEmpName());
		}
		if (Common.isNotNull(entity.getEmpIdcard())) {
			wrapper.eq(TSalaryEmployee::getEmpIdcard, entity.getEmpIdcard());
		}
		return wrapper;
	}

	@Override
	public R<List<ErrorMessage>> importDiy(InputStream inputStream) {
		List<ErrorMessage> errorMessageList = new ArrayList<>();
		ExcelUtil<TSalaryEmployeeVo> util1 = new ExcelUtil<>(TSalaryEmployeeVo.class);
		// 写法2：
		// 匿名内部类 不用额外写一个DemoDataListener
		// 这里 需要指定读用哪个class去读，然后读取第一个sheet 文件流会自动关闭
		try {
			EasyExcel.read(inputStream, TSalaryEmployeeVo.class, new ReadListener<TSalaryEmployeeVo>() {
				/**
				 * 单次缓存的数据量
				 */
				public static final int BATCH_COUNT = CommonConstants.BATCH_COUNT;
				/**
				 *临时存储
				 */
				private List<TSalaryEmployeeVo> cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);

				@Override
				public void invoke(TSalaryEmployeeVo data, AnalysisContext context) {
					ReadRowHolder readRowHolder = context.readRowHolder();
					Integer rowIndex = readRowHolder.getRowIndex();
					data.setRowIndex(rowIndex + 1);
					ErrorMessage errorMessage = util1.checkEntity(data, data.getRowIndex());
					if (Common.isNotNull(errorMessage)) {
						errorMessageList.add(errorMessage);
					} else {
						cachedDataList.add(data);
					}
					if (cachedDataList.size() >= BATCH_COUNT) {
						saveData();
						// 存储完成清理 list
						cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);
					}
				}

				@Override
				public void doAfterAllAnalysed(AnalysisContext context) {
					saveData();
				}

				/**
				 * 加上存储数据库
				 */
				private void saveData() {
					log.info("{}条数据，开始存储数据库！", cachedDataList.size());
					importTSalaryEmployee(cachedDataList, errorMessageList);
					log.info("存储数据库成功！");
				}
			}).sheet().doRead();
		} catch (Exception e) {
			log.error(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR, e);
			return R.failed(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR);
		}
		return R.ok(errorMessageList);
	}

	private void importTSalaryEmployee(List<TSalaryEmployeeVo> excelVOList, List<ErrorMessage> errorMessageList) {
		// 执行数据插入操作 组装
		for (int i = 0; i < excelVOList.size(); i++) {
			TSalaryEmployeeVo excel = excelVOList.get(i);
			// 数据合法情况

			// 插入
			insertExcel(excel);
			errorMessageList.add(new ErrorMessage(excel.getRowIndex(), CommonConstants.SAVE_SUCCESS));
		}
	}

	/**
	 * 插入excel bad record
	 */
	private void insertExcel(TSalaryEmployeeVo excel) {
		TSalaryEmployee insert = new TSalaryEmployee();
		BeanUtil.copyProperties(excel, insert);
		this.save(insert);
	}

	/**
	 * @param employee
	 * @Description: 新建薪资员工，返回值为null，保存成功。其余都是失败原因
	 * @Author: hgw
	 * @Date: 2022/8/17 16:17
	 * @return: java.lang.String
	 **/
	@Override
	public String saveNewSalaryEmployee(TSalaryEmployee employee) {
		if (Common.isNotNull(employee.getEmpName()) && Common.isNotNull(employee.getEmpIdcard())) {
			// 银行卡
			if (Common.isNotNull(employee.getBankNo())) {
				// 调用校验服务
				TCheckBankNo checkIdCard = new TCheckBankNo();
				checkIdCard.setName(employee.getEmpName());
				checkIdCard.setBankNo(employee.getBankNo());
				R<CheckBankNoVo> checkListR = HttpDaprUtil.invokeMethodPost(checkProperties.getAppUrl(), checkProperties.getAppId()
						, "/tcheckbankno/inner/checkBankNo", checkIdCard, CheckBankNoVo.class, SecurityConstants.FROM_IN);
				String pre = "姓名与卡号验证：";
				if (checkListR != null && checkListR.getData() != null) {
					CheckBankNoVo vo = checkListR.getData();
					TCheckBankNo check = (null == vo.getRes()) ? null : vo.getRes().getData();
					if (Common.isEmpty(vo)) {
						return pre + SalaryConstants.CHECK_NO_RESPONSE;
					} else if (!CommonConstants.SUCCESS.equals(vo.getRes().getCode())) {
						return pre + vo.getRes().getMsg();
					} else if (check != null && !check.getResult().equals(CommonConstants.ZERO_ONE)) {
						return pre + check.getRemark();
					}
				} else {
					return pre + SalaryConstants.CHECK_NO_RESPONSE;
				}
			}
//			if (Common.isNotNull(employee.getBankNo())) {
//				// 调用校验服务
//				TCheckBankNo checkIdCard = new TCheckBankNo();
//				checkIdCard.setName(employee.getEmpName());
//				checkIdCard.setBankNo(employee.getBankNo());
//				R<TCheckBankNo> checkListR = HttpDaprUtil.invokeMethodPost(checkProperties.getAppUrl(), checkProperties.getAppId()
//						, "/tcheckbankno/inner/checkBankNo", checkIdCard, TCheckBankNo.class, SecurityConstants.FROM_IN);
//				if (checkListR != null && checkListR.getData() != null) {
//					TCheckBankNo check = checkListR.getData();
//					if (!CommonConstants.ONE_STRING.equals(check.getResult())) {
//						return check.getMessage();
//					}
//				} else {
//					return "校验服务未找到银行卡，请联系管理员";
//				}
//			}
			// 手机号
			if (Common.isNotNull(employee.getEmpPhone())) {
				// 调用校验服务-校验手机号
				TCheckMobile checkMobile = new TCheckMobile();
				checkMobile.setMobile(employee.getEmpPhone());
				R<TCheckMobile> a = HttpDaprUtil.invokeMethodPost(checkProperties.getAppUrl(), checkProperties.getAppId()
						, "/tcheckmobile/inner/checkOneMobile", checkMobile, TCheckMobile.class, SecurityConstants.FROM_IN);
				String pre = "手机号验证：";
				if (a != null && a.getData() != null) {
					checkMobile = a.getData();
					if (checkMobile != null && Common.isNotNull(checkMobile.getStatus())) {
						if (!CommonConstants.ONE_STRING.equals(checkMobile.getStatus())) {
							return pre + checkMobile.getMessage();
						}
					} else {
						return pre + "校验服务未找到手机号，请联系管理员！";
					}
				}
			}
		}
		this.save(employee);
		return null;
	}

	/**
	 * @param inputStream
	 * @Description: 批量更新-薪资人员信息
	 * @Author: hgw
	 * @Date: 2022/10/10 17:17
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage>>
	 **/
	@Override
	public R<List<ErrorMessage>> batchUpdateSalaryEmployee(InputStream inputStream) {
		List<ErrorMessage> errorMessageList = new ArrayList<>();
		ExcelUtil<TSalaryEmployeeUpdateVo> util1 = new ExcelUtil<>(TSalaryEmployeeUpdateVo.class);
		// 写法2：
		// 匿名内部类 不用额外写一个DemoDataListener
		// 这里 需要指定读用哪个class去读，然后读取第一个sheet 文件流会自动关闭
		try {

			EasyExcelFactory.read(inputStream, TSalaryEmployeeUpdateVo.class, new ReadListener<TSalaryEmployeeUpdateVo>() {
				/**
				 * 单次缓存的数据量
				 */
				public static final int BATCH_COUNT = CommonConstants.BATCH_COUNT;
				/**
				 *临时存储
				 */
				private List<TSalaryEmployeeUpdateVo> cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);

				@Override
				public void invoke(TSalaryEmployeeUpdateVo data, AnalysisContext context) {
					ReadRowHolder readRowHolder = context.readRowHolder();
					Integer rowIndex = readRowHolder.getRowIndex();
					data.setRowIndex(rowIndex + 1);
					ErrorMessage errorMessage = util1.checkEntity(data, data.getRowIndex());
					if (Common.isNotNull(errorMessage)) {
						errorMessageList.add(errorMessage);
					} else {
						cachedDataList.add(data);
					}
					if (cachedDataList.size() >= BATCH_COUNT) {
						saveData();
						// 存储完成清理 list
						cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);
					}
				}

				@Override
				public void doAfterAllAnalysed(AnalysisContext context) {
					saveData();
				}

				/**
				 * 加上存储数据库
				 */
				private void saveData() {
					log.info("{}条数据，开始存储数据库！", cachedDataList.size());
					if (!cachedDataList.isEmpty()) {
						updateTSalaryEmployee(cachedDataList, errorMessageList);
					}
					log.info("存储数据库成功！");
				}
			}).sheet().doRead();
		} catch (Exception e) {
			log.error(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR, e);
			return R.failed(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR);
		}
		boolean isTrue = true;
		for (ErrorMessage message : errorMessageList) {
			if (!CommonConstants.SAVE_SUCCESS.equals(message.getMessage())) {
				isTrue = false;
				break;
			}
		}
		if (isTrue) {
			return R.ok();
		} else {
			return R.ok(errorMessageList);
		}
	}

	/**
	 * @Description: 批量更新
	 * @Author: hgw
	 * @Date: 2022/10/10 17:56
	 * @return: void
	 **/
	public void updateTSalaryEmployee(List<TSalaryEmployeeUpdateVo> excelVOList, List<ErrorMessage> errorMessageList) {
		/// 个性化校验逻辑
		TSalaryEmployeeUpdateVo excel;
		List<String> idCardList = new ArrayList<>();
		// 执行数据插入操作 组装
		for (TSalaryEmployeeUpdateVo vo : excelVOList) {
			idCardList.add(vo.getEmpIdcard());
		}
		List<TSalaryEmployee> empList = baseMapper.getListByIdCard(idCardList);
		// 人员Map
		Map<String, TSalaryEmployee> empMap = new HashMap<>();
		for (TSalaryEmployee e : empList) {
			empMap.put(e.getEmpIdcard() + CommonConstants.DOWN_LINE_STRING + e.getDeptNo(), e);
		}
		TSalaryEmployee emp;
		// 开户行省市的区域暂存
		String areaStr;
		boolean curTaxMonth;
		for (int i = 0; i < excelVOList.size(); i++) {
			excel = excelVOList.get(i);
			if (excel != null) {
				if (Common.isNotNull(excel.getEmpIdcard()) && Common.isNotNull(excel.getDeptNo())) {
					emp = empMap.get(excel.getEmpIdcard() + CommonConstants.DOWN_LINE_STRING + excel.getDeptNo());
					if (Common.isNotNull(emp)) {
						curTaxMonth = false;
						if (Common.isNotNull(emp.getTaxMonth()) && Common.isNotNull(excel.getTaxMonth()) && emp.getTaxMonth().length()==6) {
							curTaxMonth = true;
						} else if (Common.isNotNull(excel.getTaxMonth())) {
							emp.setTaxMonth(excel.getTaxMonth());
						}
						if (Common.isNotNull(excel.getBankName())) {
							emp.setBankName(excel.getBankName());
						}
						if (Common.isNotNull(excel.getBankSubName())) {
							emp.setBankSubName(excel.getBankSubName());
						}
						if (Common.isNotNull(excel.getBankProvince())) {
							areaStr = ExcelUtil.getRedisAreaValue(CacheConstants.AREA_VALUE + excel.getBankProvince().trim());
							if (Common.isNotNull(areaStr)) {
								emp.setBankProvince(areaStr);
								if (Common.isNotNull(excel.getBankCity())) {
									areaStr = ExcelUtil.getRedisAreaValue(CacheConstants.AREA_VALUE + excel.getBankCity().trim()
											+ CommonConstants.DOWN_LINE_STRING + excel.getBankProvince().trim());
									if (Common.isNotNull(areaStr)) {
										emp.setBankCity(areaStr);
									} else {
										errorMessageList.add(new ErrorMessage(excel.getRowIndex(), SalaryConstants.CITY_ERROR));
										continue;
									}
								}
							} else {
								errorMessageList.add(new ErrorMessage(excel.getRowIndex(), SalaryConstants.PROVINCE_ERROR));
								continue;
							}
						}
						if (Common.isNotNull(excel.getBankNo())) {
							emp.setBankNo(excel.getBankNo());
						}
						if (Common.isNotNull(excel.getEmpPhone())) {
							emp.setEmpPhone(excel.getEmpPhone());
						}
						baseMapper.updateById(emp);
						if (curTaxMonth) {
							errorMessageList.add(new ErrorMessage(excel.getRowIndex(), SalaryConstants.CUR_TAX_INFO));
						} else {
							errorMessageList.add(new ErrorMessage(excel.getRowIndex(), CommonConstants.SAVE_SUCCESS));
						}
					} else {
						errorMessageList.add(new ErrorMessage(excel.getRowIndex(), SalaryConstants.NOT_HAVE_EMP));
					}
				} else {
					errorMessageList.add(new ErrorMessage(excel.getRowIndex(), SalaryConstants.IDCARD_DEPT_NO_MUST));
				}
			} else {
				errorMessageList.add(new ErrorMessage(CommonConstants.ZERO_INT, SalaryConstants.IDCARD_DEPT_NO_MUST));
			}
		}
	}
}
