/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.MenuUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import com.yifu.cloud.plus.v1.yifu.social.entity.TIncome;
import com.yifu.cloud.plus.v1.yifu.social.entity.TIncomeDetail;
import com.yifu.cloud.plus.v1.yifu.social.service.TIncomeService;
import com.yifu.cloud.plus.v1.yifu.social.vo.TIncomeSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 收入明细表
 *
 * @author hgw
 * @date 2022-08-30 17:34:58
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tincome")
@Tag(name = "收入明细表管理")
public class TIncomeController {

	private final TIncomeService tIncomeService;

	private final MenuUtil menuUtil;

	/**
	 * 简单分页查询
	 *
	 * @param page    分页对象
	 * @param tIncome 收入明细表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TIncome>> getTIncomePage(Page<TIncome> page, TIncomeSearchVo tIncome) {
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, tIncome);
		return new R<>(tIncomeService.getTIncomePage(page, tIncome));
	}

	/**
	 * 通过id查询收入明细表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('social_tincome_get')")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tincome_get')")
	public R<TIncome> getById(@PathVariable("id") String id) {
		return R.ok(tIncomeService.getById(id));
	}

	/**
	 * 新增收入明细表
	 *
	 * @param tIncome 收入明细表
	 * @return R
	 */
	@Operation(summary = "新增收入明细表", description = "新增收入明细表：hasPermission('social_tincome_add')")
	@SysLog("新增收入明细表")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('social_tincome_add')")
	public R<Boolean> save(@RequestBody TIncome tIncome) {
		return R.ok(tIncomeService.save(tIncome));
	}

	/**
	 * 修改收入明细表
	 *
	 * @param tIncome 收入明细表
	 * @return R
	 */
	@Operation(summary = "修改收入明细表", description = "修改收入明细表：hasPermission('social_tincome_edit')")
	@SysLog("修改收入明细表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('social_tincome_edit')")
	public R<Boolean> updateById(@RequestBody TIncome tIncome) {
		return R.ok(tIncomeService.updateById(tIncome));
	}

	/**
	 * 通过id删除收入明细表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除收入明细表", description = "通过id删除收入明细表：hasPermission('social_tincome_del')")
	@SysLog("通过id删除收入明细表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tincome_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tIncomeService.removeById(id));
	}

	/**
	 * 收入明细表 批量导入
	 *
	 * @author hgw
	 * @date 2022-08-30 17:34:58
	 **/
	@SneakyThrows
	@Operation(description = "批量新增收入明细表 hasPermission('social_tincome-batch-import')")
	@SysLog("批量新增收入明细表")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('social_tincome-batch-import')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file) {
		return tIncomeService.importDiy(file.getInputStream());
	}

	/**
	 * 收入明细表 批量导出
	 *
	 * @author hgw
	 * @date 2022-08-30 17:34:58
	 **/
	@Operation(description = "导出收入明细表 hasPermission('social_tincome-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('social_tincome-export')")
	public void export(HttpServletResponse response, @RequestBody TIncomeSearchVo searchVo) {
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, searchVo);
		tIncomeService.listExport(response, searchVo);
	}

	/**
	 * @Description: 新增收入明细-详情表，同时统计；
	 * @Author: hgw
	 * @Date: 2022/8/31 16:34
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.Boolean>
	 **/
	@Inner
	@PostMapping("/inner/saveDetail")
	public Boolean saveDetail(@RequestBody TIncomeDetail tIncomeDetail) {
		return tIncomeService.saveDetail(tIncomeDetail);
	}

	/**
	 * @Description: 定时任务统计表未推送数据推送；
	 * @Author: huyc
	 * @Date: 2022/9/5 16:34
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.Boolean>
	 **/
	@Inner
	@PostMapping("/inner/pushDetail")
	public void pushDetail() {
		tIncomeService.pushDetail();
	}
}
