package com.yifu.cloud.plus.v1.yifu.social.controller;/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.util.DateUtil;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.salary.vo.HaveSalaryNoSocialSearchVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TPaymentBySalaryVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TPaymentVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.UpdateSocialFoundVo;
import com.yifu.cloud.plus.v1.yifu.social.entity.TPaymentInfo;
import com.yifu.cloud.plus.v1.yifu.social.service.TPaymentInfoService;
import com.yifu.cloud.plus.v1.yifu.social.vo.TPaymentInfoSearchVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.TPaymentInfoVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.http.HttpHeaders;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 缴费库
 * @author fxj
 * @date 2022-07-22 17:01:22
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tpaymentinfo" )
@Tag(name = "缴费库管理")
@SecurityRequirement(name = HttpHeaders.AUTHORIZATION)
public class TPaymentInfoController {

	private final TPaymentInfoService tPaymentInfoService;

	/**
	 * 简单分页查询
	 *
	 * @param page         分页对象
	 * @param tPaymentInfo 缴费库
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TPaymentInfo>> getTPaymentInfoPage(Page<TPaymentInfo> page, TPaymentInfoSearchVo tPaymentInfo) {
		return new R<>(tPaymentInfoService.getTPaymentInfoPage(page, tPaymentInfo));
	}

	/**
	 * 通过id查询缴费库
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询")
	@GetMapping("/{id}")
	public R<TPaymentInfo> getById(@PathVariable("id") String id) {
		return R.ok(tPaymentInfoService.getById(id));
	}

	/**
	 * 修改缴费库
	 *
	 * @param tPaymentInfo 缴费库
	 * @return R
	 */
	@Operation(summary = "修改缴费库", description = "修改缴费库：hasPermission('social_tpaymentinfo_edit')")
	@SysLog("修改缴费库")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_edit')")
	public R<Boolean> updateById(@RequestBody TPaymentInfo tPaymentInfo) {
		return R.ok(tPaymentInfoService.updateById(tPaymentInfo));
	}

	/**
	 * 通过id删除缴费库
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除缴费库", description = "通过id删除缴费库：hasPermission('social_tpaymentinfo_del')")
	@SysLog("通过id删除缴费库")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return tPaymentInfoService.removeInFoById(id);
	}

	/**
	 * 通过ids删除一条记录
	 *
	 * @param ids
	 * @return R
	 */
	@Operation(summary = "通过ids删除缴费库", description = "通过ids删除缴费库")
	@SysLog("删除缴费库")
	@DeleteMapping("/removeBatchByIds")
	public R<Boolean> removeBatchByIds(String ids) {
		return tPaymentInfoService.removeAllInfoById(ids);
	}

	/**
	 * 通过id查询单条记录(包含明细)
	 *
	 * @param id
	 * @return R
	 */
	@Operation(summary = "通过id查询单条记录")
	@GetMapping("/allInfo/{id}")
	public R<TPaymentInfoVo> getAllInfoById(@PathVariable("id") String id) {
		return tPaymentInfoService.getAllInfoById(id);
	}

	/**
	 * 省市社保 批量导入
	 *
	 * @author huyc
	 * @date 2022-07-22
	 **/
	@SneakyThrows
	@Operation(description = "导入省市社保 hasPermission('social_tpaymentinfo_batchImport')")
	@SysLog("批量新增缴费库")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_batchImport')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file, @RequestParam String random) {
		return tPaymentInfoService.importSocialDiy(file.getInputStream(), random);
	}

	/**
	 * 导入省市社保-合肥三险
	 *
	 * @author huyc
	 * @date 2022-07-22
	 **/
	@SneakyThrows
	@Operation(description = "导入省市社保-合肥三险")
	@SysLog("批量新增缴费库-合肥三险")
	@PostMapping("/importListSocialHeFei")
	public R<List<ErrorMessage>> importListSocialHeFei(@RequestBody MultipartFile file, @RequestParam String random,
													   @RequestParam(value = "type") String type) {
		return tPaymentInfoService.importSocialHeFeiDiy(file.getInputStream(), random, type);
	}

	/**
	 * 导入公积金
	 *
	 * @author huyc
	 * @date 2022-07-27
	 **/
	@SneakyThrows
	@Operation(description = "导入公积金 hasPermission('social_tpaymentinfo_batchImport')")
	@SysLog("导入公积金")
	@PostMapping("/batchImportPaymentFundInfo")
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_batchImport')")
	public R<List<ErrorMessage>> importListFund(@RequestBody MultipartFile file) {
		return tPaymentInfoService.batchImportPaymentFundInfo(file.getInputStream());
	}

	/**
	 * 缴费库 批量导出
	 *
	 * @author huyc
	 * @date 2022-07-22
	 **/
	@Operation(description = "导出缴费库 hasPermission('social_tpaymentinfo_export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_export')")
	public void export(HttpServletResponse response, @RequestBody TPaymentInfoSearchVo searchVo) {
		tPaymentInfoService.listExport(response, searchVo);
	}

	/**
	 * 缴费库合并导出
	 *
	 * @author huyc
	 * @date 2022-10-31
	 **/
	@Operation(description = "合并导出缴费库 hasPermission('social_tpaymentinfo_export')")
	@PostMapping("/sumExport")
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_export')")
	public void sumExport(HttpServletResponse response, @RequestBody TPaymentInfoSearchVo searchVo) {
		tPaymentInfoService.listSumExport(response, searchVo);
	}

	/**
	 * 根据条件批量删除
	 *
	 * @param empIdCard         员工身份证
	 * @param socialHouseId     社保户
	 * @param fundHouseId       公积金户
	 * @param socialCreateMonth 生成月份
	 * @param socialPayMonth    缴纳月份
	 * @return R
	 */
	@Operation(description = "根据条件批量删除")
	@SysLog("根据条件批量删除")
	@PostMapping("/removeBatchByIdsAndType")
	public R<Boolean> removeBatchByIdsAndType(@RequestParam(name = "empIdCard", required = false) String empIdCard,
									 @RequestParam(name = "socialHouseId", required = false) String socialHouseId,
									 @RequestParam(name = "fundHouseId", required = false) String fundHouseId,
									 @RequestParam(name = "socialCreateMonth", required = false) String socialCreateMonth,
									 @RequestParam(name = "socialPayMonth", required = false) String socialPayMonth) {
		return tPaymentInfoService.removeBatchByInfo(empIdCard, socialHouseId,
				fundHouseId, socialCreateMonth, socialPayMonth);
	}

	/**
	 * @Description: 薪资获取缴费库
	 * @Author: hgw
	 * @Date: 2022/8/10 18:06
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.salary.vo.TPaymentVo>
	 **/
	@Inner
	@PostMapping("/inner/selectTPaymentBySalaryVo")
	public TPaymentVo selectTPaymentBySalaryVo(@RequestBody TPaymentBySalaryVo tPaymentBySalaryVo) {
		return tPaymentInfoService.selectTPaymentBySalaryVo(tPaymentBySalaryVo);
	}

	//hgw 以下4个方法，薪资变更缴费库预估库结算状态专用
	/**
	 * @param idList
	 * @Description: 社保变为结算
	 * @Author: hgw
	 * @Date: 2022/8/16 18:22
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updatePaymentSocialStatusToSettle")
	public int updatePaymentSocialStatusToSettle(@RequestBody List<String> idList) {
		return tPaymentInfoService.updatePaymentSocialStatusToSettle(idList);
	}
	/**
	 * @param idList
	 * @Description: 社保变为结算
	 * @Author: hgw
	 * @Date: 2022/8/16 18:22
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updatePaymentSocialStatusToNoSettle")
	public int updatePaymentSocialStatusToNoSettle(@RequestBody List<String> idList) {
		return tPaymentInfoService.updatePaymentSocialStatusToNoSettle(idList);
	}
	/**
	 * @param idList
	 * @Description: 公积金变为结算
	 * @Author: hgw
	 * @Date: 2022/8/16 18:22
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updatePaymentFundStatusToSettle")
	public int updatePaymentFundStatusToSettle(@RequestBody List<String> idList) {
		return tPaymentInfoService.updatePaymentFundStatusToSettle(idList);
	}
	/**
	 * @param idList
	 * @Description: 公积金变为未结算
	 * @Author: hgw
	 * @Date: 2022/8/16 18:22
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updatePaymentFundStatusToNoSettle")
	public int updatePaymentFundStatusToNoSettle(@RequestBody List<String> idList) {
		return tPaymentInfoService.updatePaymentFundStatusToNoSettle(idList);
	}

	/**
	 * @Author fxj
	 * @Description 有工资无社保提醒：通过身份证号和月份获取对应的缴费库数据
	 * @Date 17:45 2022/8/16
	 * @Param
	 * @return
	**/
	@Inner
	@PostMapping("/inner/listByEmpdIdCard")
	public R<List<TPaymentInfo>> getPaymentinfoListByEmpdIdCard(@RequestBody HaveSalaryNoSocialSearchVo searchVo) {
		return new R<>(tPaymentInfoService.list(Wrappers.<TPaymentInfo>query().lambda()
				.in(TPaymentInfo::getEmpIdcard, searchVo.getIdCards())
				.and(obj->obj.eq(TPaymentInfo::getSocialCreateMonth,searchVo.getSettleMonth())
						.or()
						.eq(TPaymentInfo::getSocialCreateMonth, DateUtil.getYearAndMonth(searchVo.getSettleMonth(),1))
						.or()
						.eq(TPaymentInfo::getSocialCreateMonth,DateUtil.getYearAndMonth(searchVo.getSettleMonth(),1))
				)
				.groupBy(TPaymentInfo::getEmpIdcard)));
	}

	/**
	 * @param list
	 * @Description: 更新项目单位信息
	 * @Author: huyc
	 * @Date: 2022/8/25
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updatePaymentSocialAndFound")
	public void updatePaymentSocialAndFound(@RequestBody UpdateSocialFoundVo list) {
		tPaymentInfoService.updatePaymentSocialAndFound(list);
	}

	/**
	 * @Description: 定时任务推送社保缴费库明细的数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 * @return: void
	 **/
	@Operation(summary = "定时任务推送社保缴费库明细的数据", description = "定时任务推送社保缴费库明细的数据")
	@SysLog("定时任务推送社保缴费库明细的数据")
	@Inner
	@PostMapping("/inner/createPaymentSocialInfo")
	public void createPaymentSocialInfo() {
		tPaymentInfoService.createPaymentSocialInfo();
	}

	/**
	 * @Description: 定时任务推送公积金缴费库明细的数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 * @return: void
	 **/
	@Operation(summary = "定时任务推送公积金缴费库明细的数据", description = "定时任务推送公积金缴费库明细的数据")
	@SysLog("定时任务推送公积金缴费库明细的数据")
	@Inner
	@PostMapping("/inner/createPaymentFundInfo")
	public void createPaymentFundInfo() {
		tPaymentInfoService.createPaymentFundInfo();
	}

	/**
	 * @Description: 定时生成缴费库的收入数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 * @return: void
	 **/
	@Operation(summary = "定时生成缴费库的收入数据", description = "定时生成缴费库的收入数据")
	@SysLog("定时生成缴费库的收入数据")
	@Inner
	@PostMapping("/inner/createPaymentInfoIncome")
	public void createPaymentInfoIncome() {
		tPaymentInfoService.createPaymentInfoIncome();
	}

	/**
	 * @Description: 定时生成公积金的收入数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 * @return: void
	 **/
	@Operation(summary = "定时生成公积金的收入数据", description = "定时生成公积金的收入数据")
	@SysLog("定时生成公积金的收入数据")
	@Inner
	@PostMapping("/inner/createPaymentFundIncome")
	public void createPaymentFundIncome() {
		tPaymentInfoService.createPaymentFundIncome();
	}

	/**
	 * @param info
	 * @Description: 查询缴费库
	 * @Author: huyc
	 * @Date: 2022/8/25
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/getPaymentSocialAndFound")
	public TPaymentVo getPaymentSocialAndFound(@RequestBody TPaymentInfo info) {
		return tPaymentInfoService.getPaymentSocialAndFound(info);
	}

	/**
	 * @Description: 手动推送社保缴费库明细的数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 * @return: void
	 **/
	@Operation(summary = "手动推送社保缴费库明细的数据", description = "手动推送社保缴费库明细的数据")
	@SysLog("手动推送社保缴费库明细的数据")
	@PostMapping("/pushPaymentSocialFundInfo")
	public R<Boolean> pushPaymentSocialFundInfo() {
		tPaymentInfoService.pushPaymentSocialFundInfo();
		return R.ok();
	}
}
