/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.salary.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CacheConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.ResultConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.redis.RedisDistributedLock;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.util.RedisUtil;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TMobileChangeInfo;
import com.yifu.cloud.plus.v1.yifu.salary.service.TMobileChangeInfoService;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TMobileChangeInfoSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * @author fxj
 * @date 2024-12-18 15:39:56
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tmobilechangeinfo")
@Tag(name = "管理")
public class TMobileChangeInfoController {

	private final TMobileChangeInfoService tMobileChangeInfoService;

	private final RedisUtil redisUtil;
	/**
	 * 简单分页查询
	 *
	 * @param page              分页对象
	 * @param tMobileChangeInfo
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TMobileChangeInfo>> getTMobileChangeInfoPage(Page<TMobileChangeInfo> page, TMobileChangeInfoSearchVo tMobileChangeInfo) {
		return new R<>(tMobileChangeInfoService.getTMobileChangeInfoPage(page, tMobileChangeInfo));
	}

	/**
	 * 不分页查询
	 *
	 * @param tMobileChangeInfo
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	//@PreAuthorize("@pms.hasPermission('demo_tmobilechangeinfo_get')" )
	public R<List<TMobileChangeInfo>> getTMobileChangeInfoNoPage(@RequestBody TMobileChangeInfoSearchVo tMobileChangeInfo) {
		return R.ok(tMobileChangeInfoService.noPageDiy(tMobileChangeInfo));
	}

	/**
	 * 通过id查询
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('demo_tmobilechangeinfo_get')")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('demo_tmobilechangeinfo_get')")
	public R<TMobileChangeInfo> getById(@PathVariable("id") String id) {
		return R.ok(tMobileChangeInfoService.getById(id));
	}

	/**
	 * 新增
	 *
	 * @param tMobileChangeInfo
	 * @return R
	 */
	@Operation(summary = "新增", description = "新增")
	@SysLog("新增")
	@PostMapping("/add")
	public R<Boolean> save(@RequestBody TMobileChangeInfo tMobileChangeInfo) {
		// 获取redis分布式事务锁
		String key = CacheConstants.MOBILE_CHANGE_LIMIT + CommonConstants.DOWN_LINE_STRING + "mobileChangeFlag";
		String requestId;
		try {
			requestId = RedisDistributedLock.getLock(key,"10");
		} catch (Exception e) {
			throw new RuntimeException(ResultConstants.NO_GETLOCK_DATA+CommonConstants.DOWN_LINE_STRING+e.getMessage());
		}
		try {
			if (Common.isNotNull(requestId)) {
				//主动释放锁
				tMobileChangeInfo.setHandleStatus(CommonConstants.ZERO_STRING);
				if (null != redisUtil.get(CommonConstants.C_PREFIX.concat(tMobileChangeInfo.getMobileNew()))) {
					String code = (String) redisUtil.get(CommonConstants.C_PREFIX.concat(tMobileChangeInfo.getMobileNew()));
					if (!code.equals(tMobileChangeInfo.getSendCode())) {
						return R.failed("验证码错误，请核实");
					}
				} else {
					return R.failed("未找到手机号对应的验证码或验证码已超时，请重新获取");
				}
				return R.ok(tMobileChangeInfoService.save(tMobileChangeInfo));
			} else {
				return R.failed(ResultConstants.NO_GETLOCK_DATA);
			}
		}finally {
			//主动释放锁
			RedisDistributedLock.unlock(key, requestId);
		}


	}

	/**
	 * 修改
	 *
	 * @param tMobileChangeInfo
	 * @return R
	 */
	@Operation(summary = "修改", description = "修改：hasPermission('demo_tmobilechangeinfo_edit')")
	@SysLog("修改")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('demo_tmobilechangeinfo_edit')")
	public R<Boolean> updateById(@RequestBody TMobileChangeInfo tMobileChangeInfo) {
		return R.ok(tMobileChangeInfoService.updateById(tMobileChangeInfo));
	}

	/**
	 * 通过id删除
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除", description = "通过id删除：hasPermission('demo_tmobilechangeinfo_del')")
	@SysLog("通过id删除")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('demo_tmobilechangeinfo_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tMobileChangeInfoService.removeById(id));
	}

	/**
	 * 批量导入
	 *
	 * @author fxj
	 * @date 2024-12-18 15:39:56
	 **/
	@SneakyThrows
	@Operation(description = "批量新增 hasPermission('demo_tmobilechangeinfo-batch-import')")
	@SysLog("批量新增")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('demo_tmobilechangeinfo-batch-import')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file) {
		return tMobileChangeInfoService.importDiy(file.getInputStream());
	}

	/**
	 * 批量导出
	 *
	 * @author fxj
	 * @date 2024-12-18 15:39:56
	 **/
	@Operation(description = "导出 hasPermission('demo_tmobilechangeinfo-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('demo_tmobilechangeinfo-export')")
	public void export(HttpServletResponse response, @RequestBody TMobileChangeInfoSearchVo searchVo) {
		tMobileChangeInfoService.listExport(response, searchVo);
	}
}
