/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.salary.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ArrayUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.read.listener.ReadListener;
import com.alibaba.excel.read.metadata.holder.ReadRowHolder;
import com.alibaba.excel.util.ListUtils;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.check.entity.TCheckBankNo;
import com.yifu.cloud.plus.v1.check.entity.TCheckIdCard;
import com.yifu.cloud.plus.v1.check.entity.TCheckMobile;
import com.yifu.cloud.plus.v1.check.vo.CheckBankNoVo;
import com.yifu.cloud.plus.v1.check.vo.CheckBatchVo;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TSettleDomain;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CacheConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.SecurityConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.*;
import com.yifu.cloud.plus.v1.yifu.common.dapr.config.DaprCheckProperties;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.HttpDaprUtil;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TSalaryEmployee;
import com.yifu.cloud.plus.v1.yifu.salary.mapper.TSalaryEmployeeMapper;
import com.yifu.cloud.plus.v1.yifu.salary.service.TBankSetService;
import com.yifu.cloud.plus.v1.yifu.salary.service.TSalaryEmployeeService;
import com.yifu.cloud.plus.v1.yifu.salary.util.SalaryConstants;
import com.yifu.cloud.plus.v1.yifu.salary.vo.*;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 薪酬人员表
 *
 * @author hgw
 * @date 2022-08-05 11:40:14
 */
@Log4j2
@Service
@EnableConfigurationProperties({DaprCheckProperties.class})
public class TSalaryEmployeeServiceImpl extends ServiceImpl<TSalaryEmployeeMapper, TSalaryEmployee> implements TSalaryEmployeeService {

	@Autowired
	private DaprCheckProperties checkProperties;

	@Autowired
	private TBankSetService tBankSetService;
	/**
	 * 薪酬人员表简单分页查询
	 *
	 * @param tSalaryEmployee 薪酬人员表
	 * @return
	 */
	@Override
	public IPage<TSalaryEmployee> getTSalaryEmployeePage(Page<TSalaryEmployee> page, TSalaryEmployeeSearchVo tSalaryEmployee) {
		return baseMapper.getTSalaryEmployeePage(page, tSalaryEmployee);
	}

	/**
	 * 薪酬人员表批量导出
	 *
	 * @return
	 */
	@Override
	public void listExport(HttpServletResponse response, TSalaryEmployeeSearchVo searchVo) {
		String fileName = "薪酬人员查询批量导出" + DateUtil.getThisTime() + CommonConstants.XLSX;
		//获取要导出的列表
		List<TSalaryEmployeeExportVo> list = new ArrayList<>();
		long count = noPageCountDiy(searchVo);
		ServletOutputStream out = null;
		try {
			out = response.getOutputStream();
			response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
			response.setCharacterEncoding("utf-8");
			response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName, "UTF-8"));
			// 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
			//EasyExcel.write(out, TEmpBadRecord.class).sheet("不良记录").doWrite(list)
			ExcelWriter excelWriter = EasyExcel.write(out, TSalaryEmployeeExportVo.class).build();
			int index = 0;
			if (count > CommonConstants.ZERO_INT) {
				WriteSheet writeSheet;
				ExcelUtil<TSalaryEmployeeExportVo> util;
				for (int i = 0; i <= count; ) {
					// 获取实际记录
					searchVo.setLimitStart(i);
					searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
					list = noPageDiy(searchVo);
					/*if (Common.isNotNull(list)) {
						util = new ExcelUtil<>(TSalaryEmployeeExportVo.class);
						for (TSalaryEmployeeExportVo vo : list) {
							util.convertEntity(vo, null, null, null);
						}
					}*/
					if (Common.isNotNull(list)) {
						writeSheet = EasyExcel.writerSheet("薪酬人员表" + index).build();
						excelWriter.write(list, writeSheet);
						index++;
					}
					i = i + CommonConstants.EXCEL_EXPORT_LIMIT;
					if (Common.isNotNull(list)) {
						list.clear();
					}
				}
			} else {
				WriteSheet writeSheet = EasyExcel.writerSheet("薪酬人员表" + index).build();
				excelWriter.write(list, writeSheet);
			}
			if (Common.isNotNull(list)) {
				list.clear();
			}
			out.flush();
			excelWriter.finish();
		} catch (Exception e) {
			log.error("执行异常", e);
		} finally {
			try {
				if (null != out) {
					out.close();
				}
			} catch (IOException e) {
				log.error("执行异常", e);
			}
		}
	}

	@Override
	public List<TSalaryEmployeeExportVo> noPageDiy(TSalaryEmployeeSearchVo searchVo) {
		return baseMapper.noPageDiy(searchVo,Common.getList(searchVo.getIds()));
	}

	private Long noPageCountDiy(TSalaryEmployeeSearchVo searchVo) {
		LambdaQueryWrapper<TSalaryEmployee> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TSalaryEmployee::getId, idList);
		}
		if (Common.isNotNull(searchVo.getAuthSql())) {
			if (Common.isNotNull(searchVo.getAuthSql()) && searchVo.getAuthSql().contains("a.")) {
				searchVo.setAuthSql(searchVo.getAuthSql().replace("a.", " "));
			}
			wrapper.last(searchVo.getAuthSql());
		}
		return baseMapper.selectCount(wrapper);
	}

	private LambdaQueryWrapper buildQueryWrapper(TSalaryEmployeeSearchVo entity) {
		LambdaQueryWrapper<TSalaryEmployee> wrapper = Wrappers.lambdaQuery();
		wrapper.ne(TSalaryEmployee::getId,CommonConstants.ZERO_STRING);
		if (ArrayUtil.isNotEmpty(entity.getCreateTimes())) {
			wrapper.ge(TSalaryEmployee::getCreateTime, entity.getCreateTimes()[0])
					.le(TSalaryEmployee::getCreateTime,
							entity.getCreateTimes()[1]);
		}
		if (Common.isNotNull(entity.getCreateName())) {
			wrapper.eq(TSalaryEmployee::getCreateName, entity.getCreateName());
		}
		if (Common.isNotNull(entity.getEmpName())) {
			wrapper.eq(TSalaryEmployee::getEmpName, entity.getEmpName());
		}
		if (Common.isNotNull(entity.getEmpIdcard())) {
			wrapper.eq(TSalaryEmployee::getEmpIdcard, entity.getEmpIdcard());
		}
		return wrapper;
	}

	@Override
	public R<List<ErrorMessage>> importDiy(InputStream inputStream) {
		List<ErrorMessage> errorMessageList = new ArrayList<>();
		ExcelUtil<TSalaryEmployeeVo> util1 = new ExcelUtil<>(TSalaryEmployeeVo.class);
		// 写法2：
		// 匿名内部类 不用额外写一个DemoDataListener
		// 这里 需要指定读用哪个class去读，然后读取第一个sheet 文件流会自动关闭
		try {
			EasyExcel.read(inputStream, TSalaryEmployeeVo.class, new ReadListener<TSalaryEmployeeVo>() {
				/**
				 * 单次缓存的数据量
				 */
				public static final int BATCH_COUNT = CommonConstants.BATCH_COUNT;
				/**
				 *临时存储
				 */
				private List<TSalaryEmployeeVo> cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);

				@Override
				public void invoke(TSalaryEmployeeVo data, AnalysisContext context) {
					ReadRowHolder readRowHolder = context.readRowHolder();
					Integer rowIndex = readRowHolder.getRowIndex();
					data.setRowIndex(rowIndex + 1);
					ErrorMessage errorMessage = util1.checkEntity(data, data.getRowIndex());
					if (Common.isNotNull(errorMessage)) {
						errorMessageList.add(errorMessage);
					} else {
						cachedDataList.add(data);
					}
					if (cachedDataList.size() >= BATCH_COUNT) {
						saveData();
						// 存储完成清理 list
						cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);
					}
				}

				@Override
				public void doAfterAllAnalysed(AnalysisContext context) {
					saveData();
				}

				/**
				 * 加上存储数据库
				 */
				private void saveData() {
					log.info("{}条数据，开始存储数据库！", cachedDataList.size());
					importTSalaryEmployee(cachedDataList, errorMessageList);
					log.info("存储数据库成功！");
				}
			}).sheet().doRead();
		} catch (Exception e) {
			log.error(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR, e);
			return R.failed(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR);
		}
		return R.ok(errorMessageList);
	}

	private void importTSalaryEmployee(List<TSalaryEmployeeVo> excelVOList, List<ErrorMessage> errorMessageList) {
		// 执行数据插入操作 组装
		for (int i = 0; i < excelVOList.size(); i++) {
			TSalaryEmployeeVo excel = excelVOList.get(i);
			// 数据合法情况

			// 插入
			insertExcel(excel);
			errorMessageList.add(new ErrorMessage(excel.getRowIndex(), CommonConstants.SAVE_SUCCESS));
		}
	}

	/**
	 * 插入excel bad record
	 */
	private void insertExcel(TSalaryEmployeeVo excel) {
		TSalaryEmployee insert = new TSalaryEmployee();
		BeanUtil.copyProperties(excel, insert);
		this.save(insert);
	}

	/**
	 * @param notLabour 是：非劳务费人员。 否：劳务费，不保存
	 * @param employee
	 * @Description: 新建薪资员工，返回值为null，保存成功。其余都是失败原因
	 * @Author: hgw
	 * @Date: 2022/8/17 16:17
	 * @return: java.lang.String
	 **/
	@Override
	public String saveNewSalaryEmployee(boolean notLabour,TSalaryEmployee employee) {
		String pre = this.checkNewEmpBankAndPhone(employee, false);
		if (pre != null) return pre;
		if (notLabour) {
			this.save(employee);
		}
		return null;
	}

	@Override
	public String saveNewEmployeeList(List<TSalaryEmployee> saveNewEmpList, List<ErrorMessage> errorList) {
		if (saveNewEmpList != null && !saveNewEmpList.isEmpty()) {
			// 银行卡
			List<TCheckBankNo> bankList = new ArrayList<>();
			List<String> phoneList = new ArrayList<>();
			TCheckBankNo checkBankNo;
			for (TSalaryEmployee employee : saveNewEmpList) {
				if (Common.isNotNull(employee.getBankNo())
						&& (Common.isEmpty(employee.getIssueStatus()) || employee.getIssueStatus().equals(CommonConstants.ZERO_INT))
						&& (Common.isEmpty(employee.getSalaryGiveTime())
						|| CommonConstants.ZERO_STRING.equals(employee.getSalaryGiveTime())
						|| (CommonConstants.ONE_STRING.equals(employee.getSalaryGiveTime()) && Common.isNotNull(employee.getBankNo())))) {
					checkBankNo = new TCheckBankNo();
					checkBankNo.setName(employee.getEmpName());
					checkBankNo.setBankNo(employee.getBankNo());
					bankList.add(checkBankNo);
				}
				if (Common.isNotNull(employee.getEmpPhone())) {
					phoneList.add(employee.getEmpPhone());
				}

			}
			// 批量校验 卡号与手机号
			Map<String, Boolean> bankMap = new HashMap<>();
			Map<String, Boolean> phoneMap = new HashMap<>();
			if (!bankList.isEmpty()) {
				R<CheckBatchVo> checkListR = HttpDaprUtil.invokeMethodPost(checkProperties.getAppUrl(), checkProperties.getAppId()
						, "/tcheckbankno/inner/checkBankNoBatch", bankList, CheckBatchVo.class, SecurityConstants.FROM_IN);
				if (checkListR != null && checkListR.getData() != null) {
					bankMap = checkListR.getData().getCheckMap();
				}
			}
			if (!phoneList.isEmpty()) {
				R<CheckBatchVo> phoneR = HttpDaprUtil.invokeMethodPost(checkProperties.getAppUrl(), checkProperties.getAppId()
						, "/tcheckmobile/inner/checkMobileBatch", phoneList, CheckBatchVo.class, SecurityConstants.FROM_IN);
				if (phoneR != null && phoneR.getData() != null) {
					phoneMap = phoneR.getData().getCheckMap();
				}
			}

			boolean isFalse = false;
			for (TSalaryEmployee employee : saveNewEmpList) {
				if (Common.isNotNull(employee.getBankNo())
						&& (Common.isEmpty(employee.getIssueStatus()) || employee.getIssueStatus().equals(CommonConstants.ZERO_INT))
						&& (Common.isEmpty(employee.getSalaryGiveTime())
							|| CommonConstants.ZERO_STRING.equals(employee.getSalaryGiveTime())
							|| (CommonConstants.ONE_STRING.equals(employee.getSalaryGiveTime()) && Common.isNotNull(employee.getBankNo())))) {
					if (bankMap.get(employee.getBankNo()) != null) {
						if (Boolean.FALSE.equals(bankMap.get(employee.getBankNo()))) {
							errorList.add(new ErrorMessage((employee.getLineNums() + 2), "第" + (employee.getLineNums() + 2) + "行：该员工新建失败：【姓名与卡号验证：认证不一致】"));
							isFalse = true;
						}
					} else {
						errorList.add(new ErrorMessage((employee.getLineNums() + 2), "第" + (employee.getLineNums() + 2) + "行：该员工新建失败：【姓名与卡号验证：" + SalaryConstants.CHECK_NO_RESPONSE + "】"));
						isFalse = true;
					}
				}
				if (phoneMap.get(employee.getEmpPhone()) != null) {
					if (Boolean.FALSE.equals(phoneMap.get(employee.getEmpPhone()))) {
						errorList.add(new ErrorMessage((employee.getLineNums() + 2), "第" + (employee.getLineNums() + 2) + "行：该员工新建失败：【手机号验证：手机号错误】"));
						isFalse = true;
					}
				} else {
					errorList.add(new ErrorMessage((employee.getLineNums() + 2), "第" + (employee.getLineNums() + 2) + "行：该员工新建失败：【手机号验证：" + SalaryConstants.CHECK_NO_RESPONSE + "】"));
					isFalse = true;
				}
			}
			if (isFalse) {
				return null;
			}
			this.saveBatch(saveNewEmpList);
		}
		return null;
	}

	/**
	 * @param employee
	 * @Description: 校验卡号与手机号
	 * @Author: hgw
	 * @Date: 2023/1/17 16:27
	 * @return: java.lang.String
	 **/
	private String checkNewEmpBankAndPhone(TSalaryEmployee employee, boolean isCheckIdCard) {
		if (Common.isNotNull(employee.getEmpName()) && Common.isNotNull(employee.getEmpIdcard())) {
			// 身份证校验（薪资人员导入才需要）
			if (isCheckIdCard) {
				// 调用校验服务
				String pre = "姓名与身份证验证：";
				TCheckIdCard checkIdCard = new TCheckIdCard();
				checkIdCard.setName(employee.getEmpName());
				checkIdCard.setIdCard(employee.getEmpIdcard());
				R<TCheckIdCard> checkListR = HttpDaprUtil.invokeMethodPost(checkProperties.getAppUrl(), checkProperties.getAppId()
						, "/tcheckidcard/inner/checkIdCardSingle", checkIdCard, TCheckIdCard.class, SecurityConstants.FROM_IN);
				if (checkListR != null && checkListR.getData() != null) {
					TCheckIdCard check = checkListR.getData();
					if (check == null || Common.isEmpty(check.getIsTrue())) {
						return pre + "校验服务器未返回，请联系管理员！";
					} else if (check.getIsTrue() == CommonConstants.ZERO_INT) {
						return pre + check.getReason();
					}
				} else {
					return pre + "校验服务器未返回，请联系管理员！";
				}
			}
			// 银行卡
			// 代发户的，不校验卡号，下次使用的时候校验卡号
			if (Common.isNotNull(employee.getBankNo()) && (Common.isEmpty(employee.getIssueStatus()) || employee.getIssueStatus().equals(CommonConstants.ZERO_INT))) {
				// 调用校验服务
				TCheckBankNo checkBankNo = new TCheckBankNo();
				checkBankNo.setName(employee.getEmpName());
				checkBankNo.setBankNo(employee.getBankNo());
				R<CheckBankNoVo> checkListR = HttpDaprUtil.invokeMethodPost(checkProperties.getAppUrl(), checkProperties.getAppId()
						, "/tcheckbankno/inner/checkBankNo", checkBankNo, CheckBankNoVo.class, SecurityConstants.FROM_IN);
				String pre = "姓名与卡号验证：";
				if (checkListR != null && checkListR.getData() != null) {
					CheckBankNoVo vo = checkListR.getData();
					TCheckBankNo check = (null == vo.getRes()) ? null : vo.getRes().getData();
					if (Common.isEmpty(vo)) {
						return pre + SalaryConstants.CHECK_NO_RESPONSE;
					} else if (!CommonConstants.SUCCESS.equals(vo.getRes().getCode())) {
						return pre + vo.getRes().getMsg();
					} else if (check != null && !check.getResult().equals(CommonConstants.ZERO_ONE)) {
						return pre + check.getRemark();
					}
				} else {
					return pre + SalaryConstants.CHECK_NO_RESPONSE;
				}
			}
			// 手机号
			if (Common.isNotNull(employee.getEmpPhone())) {
				// 调用校验服务-校验手机号
				TCheckMobile checkMobile = new TCheckMobile();
				checkMobile.setMobile(employee.getEmpPhone());
				R<TCheckMobile> a = HttpDaprUtil.invokeMethodPost(checkProperties.getAppUrl(), checkProperties.getAppId()
						, "/tcheckmobile/inner/checkOneMobile", checkMobile, TCheckMobile.class, SecurityConstants.FROM_IN);
				String pre = "手机号验证：";
				if (a != null && a.getData() != null) {
					checkMobile = a.getData();
					if (checkMobile != null && Common.isNotNull(checkMobile.getStatus())) {
						if (!CommonConstants.ONE_STRING.equals(checkMobile.getStatus())) {
							return pre + checkMobile.getMessage();
						}
					} else {
						return pre + "校验服务未找到手机号，请联系管理员！";
					}
				}
			}
		}
		return null;
	}

	@Override
	public R<List<ErrorMessage>> batchUpdateSalaryEmployee(InputStream inputStream) {
		List<ErrorMessage> errorMessageList = new ArrayList<>();
		ExcelUtil<TSalaryEmployeeUpdateVo> util1 = new ExcelUtil<>(TSalaryEmployeeUpdateVo.class);
		// 写法2：
		// 匿名内部类 不用额外写一个DemoDataListener
		// 这里 需要指定读用哪个class去读，然后读取第一个sheet 文件流会自动关闭
		try {

			EasyExcelFactory.read(inputStream, TSalaryEmployeeUpdateVo.class, new ReadListener<TSalaryEmployeeUpdateVo>() {
				/**
				 * 单次缓存的数据量
				 */
				public static final int BATCH_COUNT = CommonConstants.BATCH_COUNT;
				/**
				 *临时存储
				 */
				private List<TSalaryEmployeeUpdateVo> cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);

				@Override
				public void invoke(TSalaryEmployeeUpdateVo data, AnalysisContext context) {
					ReadRowHolder readRowHolder = context.readRowHolder();
					Integer rowIndex = readRowHolder.getRowIndex();
					data.setRowIndex(rowIndex + 1);
					ErrorMessage errorMessage = util1.checkEntity(data, data.getRowIndex());
					if (Common.isNotNull(errorMessage)) {
						errorMessageList.add(errorMessage);
					} else {
						cachedDataList.add(data);
					}
					if (cachedDataList.size() >= BATCH_COUNT) {
						saveData();
						// 存储完成清理 list
						cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);
					}
				}

				@Override
				public void doAfterAllAnalysed(AnalysisContext context) {
					saveData();
				}

				/**
				 * 加上存储数据库
				 */
				private void saveData() {
					log.info("{}条数据，开始存储数据库！", cachedDataList.size());
					if (!cachedDataList.isEmpty()) {
						updateTSalaryEmployee(cachedDataList, errorMessageList);
					}
					log.info("存储数据库成功！");
				}
			}).sheet().doRead();
		} catch (Exception e) {
			log.error(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR, e);
			return R.failed(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR);
		}
		boolean isTrue = true;
		for (ErrorMessage message : errorMessageList) {
			if (!CommonConstants.SAVE_SUCCESS.equals(message.getMessage())) {
				isTrue = false;
				break;
			}
		}
		if (isTrue) {
			return R.ok();
		} else {
			return R.ok(errorMessageList);
		}
	}

	/**
	 * @Description: 批量更新
	 * @Author: hgw
	 * @Date: 2022/10/10 17:56
	 * @return: void
	 **/
	public void updateTSalaryEmployee(List<TSalaryEmployeeUpdateVo> excelVOList, List<ErrorMessage> errorMessageList) {
		/// 个性化校验逻辑
		TSalaryEmployeeUpdateVo excel;
		List<String> idCardList = new ArrayList<>();
		// 执行数据插入操作 组装
		for (TSalaryEmployeeUpdateVo vo : excelVOList) {
			idCardList.add(vo.getEmpIdcard());
		}
		List<TSalaryEmployee> empList = baseMapper.getListByIdCard(idCardList);
		// 人员Map
		Map<String, TSalaryEmployee> empMap = new HashMap<>();
		for (TSalaryEmployee e : empList) {
			empMap.put(e.getEmpIdcard(), e);
		}
		TSalaryEmployee emp;
		// 开户行省市的区域暂存
		String areaStr;
		boolean curTaxMonth;
		Map<String, String> bankMap = tBankSetService.getBankMap(null);
		for (int i = 0; i < excelVOList.size(); i++) {
			excel = excelVOList.get(i);
			if (excel != null) {
				if (Common.isNotNull(excel.getEmpIdcard())) {
					emp = empMap.get(excel.getEmpIdcard());
					if (Common.isNotNull(emp)) {
						curTaxMonth = false;
						if (Common.isNotNull(emp.getTaxMonth()) && Common.isNotNull(excel.getTaxMonth()) && emp.getTaxMonth().length()==6) {
							curTaxMonth = true;
						} else if (Common.isNotNull(excel.getTaxMonth())) {
							emp.setTaxMonth(excel.getTaxMonth());
						}
						if (Common.isNotNull(excel.getBankName())) {
							if (bankMap.get(emp.getBankName()) == null) {
								errorMessageList.add(new ErrorMessage(excel.getRowIndex(), SalaryConstants.BANK_NAME_ERROR));
								continue;
							}
							emp.setBankName(excel.getBankName());
						}
						if (Common.isNotNull(excel.getBankSubName())) {
							emp.setBankSubName(excel.getBankSubName());
						}
						if (Common.isNotNull(excel.getBankProvince())) {
							areaStr = ExcelUtil.getRedisAreaValue(CacheConstants.AREA_VALUE + excel.getBankProvince().trim());
							if (Common.isNotNull(areaStr)) {
								emp.setBankProvince(areaStr);
								if (Common.isNotNull(excel.getBankCity())) {
									areaStr = ExcelUtil.getRedisAreaValue(CacheConstants.AREA_VALUE + excel.getBankCity().trim()
											+ CommonConstants.DOWN_LINE_STRING + excel.getBankProvince().trim());
									if (Common.isNotNull(areaStr)) {
										emp.setBankCity(areaStr);
									} else {
										errorMessageList.add(new ErrorMessage(excel.getRowIndex(), SalaryConstants.CITY_ERROR));
										continue;
									}
								}
							} else {
								errorMessageList.add(new ErrorMessage(excel.getRowIndex(), SalaryConstants.PROVINCE_ERROR));
								continue;
							}
						}
						if (Common.isNotNull(excel.getBankNo())) {
							emp.setBankNo(excel.getBankNo());
						}
						if (Common.isNotNull(excel.getEmpPhone())) {
							emp.setEmpPhone(excel.getEmpPhone());
						}
						if (Common.isNotNull(excel.getEmpName())) {
							emp.setEmpName(excel.getEmpName());
						}
						baseMapper.updateById(emp);
						if (curTaxMonth) {
							errorMessageList.add(new ErrorMessage(excel.getRowIndex(), SalaryConstants.CUR_TAX_INFO));
						} else {
							errorMessageList.add(new ErrorMessage(excel.getRowIndex(), CommonConstants.SAVE_SUCCESS));
						}
					} else {
						errorMessageList.add(new ErrorMessage(excel.getRowIndex(), SalaryConstants.NOT_HAVE_EMP));
					}
				} else {
					errorMessageList.add(new ErrorMessage(excel.getRowIndex(), SalaryConstants.ID_CARD_MUST));
				}
			} else {
				errorMessageList.add(new ErrorMessage(CommonConstants.ZERO_INT, SalaryConstants.DATA_MUST));
			}
		}
	}

	@Override
	public TSalaryEmployee getByEmpIdCard(String empIdCard) {
		return baseMapper.getByEmpIdCard(empIdCard);
	}

	@Override
	public Map<String, TSalaryEmployee> getByEmpIdCardList(List<String> idCardList) {
		List<TSalaryEmployee> empList = baseMapper.getListByIdCard(idCardList);
		Map<String, TSalaryEmployee> map = new HashMap<>();
		if (empList != null && !empList.isEmpty()) {
			for (TSalaryEmployee e : empList) {
				map.put(e.getEmpIdcard(), e);
			}
		}
		return map;
	}

	/**
	 * @param employee
	 * @Description: 代发户的，下次校验银行卡，同时变更校验flag
	 * @Author: hgw
	 * @Date: 2023/1/10 18:10
	 * @return: boolean ： true：已验证通过；false：银行卡校验失败
	 **/
	@Override
	public boolean checkBankInfo(TSalaryEmployee employee) {
		// 调用校验服务
		TCheckBankNo checkIdCard = new TCheckBankNo();
		checkIdCard.setName(employee.getEmpName());
		checkIdCard.setBankNo(employee.getBankNo());
		R<CheckBankNoVo> checkListR = HttpDaprUtil.invokeMethodPost(checkProperties.getAppUrl(), checkProperties.getAppId()
				, "/tcheckbankno/inner/checkBankNo", checkIdCard, CheckBankNoVo.class, SecurityConstants.FROM_IN);
		if (checkListR != null && checkListR.getData() != null) {
			CheckBankNoVo vo = checkListR.getData();
			TCheckBankNo check = (null == vo.getRes()) ? null : vo.getRes().getData();
			if (check != null && check.getResult().equals(CommonConstants.ZERO_ONE)) {
				employee.setIssueStatus(CommonConstants.ZERO_INT);
				this.updateById(employee);
				return true;
			} else {
				return false;
			}
		} else {
			return false;
		}
	}

	@Override
	public boolean checkBankInfoList(List<TSalaryEmployee> empList, List<ErrorMessage> errorList) {
		// 批量校验 卡号与手机号
		Map<String, Boolean> bankMap = new HashMap<>();
		List<TCheckBankNo> bankList = new ArrayList<>();
		TCheckBankNo checkBankNo;
		for (TSalaryEmployee employee : empList) {
			if (Common.isNotNull(employee.getBankNo())) {
				checkBankNo = new TCheckBankNo();
				checkBankNo.setName(employee.getEmpName());
				checkBankNo.setBankNo(employee.getBankNo());
				bankList.add(checkBankNo);
			}
		}
		if (!bankList.isEmpty()) {
			R<CheckBatchVo> checkListR = HttpDaprUtil.invokeMethodPost(checkProperties.getAppUrl(), checkProperties.getAppId()
					, "/tcheckbankno/inner/checkBankNoBatch", bankList, CheckBatchVo.class, SecurityConstants.FROM_IN);
			if (checkListR != null && checkListR.getData() != null) {
				bankMap = checkListR.getData().getCheckMap();
			}
			for (TSalaryEmployee employee : empList) {
				employee.setIssueStatus(CommonConstants.ZERO_INT);
				if (bankMap.get(employee.getBankNo()) != null && Boolean.FALSE.equals(bankMap.get(employee.getBankNo()))) {
					errorList.add(new ErrorMessage((employee.getLineNums() + 2), "第" + (employee.getLineNums() + 2)
							+ "行：-上次代发户未校验【姓名与卡号】，本次校验结果：不匹配。请去薪酬人员查询处更新卡号信息！"));
					return false;
				}
			}
			this.saveBatch(empList);
		}
		return true;
	}

	/**
	 * @param vo
	 * @Description: 批量更新-薪资人员信息
	 * @Author: hgw
	 * @Date: 2022/10/10 17:17
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage>>
	 **/
	@Override
	public R<List<ErrorMessage>> importEmployee(SalaryEmployeeImportVo vo) {
		List<ErrorMessage> errorMessageList = new ArrayList<>();
		List<String> idCardList = new ArrayList<>();
		// 执行数据插入操作 组装
		List<TSalaryEmployee> excelVOList = vo.getEmpList();
		for (TSalaryEmployee emp : excelVOList) {
			idCardList.add(emp.getEmpIdcard());
		}
		List<TSalaryEmployee> empList = baseMapper.getListByIdCard(idCardList);
		// 人员Map
		Map<String, TSalaryEmployee> empMap = new HashMap<>();
		for (TSalaryEmployee e : empList) {
			empMap.put(e.getEmpIdcard(), e);
		}
		TSalaryEmployee emp;
		// 开户行省市的区域暂存
		String areaProvince;
		String areaCity;
		boolean curTaxMonth;
		TSalaryEmployee excel;
		// 是否可更新
		boolean canUpdate = Common.isNotNull(vo.getIsUpdate()) && CommonConstants.ONE_STRING.equals(vo.getIsUpdate());
		List<TSalaryEmployee> saveList = new ArrayList<>();
		List<TSalaryEmployee> updateList = new ArrayList<>();
		TSettleDomain dept = vo.getDept();
		Map<String, String> bankMap = tBankSetService.getBankMap(null);
		for (int i = 0; i < excelVOList.size(); i++) {
			excel = excelVOList.get(i);
			if (excel != null) {
				if (Common.isNotNull(excel.getEmpIdcard())) {
					emp = empMap.get(excel.getEmpIdcard());
					if (Common.isNotNull(emp)) {
						if (canUpdate) {
							curTaxMonth = false;
							if (Common.isNotNull(emp.getTaxMonth())) {
								curTaxMonth = true;
							} else if (Common.isNotNull(excel.getTaxMonth())) {
								if (excel.getTaxMonth().length() == 6) {
									try {
										Integer.parseInt(excel.getTaxMonth());
									} catch (NumberFormatException e) {
										errorMessageList.add(new ErrorMessage((i + 2), "新增员工，计税月份错误"));
										continue;
									}
									emp.setTaxMonth(excel.getTaxMonth());
								} else {
									errorMessageList.add(new ErrorMessage((i + 2), SalaryConstants.TAX_MONTH_LENGTH));
									continue;
								}
							}
							if (Common.isNotNull(excel.getBankName())) {
								if (bankMap.get(emp.getBankName()) == null) {
									errorMessageList.add(new ErrorMessage((i + 2), SalaryConstants.BANK_NAME_ERROR));
									continue;
								}
								emp.setBankName(excel.getBankName());
							}
							if (Common.isNotNull(excel.getBankSubName())) {
								emp.setBankSubName(excel.getBankSubName());
							}
							if (Common.isNotNull(excel.getBankProvince())) {
								areaProvince = ExcelUtil.getRedisAreaValue(CacheConstants.AREA_VALUE + excel.getBankProvince().trim());
								if (Common.isNotNull(areaProvince)) {
									emp.setBankProvince(areaProvince);
									if (Common.isNotNull(excel.getBankCity())) {
										areaCity = ExcelUtil.getRedisAreaValue(CacheConstants.AREA_VALUE + excel.getBankCity().trim()
												+ CommonConstants.DOWN_LINE_STRING + excel.getBankProvince().trim());
										if (Common.isNotNull(areaCity)) {
											emp.setBankCity(areaCity);
										} else {
											errorMessageList.add(new ErrorMessage((i + 2), SalaryConstants.CITY_ERROR));
											continue;
										}
									}
								} else {
									errorMessageList.add(new ErrorMessage((i + 2), SalaryConstants.PROVINCE_ERROR));
									continue;
								}
							}
							if (Common.isNotNull(excel.getBankNo())) {
								emp.setBankNo(excel.getBankNo());
							}
							if (Common.isNotNull(excel.getEmpPhone())) {
								emp.setEmpPhone(excel.getEmpPhone());
							}
							if (Common.isNotNull(excel.getEmpName())) {
								emp.setEmpName(excel.getEmpName());
							}
							emp.setUnitId(dept.getCustomerId());
							emp.setUnitName(dept.getCustomerName());
							emp.setUnitNo(dept.getCustomerNo());
							emp.setDeptId(dept.getId());
							emp.setDeptName(dept.getDepartName());
							emp.setDeptNo(dept.getDepartNo());
							emp.setInvoiceTitle(dept.getInvoiceTitleSalary());
							updateList.add(emp);
							if (curTaxMonth) {
								errorMessageList.add(new ErrorMessage((i + 2), SalaryConstants.CUR_TAX_INFO));
							} else {
								errorMessageList.add(new ErrorMessage((i + 2), CommonConstants.SAVE_SUCCESS));
							}
						} else {
							errorMessageList.add(new ErrorMessage((i + 2), "已存在，不可更新"));
						}
					} else {
						// 新增人员
						if (Common.isEmpty(excel.getEmpName()) || Common.isEmpty(excel.getEmpIdcard())
								|| Common.isEmpty(excel.getEmpPhone()) || Common.isEmpty(excel.getBankName())
								|| Common.isEmpty(excel.getBankNo()) || Common.isEmpty(excel.getBankProvince())
								|| Common.isEmpty(excel.getBankCity()) || Common.isEmpty(excel.getTaxMonth())) {
							errorMessageList.add(new ErrorMessage((i + 2), "新增员工，除了支行，其他必填"));
							continue;
						} else {
							if (excel.getTaxMonth().length() == 6) {
								try {
									Integer.parseInt(excel.getTaxMonth());
								} catch (NumberFormatException e) {
									errorMessageList.add(new ErrorMessage((i + 2), "新增员工，计税月份错误"));
									continue;
								}
							} else {
								errorMessageList.add(new ErrorMessage((i + 2), SalaryConstants.TAX_MONTH_LENGTH));
								continue;
							}
							String pre = this.checkNewEmpBankAndPhone(excel, true);
							if (pre != null) {
								errorMessageList.add(new ErrorMessage((i + 2), pre));
								continue;
							} else {
								areaProvince = ExcelUtil.getRedisAreaValue(CacheConstants.AREA_VALUE + excel.getBankProvince().trim());
								if (Common.isNotNull(areaProvince)) {
									areaCity = ExcelUtil.getRedisAreaValue(CacheConstants.AREA_VALUE + excel.getBankCity().trim()
											+ CommonConstants.DOWN_LINE_STRING + excel.getBankProvince().trim());
									excel.setBankProvince(areaProvince);
									if (Common.isNotNull(areaCity)) {
										excel.setBankCity(areaCity);
									} else {
										errorMessageList.add(new ErrorMessage((i + 2), SalaryConstants.CITY_ERROR));
										continue;
									}
								} else {
									errorMessageList.add(new ErrorMessage((i + 2), SalaryConstants.PROVINCE_ERROR));
									continue;
								}
								excel.setUnitId(dept.getCustomerId());
								excel.setUnitName(dept.getCustomerName());
								excel.setUnitNo(dept.getCustomerNo());
								excel.setDeptId(dept.getId());
								excel.setDeptName(dept.getDepartName());
								excel.setDeptNo(dept.getDepartNo());
								excel.setInvoiceTitle(dept.getInvoiceTitleSalary());
								saveList.add(excel);
								errorMessageList.add(new ErrorMessage((i + 2), CommonConstants.SAVE_SUCCESS));
							}
						}
					}
				} else {
					errorMessageList.add(new ErrorMessage((i+2), SalaryConstants.ID_CARD_MUST));
				}
			} else {
				errorMessageList.add(new ErrorMessage(CommonConstants.ZERO_INT, SalaryConstants.DATA_MUST));
			}
		}
		if (!updateList.isEmpty()) {
			this.updateBatchById(updateList);
		}
		if (!saveList.isEmpty()) {
			this.saveBatch(saveList);
		}
		boolean isTrue = true;
		for (ErrorMessage message : errorMessageList) {
			if (!CommonConstants.SAVE_SUCCESS.equals(message.getMessage())) {
				isTrue = false;
				break;
			}
		}
		if (isTrue) {
			return R.ok();
		} else {
			return R.ok(errorMessageList);
		}
	}

}
