/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.social.entity.SysHouseHoldInfo;
import com.yifu.cloud.plus.v1.yifu.social.service.SysHouseHoldInfoService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;


/**
 * 社保户公积金户数据
 *
 * @author fxj
 * @date 2022-07-12 08:53:19
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/syshouseholdinfo" )
@Tag(name = "社保户公积金户数据管理")
public class SysHouseHoldInfoController {

    private final SysHouseHoldInfoService sysHouseHoldInfoService;

    /**
     * 简单分页查询
     * @param page 分页对象
     * @param sysHouseHoldInfo 社保户公积金户数据
     * @return
     */
	@Operation(summary = "简单分页查询", description = "简单分页查询")
    @GetMapping("/page")
    public R<IPage<SysHouseHoldInfo>> getSysHouseHoldInfoPage(Page<SysHouseHoldInfo> page, SysHouseHoldInfo sysHouseHoldInfo) {
        return new R<>(sysHouseHoldInfoService.getSysHouseHoldInfoPage(page,sysHouseHoldInfo));
    }

    /**
     * 不分页查询
     * @param sysHouseHoldInfo 社保户公积金户数据
     * @return
     */
    @Operation(summary = "不分页查询", description = "不分页查询")
    @PostMapping("/noPage" )
    public R<List<SysHouseHoldInfo>> getSysHouseHoldInfoNoPage(@RequestBody SysHouseHoldInfo sysHouseHoldInfo) {
        return R.ok(sysHouseHoldInfoService.list(Wrappers.query(sysHouseHoldInfo)));
    }

    /**
     * 通过id查询社保户公积金户数据
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id查询", description = "通过id查询：hasPermission('demo_syshouseholdinfo_get')")
    @GetMapping("/{id}" )
    public R<SysHouseHoldInfo> getById(@PathVariable("id" ) String id) {
        return R.ok(sysHouseHoldInfoService.getById(id));
    }

    /**
     * 新增社保户公积金户数据
     * @param sysHouseHoldInfo 社保户公积金户数据
     * @return R
     */
    @Operation(summary = "新增社保户公积金户数据", description = "新增社保户公积金户数据：hasPermission('demo_syshouseholdinfo_add')")
    @SysLog("新增社保户公积金户数据" )
    @PostMapping
    @PreAuthorize("@pms.hasPermission('demo_syshouseholdinfo_add')" )
    public R<Boolean> save(@RequestBody SysHouseHoldInfo sysHouseHoldInfo) {
        return sysHouseHoldInfoService.saveInfo(sysHouseHoldInfo);
    }

    /**
     * 修改社保户公积金户数据
     * @param sysHouseHoldInfo 社保户公积金户数据
     * @return R
     */
    @Operation(summary = "修改社保户公积金户数据", description = "修改社保户公积金户数据：hasPermission('demo_syshouseholdinfo_edit')")
    @SysLog("修改社保户公积金户数据" )
    @PutMapping
    @PreAuthorize("@pms.hasPermission('demo_syshouseholdinfo_edit')" )
    public R<Boolean> updateById(@RequestBody SysHouseHoldInfo sysHouseHoldInfo) {
        return sysHouseHoldInfoService.updateByIdAsso(sysHouseHoldInfo);
    }

    /**
     * 通过id删除社保户公积金户数据
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id删除社保户公积金户数据", description = "通过id删除社保户公积金户数据：hasPermission('demo_syshouseholdinfo_del')")
    @SysLog("通过id删除社保户公积金户数据" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_syshouseholdinfo_del')" )
    public R<Boolean> removeById(@PathVariable String id) {
        return R.ok(sysHouseHoldInfoService.removeById(id));
    }

	/**
	 * @param flag 0/1
	 * @param name 户名
	 * @return 上级部门用户列表
	 */
	@PostMapping("/getHouseList")
	@Operation(summary = "获取所有的户名", description = "获取所有的户名")
	public R<List<SysHouseHoldInfo>> getSysUserList(@RequestParam String flag,
										   @RequestParam(required = false) String name) {
		List<SysHouseHoldInfo> houseHoldList;
		LambdaQueryWrapper<SysHouseHoldInfo> wrapper = Wrappers.lambdaQuery();
		wrapper.eq(SysHouseHoldInfo::getDelFlag, CommonConstants.ZERO_STRING);
		if (CommonConstants.ZERO_STRING.equals(flag)) {
			wrapper.eq(SysHouseHoldInfo::getType, CommonConstants.ZERO_STRING);
			if (Common.isNotNull(name)) {
				wrapper.like(SysHouseHoldInfo::getName, name);
			}
		} else {
			wrapper.eq(SysHouseHoldInfo::getType, CommonConstants.ONE_STRING);
			if (Common.isNotNull(name)) {
				wrapper.like(SysHouseHoldInfo::getName, name);
			}
		}
		houseHoldList = sysHouseHoldInfoService.list(wrapper);
		return R.ok(houseHoldList);
	}

}
