package com.yifu.cloud.plus.v1.yifu.social.controller;/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CacheConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.ResultConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.redis.RedisDistributedLock;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.DateUtil;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.ErrorDetailVO;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.MenuUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import com.yifu.cloud.plus.v1.yifu.insurances.vo.EkpSocialViewVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.HaveSalaryNoSocialSearchVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TPaymentBySalaryVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TPaymentVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.UpdateSocialFoundVo;
import com.yifu.cloud.plus.v1.yifu.social.entity.TPaymentInfo;
import com.yifu.cloud.plus.v1.yifu.social.service.TPaymentInfoService;
import com.yifu.cloud.plus.v1.yifu.social.vo.ChangeDeptVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.TPaymentInfoNewVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.TPaymentInfoSearchVo;
import com.yifu.cloud.plus.v1.yifu.social.vo.TPaymentInfoVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 缴费库
 * @author fxj
 * @date 2022-07-22 17:01:22
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tpaymentinfo" )
@Tag(name = "缴费库管理")
public class TPaymentInfoController {

	private final TPaymentInfoService tPaymentInfoService;

	private final MenuUtil menuUtil;

	/**
	 * 简单分页查询
	 *
	 * @param page         分页对象
	 * @param tPaymentInfo 缴费库
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TPaymentInfo>> getTPaymentInfoPage(Page<TPaymentInfo> page, TPaymentInfoSearchVo tPaymentInfo) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		//特殊条件如社保户、缴纳地的查询做限制处理，限制到所用用户，因为此部分数据量较大
		R<IPage<TPaymentInfo>> res = tPaymentInfoService.specialHandle(page,tPaymentInfo);
		if (null != res){
			return res;
		}
		//查询数据
		return new R<>(tPaymentInfoService.getTPaymentInfoPage(page, tPaymentInfo));
	}

	/**
	 * 简单分页查询
	 *
	 * @param page         分页对象
	 * @param tPaymentInfo 缴费库
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/newPage")
	public R<IPage<TPaymentInfoNewVo>> getTPaymentInfoNewPage(Page<TPaymentInfo> page, TPaymentInfoSearchVo tPaymentInfo) {
		//特殊条件如社保户、缴纳地的查询做限制处理，限制到所用用户，因为此部分数据量较大
		R<IPage<TPaymentInfoNewVo>> res = tPaymentInfoService.specialHandleNew(page,tPaymentInfo);
		if (null != res){
			return res;
		}
		return  tPaymentInfoService.getTPaymentInfoNewPage(page, tPaymentInfo);
	}
	/**
	 * 实缴查询详情
	 *
	 * @param tPaymentInfo
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/newPageInfo")
	public R<TPaymentInfoNewVo> getTPaymentInfoNewPageInfo(TPaymentInfoSearchVo tPaymentInfo) {
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, tPaymentInfo);
		if (Common.isNotNull(tPaymentInfo.getAuthSql())) {
			if (tPaymentInfo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
				tPaymentInfo.setAuthSql(tPaymentInfo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "a.SETTLE_DOMAIN_ID"));
			}
			if (tPaymentInfo.getAuthSql().contains("1=2 EXISTS")) {
				tPaymentInfo.setAuthSql(tPaymentInfo.getAuthSql().replace("1=2 EXISTS", "EXISTS"));
			}
		}
		return R.ok(tPaymentInfoService.getTPaymentInfoNewPageInfo(tPaymentInfo));
	}

	/**
	 * 通过id查询缴费库
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询")
	@GetMapping("/{id}")
	public R<TPaymentInfo> getById(@PathVariable("id") String id) {
		return R.ok(tPaymentInfoService.getById(id));
	}

	/**
	 * 修改缴费库
	 *
	 * @param tPaymentInfo 缴费库
	 * @return R
	 */
	@Operation(summary = "修改缴费库", description = "修改缴费库：hasPermission('social_tpaymentinfo_edit')")
	@SysLog("修改缴费库")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_edit')")
	public R<Boolean> updateById(@RequestBody TPaymentInfo tPaymentInfo) {
		return R.ok(tPaymentInfoService.updateById(tPaymentInfo));
	}

	/**
	 * 通过id删除缴费库
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除缴费库", description = "通过id删除缴费库：hasPermission('social_tpaymentinfo_del')")
	@SysLog("通过id删除缴费库")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return tPaymentInfoService.removeInFoById(id);
	}

	/**
	 * 通过ids删除一条记录
	 *
	 * @param ids
	 * @return R
	 */
	@Operation(summary = "通过ids删除缴费库", description = "通过ids删除缴费库")
	@SysLog("删除缴费库")
	@DeleteMapping("/removeBatchByIds")
	public R<Boolean> removeBatchByIds(String ids) {
		return tPaymentInfoService.removeAllInfoById(ids);
	}

	/**
	 * 通过id查询单条记录(包含明细)
	 *
	 * @param id
	 * @return R
	 */
	@Operation(summary = "通过id查询单条记录")
	@GetMapping("/allInfo/{id}")
	public R<TPaymentInfoVo> getAllInfoById(@PathVariable("id") String id) {
		return tPaymentInfoService.getAllInfoById(id);
	}

	/**
	 * 省市社保 批量导入
	 *
	 * @author huyc
	 * @date 2022-07-22
	 **/
	@SneakyThrows
	@Operation(description = "导入省市社保 hasPermission('social_tpaymentinfo_batchImport')")
	@SysLog("批量新增缴费库")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_batchImport')")
	public R<List<ErrorDetailVO>> importListAdd(@RequestBody MultipartFile file, @RequestParam String random) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		// 获取redis分布式事务锁
		String key = CacheConstants.PAYMENT_SOCIAL_IMPORT + CommonConstants.DOWN_LINE_STRING + user.getId();
		String requestId;
		try {
			requestId = RedisDistributedLock.getLock(key,"10");
		} catch (Exception e) {
			throw new RuntimeException(ResultConstants.NO_GETLOCK_DATA+CommonConstants.DOWN_LINE_STRING+e.getMessage());
		}
		try {
			if (Common.isNotNull(requestId)) {
				return tPaymentInfoService.importSocialDiy(file.getInputStream());
			} else {
				return R.failed(ResultConstants.NO_GETLOCK_DATA);
			}
		}finally {
			//主动释放锁
			RedisDistributedLock.unlock(key, requestId);
		}
	}

	/**
	 * 导入省市社保-合肥三险
	 *
	 * @author huyc
	 * @date 2022-07-22
	 **/
	@SneakyThrows
	@Operation(description = "导入省市社保-合肥三险")
	@SysLog("批量新增缴费库-合肥三险")
	@PostMapping("/importListSocialHeFei")
	public R<List<ErrorDetailVO>> importListSocialHeFei(@RequestBody MultipartFile file, @RequestParam String random,
													   @RequestParam(value = "type") String type) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		// 获取redis分布式事务锁
		String key = CacheConstants.PAYMENT_SOCIAL_THREE_IMPORT + CommonConstants.DOWN_LINE_STRING + user.getId();
		String requestId;
		try {
			requestId = RedisDistributedLock.getLock(key,"10");
		} catch (Exception e) {
			throw new RuntimeException(ResultConstants.NO_GETLOCK_DATA+CommonConstants.DOWN_LINE_STRING+e.getMessage());
		}
		try {
			if (Common.isNotNull(requestId)) {
				return tPaymentInfoService.importSocialHeFeiDiy(file.getInputStream(), random, type);
			} else {
				return R.failed(ResultConstants.NO_GETLOCK_DATA);
			}
		}finally {
			//主动释放锁
			RedisDistributedLock.unlock(key, requestId);
		}
	}

	/**
	 * 导入公积金
	 *
	 * @author huyc
	 * @date 2022-07-27
	 **/
	@SneakyThrows
	@Operation(description = "导入公积金 hasPermission('social_tpaymentinfo_batchImport')")
	@SysLog("导入公积金")
	@PostMapping("/batchImportPaymentFundInfo")
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_batchImport')")
	public R<List<ErrorDetailVO>> importListFund(@RequestBody MultipartFile file) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		// 获取redis分布式事务锁
		String key = CacheConstants.PAYMENT_FUND_IMPORT + CommonConstants.DOWN_LINE_STRING + user.getId();
		String requestId;
		try {
			requestId = RedisDistributedLock.getLock(key,"10");
		} catch (Exception e) {
			throw new RuntimeException(ResultConstants.NO_GETLOCK_DATA+CommonConstants.DOWN_LINE_STRING+e.getMessage());
		}
		try {
			if (Common.isNotNull(requestId)) {
				return tPaymentInfoService.batchImportPaymentFundInfo(file.getInputStream(),0);
			} else {
				return R.failed(ResultConstants.NO_GETLOCK_DATA);
			}
		}finally {
			//主动释放锁
			RedisDistributedLock.unlock(key, requestId);
		}
	}
	/**
	 * 导入公积金-多比例
	 *
	 * @author huyc
	 * @date 2022-07-27
	 **/
	@SneakyThrows
	@Operation(description = "导入公积金-多比例 hasPermission('social_tpaymentinfo_batchImport')")
	@SysLog("导入公积金-多比例")
	@PostMapping("/batchImportPaymentFundInfoNew")
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_batchImport')")
	public R<List<ErrorDetailVO>> importListFundNew(@RequestBody MultipartFile file) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		// 获取redis分布式事务锁
		String key = CacheConstants.PAYMENT_FUND_IMPORT + CommonConstants.DOWN_LINE_STRING + user.getId();
		String requestId;
		try {
			requestId = RedisDistributedLock.getLock(key,"10");
		} catch (Exception e) {
			throw new RuntimeException(ResultConstants.NO_GETLOCK_DATA+CommonConstants.DOWN_LINE_STRING+e.getMessage());
		}
		try {
			if (Common.isNotNull(requestId)) {
				return tPaymentInfoService.batchImportPaymentFundInfo(file.getInputStream(),1);
			} else {
				return R.failed(ResultConstants.NO_GETLOCK_DATA);
			}
		}finally {
			//主动释放锁
			RedisDistributedLock.unlock(key, requestId);
		}
	}
	/**
	 * 缴费库 批量导出
	 *
	 * @author huyc
	 * @date 2022-07-22
	 **/
	@SysLog("实缴导出列表不合并导出")
	@Operation(description = "导出缴费库 hasPermission('social_tpaymentinfo_export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_export')")
	public void export(HttpServletResponse response, @RequestBody TPaymentInfoSearchVo searchVo) {
		tPaymentInfoService.exportAsso(response, searchVo, CommonConstants.ZERO_INT);
	}

	/**
	 * @Author fxj
	 * @Description 实缴导入列表导出不合并数据校验，生成月份为空时 数据量超出2W 前端提示
	 * @Date 14:57 2024/8/5
	 * @Param
	 * @return
	**/
	@SysLog("实缴导出列表不合并导出--检测")
	@Operation(description = "导出缴费库检测 ")
	@PostMapping("/exportCheck")
	public R<Integer> exportCheck(HttpServletResponse response, @RequestBody TPaymentInfoSearchVo searchVo) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		// 获取redis分布式事务锁
		String key = CacheConstants.PAYMENT_EXPORT_REPEAT + CommonConstants.DOWN_LINE_STRING + user.getId();
		String requestId;
		try {
			requestId = RedisDistributedLock.getLock(key,"10");
		} catch (Exception e) {
			throw new RuntimeException(ResultConstants.NO_GETLOCK_DATA+CommonConstants.DOWN_LINE_STRING+e.getMessage());
		}
		try {
			if (Common.isNotNull(requestId)) {
				//查询数据
				return tPaymentInfoService.listExportCheck(response, searchVo);
			} else {
				return R.failed(ResultConstants.NO_GETLOCK_DATA);
			}
		}finally {
			//主动释放锁
			RedisDistributedLock.unlock(key, requestId);
		}

	}

	/**
	 * 缴费库合并导出
	 *
	 * @author huyc
	 * @date 2022-10-31
	 **/
	@SysLog("缴费库合并导出")
	@Operation(description = "合并导出缴费库 hasPermission('social_tpaymentinfo_export')")
	@PostMapping("/sumExport")
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_export')")
	public void sumExport(HttpServletResponse response, @RequestBody TPaymentInfoSearchVo searchVo) {
		tPaymentInfoService.exportAsso(response, searchVo, CommonConstants.ONE_INT);
	}

	/**
	 * @Author fxj
	 * @Description 缴费库合并导出 检车 生成月份为空 数据条数超出2W条后 前端给提示
	 * @Date 15:09 2024/8/5
	 * @Param
	 * @return
	**/
	@SysLog("缴费库合并导出检测")
	@Operation(description = "合并导出缴费库检测")
	@PostMapping("/sumExportCheck")
	public R<Integer> sumExportCheck(HttpServletResponse response, @RequestBody TPaymentInfoSearchVo searchVo) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		// 获取redis分布式事务锁
		String key = CacheConstants.PAYMENT_EXPORT_REPEAT + CommonConstants.DOWN_LINE_STRING+user.getId();
		String requestId;
		try {
			requestId = RedisDistributedLock.getLock(key,"10");
		} catch (Exception e) {
			throw new RuntimeException(ResultConstants.NO_GETLOCK_DATA+CommonConstants.DOWN_LINE_STRING+e.getMessage());
		}
		try {
			if (Common.isNotNull(requestId)) {
				//查询数据
				return tPaymentInfoService.listSumExportCheck(response, searchVo);
			} else {
				return R.failed(ResultConstants.NO_GETLOCK_DATA);
			}
		}finally {
			//主动释放锁
			RedisDistributedLock.unlock(key, requestId);
		}

	}

	/**
	 * 缴费库查询合并导出
	 *
	 * @author huyc
	 * @date 2023-05-19
	 **/
	@Operation(description = "缴费库查询合并导出 hasPermission('social_tpaymentinfo_export')")
	@PostMapping("/sumSearchExport")
	@PreAuthorize("@pms.hasPermission('social_tpaymentinfo_export')")
	@SysLog("实缴查询列表合并导出")
	public void sumSearchExport(HttpServletResponse response, @RequestBody TPaymentInfoSearchVo searchVo) {
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, searchVo);
		if (Common.isNotNull(searchVo.getAuthSql())) {
			if (searchVo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
				searchVo.setAuthSql(searchVo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "a.SETTLE_DOMAIN_ID"));
			}
			if (searchVo.getAuthSql().contains("1=2 EXISTS")) {
				searchVo.setAuthSql(searchVo.getAuthSql().replace("1=2 EXISTS", "EXISTS"));
			}
		}
		tPaymentInfoService.sumSearchExport(response, searchVo);
	}

	/**
	 * @Author fxj
	 * @Description 缴费库查询合并导出 校验， 生成月份为空时 导出条数超出2W 前端给提示
	 * @Date 15:13 2024/8/5
	 * @Param
	 * @return
	**/
	@Operation(description = "缴费库查询合并导出检测")
	@PostMapping("/sumSearchExportCheck")
	@SysLog("实缴查询列表合并导出检测")
	public R<Integer> sumSearchExportCheck(HttpServletResponse response, @RequestBody TPaymentInfoSearchVo searchVo) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		// 获取redis分布式事务锁
		String key = CacheConstants.PAYMENT_EXPORT_REPEAT + CommonConstants.DOWN_LINE_STRING + user.getId();
		String requestId;
		try {
			requestId = RedisDistributedLock.getLock(key,"10");
		} catch (Exception e) {
			throw new RuntimeException(ResultConstants.NO_GETLOCK_DATA+CommonConstants.DOWN_LINE_STRING+e.getMessage());
		}
		try {
			if (Common.isNotNull(requestId)) {
				//查询数据
				menuUtil.setAuthSql(user, searchVo);
				if (Common.isNotNull(searchVo.getAuthSql())) {
					if (searchVo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
						searchVo.setAuthSql(searchVo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "a.SETTLE_DOMAIN_ID"));
					}
					if (searchVo.getAuthSql().contains("1=2 EXISTS")) {
						searchVo.setAuthSql(searchVo.getAuthSql().replace("1=2 EXISTS", "EXISTS"));
					}
				}
				return tPaymentInfoService.sumSearchExportCheck(response, searchVo);
			} else {
				return R.failed(ResultConstants.NO_GETLOCK_DATA);
			}
		}finally {
			//主动释放锁
			RedisDistributedLock.unlock(key, requestId);
		}

	}

	/**
	 * 根据条件批量删除
	 *
	 * @param empIdCard         员工身份证
	 * @param socialHouseId     社保户
	 * @param fundHouseId       公积金户
	 * @param socialCreateMonth 生成月份
	 * @param socialPayMonth    缴纳月份
	 * @return R
	 */
	@Operation(description = "根据条件批量删除")
	@SysLog("根据条件批量删除")
	@PostMapping("/removeBatchByIdsAndType")
	public R<Boolean> removeBatchByIdsAndType(@RequestParam(name = "empIdCard", required = false) String empIdCard,
									 @RequestParam(name = "socialHouseId", required = false) String socialHouseId,
									 @RequestParam(name = "fundHouseId", required = false) String fundHouseId,
									 @RequestParam(name = "socialCreateMonth", required = false) String socialCreateMonth,
									 @RequestParam(name = "socialPayMonth", required = false) String socialPayMonth) {
		return tPaymentInfoService.removeBatchByInfo(empIdCard, socialHouseId,
				fundHouseId, socialCreateMonth, socialPayMonth);
	}

	/**
	 * @Description: 薪资获取缴费库
	 * @Author: hgw
	 * @Date: 2022/8/10 18:06
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.salary.vo.TPaymentVo>
	 **/
	@Inner
	@PostMapping("/inner/selectTPaymentBySalaryVo")
	public TPaymentVo selectTPaymentBySalaryVo(@RequestBody TPaymentBySalaryVo tPaymentBySalaryVo) {
		return tPaymentInfoService.selectTPaymentBySalaryVo(tPaymentBySalaryVo);
	}

	//hgw 以下4个方法，薪资变更缴费库预估库结算状态专用
	/**
	 * @param idList
	 * @Description: 社保变为结算
	 * @Author: hgw
	 * @Date: 2022/8/16 18:22
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updatePaymentSocialStatusToSettle")
	public int updatePaymentSocialStatusToSettle(@RequestBody List<String> idList) {
		return tPaymentInfoService.updatePaymentSocialStatusToSettle(idList);
	}
	/**
	 * @param idList
	 * @Description: 社保变为结算
	 * @Author: hgw
	 * @Date: 2022/8/16 18:22
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updatePaymentSocialStatusToNoSettle")
	public int updatePaymentSocialStatusToNoSettle(@RequestBody List<String> idList) {
		return tPaymentInfoService.updatePaymentSocialStatusToNoSettle(idList);
	}
	/**
	 * @param idList
	 * @Description: 公积金变为结算
	 * @Author: hgw
	 * @Date: 2022/8/16 18:22
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updatePaymentFundStatusToSettle")
	public int updatePaymentFundStatusToSettle(@RequestBody List<String> idList) {
		return tPaymentInfoService.updatePaymentFundStatusToSettle(idList);
	}
	/**
	 * @param idList
	 * @Description: 公积金变为未结算
	 * @Author: hgw
	 * @Date: 2022/8/16 18:22
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updatePaymentFundStatusToNoSettle")
	public int updatePaymentFundStatusToNoSettle(@RequestBody List<String> idList) {
		return tPaymentInfoService.updatePaymentFundStatusToNoSettle(idList);
	}

	/**
	 * @Author fxj
	 * @Description 有工资无社保提醒：通过身份证号和月份获取对应的缴费库数据
	 * @Date 17:45 2022/8/16
	 * @Param
	 * @return
	**/
	@Inner
	@PostMapping("/inner/listByEmpdIdCard")
	public R<List<TPaymentInfo>> getPaymentinfoListByEmpdIdCard(@RequestBody HaveSalaryNoSocialSearchVo searchVo) {
		return new R<>(tPaymentInfoService.list(Wrappers.<TPaymentInfo>query().lambda()
				.in(TPaymentInfo::getEmpIdcard, searchVo.getIdCards())
				.and(obj->obj.eq(TPaymentInfo::getSocialCreateMonth,searchVo.getSettleMonth())
						.or()
						.eq(TPaymentInfo::getSocialCreateMonth, DateUtil.getYearAndMonth(searchVo.getSettleMonth(),1))
						.or()
						.eq(TPaymentInfo::getSocialCreateMonth,DateUtil.getYearAndMonth(searchVo.getSettleMonth(),1))
				)
				.groupBy(TPaymentInfo::getEmpIdcard)));
	}

	/**
	 * @param list
	 * @Description: 更新项目单位信息
	 * @Author: huyc
	 * @Date: 2022/8/25
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/updatePaymentSocialAndFound")
	public void updatePaymentSocialAndFound(@RequestBody UpdateSocialFoundVo list) {
		tPaymentInfoService.updatePaymentSocialAndFound(list);
	}

	/**
	 * @Description: 定时任务推送社保缴费库明细的数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 * @return: void
	 **/
	@Operation(summary = "定时任务推送社保缴费库明细的数据", description = "定时任务推送社保缴费库明细的数据")
	@SysLog("定时任务推送社保缴费库明细的数据")
	@Inner
	@PostMapping("/inner/createPaymentSocialInfo")
	public void createPaymentSocialInfo() {
		tPaymentInfoService.createPaymentSocialInfo();
	}

	/**
	 * @Description: 定时任务推送公积金缴费库明细的数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 * @return: void
	 **/
	@Operation(summary = "定时任务推送公积金缴费库明细的数据", description = "定时任务推送公积金缴费库明细的数据")
	@SysLog("定时任务推送公积金缴费库明细的数据")
	@Inner
	@PostMapping("/inner/createPaymentFundInfo")
	public void createPaymentFundInfo() {
		tPaymentInfoService.createPaymentFundInfo();
	}

	/**
	 * @Description: 定时生成缴费库的收入数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 * @return: void
	 **/
	@Operation(summary = "定时生成缴费库的收入数据", description = "定时生成缴费库的收入数据")
	@SysLog("定时生成缴费库的收入数据")
	@Inner
	@PostMapping("/inner/createPaymentInfoIncome")
	public void createPaymentInfoIncome() {
		tPaymentInfoService.createPaymentInfoIncome();
	}

	/**
	 * @Description: 定时生成公积金的收入数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 * @return: void
	 **/
	@Operation(summary = "定时生成公积金的收入数据", description = "定时生成公积金的收入数据")
	@SysLog("定时生成公积金的收入数据")
	@Inner
	@PostMapping("/inner/createPaymentFundIncome")
	public void createPaymentFundIncome() {
		tPaymentInfoService.createPaymentFundIncome();
	}

	/**
	 * @Description: 定时任务标识根据项目不需要生成收入的数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 * @return: void
	 **/
	@Operation(summary = "定时任务标识根据项目不需要生成收入的数据", description = "定时任务标识根据项目不需要生成收入的数据")
	@SysLog("定时任务标识根据项目不需要生成收入的数据")
	@Inner
	@PostMapping("/inner/signPaymentNoIncomeFlag")
	public void signPaymentNoIncomeFlag() {
		tPaymentInfoService.signPaymentNoIncomeFlag();
	}

	/**
	 * @param info
	 * @Description: 查询缴费库
	 * @Author: huyc
	 * @Date: 2022/8/25
	 * @return: int
	 **/
	@Inner
	@PostMapping("/inner/getPaymentSocialAndFound")
	public TPaymentVo getPaymentSocialAndFound(@RequestBody TPaymentInfo info) {
		return tPaymentInfoService.getPaymentSocialAndFound(info);
	}

	/**
	 * @Description: 手动推送社保缴费库明细的数据
	 * @Author: huyc
	 * @Date: 2022/8/30
	 * @return: void
	 **/
	@Operation(summary = "手动推送社保缴费库明细的数据", description = "手动推送社保缴费库明细的数据")
	@SysLog("手动推送社保缴费库明细的数据")
	@PostMapping("/pushPaymentSocialFundInfo")
	public R<Boolean> pushPaymentSocialFundInfo(@RequestBody TPaymentInfoSearchVo searchVo) {
		return tPaymentInfoService.pushPaymentSocialFundInfo(searchVo);
	}

	/**
	 * 费用划转，EKP通知MVP跟着一起划转
	 * @author  hgw
	 * @return R
	 */
	@Operation(summary = "费用划转--EKP调用接口", description = "费用划转--EKP调用接口")
	@SysLog("费用划转--EKP调用接口")
	@PostMapping("/changeDeptByEkp")
	public R<String> changeDeptByEkp(@RequestBody ChangeDeptVo changeDeptVo) {
		return tPaymentInfoService.changeDeptByEkp(changeDeptVo);
	}

	/**
	 * @Description: 批量标识可以推送的数据
	 * @Author: huyc
	 * @Date: 2022/11/16
	 * @return: R
	 **/
	@Operation(summary = "批量标识可以推送的数据", description = "批量标识可以推送的数据")
	@SysLog("批量标识可以推送的数据")
	@PostMapping("/batchUpdatePaymentInfo")
	public R<Boolean> batchUpdatePaymentInfo(@RequestBody TPaymentInfoSearchVo searchVo) {
		return tPaymentInfoService.batchUpdatePaymentInfo(searchVo);
	}

	/**
	 * @param viewVo
	 * @Description: 更新结算信息
	 * @Author: huyc
	 * @Date: 2022/8/25
	 * @return:
	 **/
	@Inner
	@PostMapping("/inner/updateSocialSettleStatus")
	public Boolean updateSocialSettleStatus(@RequestBody List<EkpSocialViewVo> viewVo) {
		return tPaymentInfoService.updateSocialSettleStatus(viewVo);
	}


	/**
	 * 税务社保明细导入
	 *
	 * @author huyc
	 * @date 2024-5-9
	 **/
	@SneakyThrows
	@Operation(description = "税务社保明细导入")
	@SysLog("税务社保明细导入")
	@PostMapping("/importTaxSocialInfoList")
	public R<List<ErrorMessage>> importTaxSocialInfoList(@RequestBody MultipartFile file) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		// 获取redis分布式事务锁
		String key = CacheConstants.PAYMENT_SOCIAL_THREE_IMPORT + CommonConstants.DOWN_LINE_STRING + user.getId();
		String requestId;
		try {
			requestId = RedisDistributedLock.getLock(key,"10");
		} catch (Exception e) {
			throw new RuntimeException(ResultConstants.NO_GETLOCK_DATA+CommonConstants.DOWN_LINE_STRING+e.getMessage());
		}
		try {
			if (Common.isNotNull(requestId)) {
				return tPaymentInfoService.importTaxSocialInfoDiy(file.getInputStream());
			} else {
				return R.failed(ResultConstants.NO_GETLOCK_DATA);
			}
		}finally {
			//主动释放锁
			RedisDistributedLock.unlock(key, requestId);
		}
	}
}
