/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.salary.service.impl;

import cn.hutool.core.util.ArrayUtil;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.DateUtil;
import com.yifu.cloud.plus.v1.yifu.common.mybatis.base.BaseEntity;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TSalaryStandardIssue;
import com.yifu.cloud.plus.v1.yifu.salary.mapper.TSalaryStandardIssueMapper;
import com.yifu.cloud.plus.v1.yifu.salary.service.TSalaryStandardIssueService;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TSalaryStandardIssueExportVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TSalaryStandardIssueSearchVo;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Service;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;

/**
 * 薪资代发户配置
 *
 * @author hgw
 * @date 2023-01-10 11:34:27
 */
@Log4j2
@Service
public class TSalaryStandardIssueServiceImpl extends ServiceImpl<TSalaryStandardIssueMapper, TSalaryStandardIssue> implements TSalaryStandardIssueService {
	/**
	 * 薪资代发户配置简单分页查询
	 *
	 * @param tSalaryStandardIssue 薪资代发户配置
	 * @return
	 */
	@Override
	public IPage<TSalaryStandardIssue> getTSalaryStandardIssuePage(Page<TSalaryStandardIssue> page, TSalaryStandardIssueSearchVo tSalaryStandardIssue) {
		return baseMapper.getTSalaryStandardIssuePage(page, tSalaryStandardIssue);
	}

	/**
	 * 薪资代发户配置批量导出
	 *
	 * @param searchVo 薪资代发户配置
	 * @return
	 */
	@Override
	public void listExport(HttpServletResponse response, TSalaryStandardIssueSearchVo searchVo) {
		String fileName = "代发户配置批量导出" + DateUtil.getThisTime() + ".xlsx";
		//获取要导出的列表
		List<TSalaryStandardIssueExportVo> list = new ArrayList<>();
		long count = noPageCountDiy(searchVo);
		ServletOutputStream out = null;
		try {
			out = response.getOutputStream();
			response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
			response.setCharacterEncoding("utf-8");
			response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName, CommonConstants.UTF8));
			// 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
			ExcelWriter excelWriter = EasyExcelFactory.write(out, TSalaryStandardIssueExportVo.class).build();
			int index = 0;
			if (count > CommonConstants.ZERO_INT) {
				for (int i = 0; i <= count; i = i + CommonConstants.EXCEL_EXPORT_LIMIT) {
					// 获取实际记录
					searchVo.setLimitStart(i);
					searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
					list = exportList(searchVo);
					if (Common.isNotNull(list)) {
						WriteSheet writeSheet = EasyExcelFactory.writerSheet("代发户配置" + index).build();
						excelWriter.write(list, writeSheet);
						index++;
						list.clear();
					}
				}
			} else {
				WriteSheet writeSheet = EasyExcelFactory.writerSheet("代发户配置" + index).build();
				excelWriter.write(list, writeSheet);
			}
			if (Common.isNotNull(list)) {
				list.clear();
			}
			out.flush();
			excelWriter.finish();
		} catch (Exception e) {
			log.error("执行异常", e);
		} finally {
			try {
				if (null != out) {
					out.close();
				}
			} catch (IOException e) {
				log.error("执行异常", e);
			}
		}
	}

	@Override
	public List<TSalaryStandardIssue> noPageDiy(TSalaryStandardIssueSearchVo searchVo) {
		LambdaQueryWrapper<TSalaryStandardIssue> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TSalaryStandardIssue::getId, idList);
		}
		if (searchVo.getLimitStart() >= 0 && searchVo.getLimitEnd() > 0) {
			wrapper.last(" limit " + searchVo.getLimitStart() + "," + searchVo.getLimitEnd());
		}
		wrapper.orderByDesc(BaseEntity::getCreateTime);
		return baseMapper.selectList(wrapper);
	}

	/**
	 * @Description: 导出
	 * @Author: hgw
	 * @Date: 2023/1/18 10:31
	 * @return: java.util.List<com.yifu.cloud.plus.v1.yifu.salary.vo.TSalaryStandardIssueExportVo>
	 **/
	private List<TSalaryStandardIssueExportVo> exportList(TSalaryStandardIssueSearchVo searchVo) {
		return baseMapper.getTSalaryStandardIssueVoList(searchVo);
	}

	private Long noPageCountDiy(TSalaryStandardIssueSearchVo searchVo) {
		LambdaQueryWrapper<TSalaryStandardIssue> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TSalaryStandardIssue::getId, idList);
		}
		return baseMapper.selectCount(wrapper);
	}

	private LambdaQueryWrapper buildQueryWrapper(TSalaryStandardIssueSearchVo entity) {
		LambdaQueryWrapper<TSalaryStandardIssue> wrapper = Wrappers.lambdaQuery(entity);
		if (ArrayUtil.isNotEmpty(entity.getCreateTimes())) {
			wrapper.ge(TSalaryStandardIssue::getCreateTime, entity.getCreateTimes()[0])
					.le(TSalaryStandardIssue::getCreateTime,
							entity.getCreateTimes()[1]);
		}
		if (Common.isNotNull(entity.getCreateName())) {
			wrapper.eq(TSalaryStandardIssue::getCreateName, entity.getCreateName());
		}
		return wrapper;
	}

}
