/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeProject;
import com.yifu.cloud.plus.v1.yifu.archives.service.TEmployeeProjectService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.EmpStatusVo;
import com.yifu.cloud.plus.v1.yifu.archives.vo.EmployeeProjectExportParamVO;
import com.yifu.cloud.plus.v1.yifu.archives.vo.TSettleDomainDataVo;
import com.yifu.cloud.plus.v1.yifu.archives.vo.UpProjectSocialFundVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.MenuUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.util.List;
import java.util.Map;


/**
 * 项目档案表
 *
 * @author huyc
 * @date 2022-06-20 09:19:40
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/temployeeproject" )
@Tag(name = "项目档案表管理")
public class TEmployeeProjectController {

    private final TEmployeeProjectService tEmployeeProjectService;

	private final MenuUtil menuUtil;

	/**
	 * @param empId 人员档案id
	 * @Description: 根据人员档案id获取项目list
	 * @Author: hgw
	 * @Date: 2022/6/21 19:19
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeProject>>
	 **/
	@Operation(summary = "根据人员档案id获取项目list,projectStatus：1或不传，则查全部；0，查在项。", description = "根据人员档案id获取项目list")
	@GetMapping("/getListByEmpId" )
	public R<List<TEmployeeProject>> getListByEmpId(@RequestParam String empId, @RequestParam(required = false) Integer projectStatus) {
		if (projectStatus == null) {
			projectStatus = CommonConstants.ONE_INT;
		}
		return R.ok(tEmployeeProjectService.getListByEmpId(empId, projectStatus));
	}

    /**
     * 分页查询
     * @param page 分页对象
     * @param tEmployeeProject 项目档案表
     * @return
     */
    @Operation(summary = "分页查询", description = "分页查询")
    @GetMapping("/page" )
    public R<IPage<TEmployeeProject>> getTEmployeeProjectPage(Page page, TEmployeeProject tEmployeeProject) {
		YifuUser user = SecurityUtils.getUser();
		if (user == null || Common.isEmpty(user.getId())) {
			return R.failed(CommonConstants.PLEASE_LOG_IN);
		}
		menuUtil.setAuthSql(user, tEmployeeProject);
        return R.ok(tEmployeeProjectService.getTEmployeeProjectInfoPage(page,tEmployeeProject));
    }

    /**
     * 通过id查询项目档案表
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id查询", description = "通过id查询")
    @GetMapping("/{id}" )
    public R getById(@PathVariable("id" ) String id) {
        return R.ok(tEmployeeProjectService.getById(id));
    }

    /**
     * 新增项目档案表
     * @param tEmployeeProject 项目档案表
     * @return R
     */
    @Operation(summary = "新增项目档案表", description = "新增项目档案表")
    @SysLog("新增项目档案表" )
    @PostMapping
    @PreAuthorize("@pms.hasPermission('archives_temployeeproject_add')" )
    public R save(@Valid @RequestBody TEmployeeProject tEmployeeProject) {
        return tEmployeeProjectService.saveEmp(tEmployeeProject);
    }

    /**
     * 修改项目档案表
     * @param tEmployeeProject 项目档案表
     * @return R
     */
    @Operation(summary = "修改项目档案表", description = "修改项目档案表")
    @SysLog("修改项目档案表" )
    @PutMapping
    @PreAuthorize("@pms.hasPermission('archives_temployeeproject_edit')" )
    public R updateById(@RequestBody TEmployeeProject tEmployeeProject) {
        return tEmployeeProjectService.updateProject(tEmployeeProject);
    }

    /**
     * 通过id删除项目档案表
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id删除项目档案表", description = "通过id删除项目档案表")
    @SysLog("通过id删除项目档案表" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('archives_temployeeproject_del')" )
    public R removeById(@PathVariable String id) {
        return tEmployeeProjectService.removeProjectInfo(id);
    }

	/**
	 * 新增项目档案校验
	 * @param tEmployeeProject 项目档案表
	 * @Author huyc
	 * @Date 2022-06-20
	 * @return R
	 */
	@Operation(summary = "新增项目档案校验", description = "新增项目档案校验")
	@SysLog("新增项目档案校验" )
	@PostMapping("/check")
	@PreAuthorize("@pms.hasPermission('archives_temployeeproject_add')" )
	public R addCheck(@RequestBody TEmployeeProject tEmployeeProject) {
		return tEmployeeProjectService.addCheck(tEmployeeProject);
	}

	/**
	 * 复档复项相关操作
	 * @param jsonObject
	 * @Author huyc
	 * @Date 2022-06-20
	 * @return R
	 */
	@Operation(summary = "复档复项相关操作", description = "复档复项相关操作")
	@SysLog("复档复项相关操作" )
	@PostMapping("/reEmpInfo")
	public R reEmpInfo(@RequestBody JSONObject jsonObject) {
		return tEmployeeProjectService.reEmpInfo(jsonObject);
	}

	/**
	 * 减项
	 * @param tEmployeeProject
	 * @Author huyc
	 * @Date 2022-06-20
	 */
	@Operation(summary = "减项", description = "减项")
	@SysLog("减项" )
	@PostMapping("/deleteEmpPro")
	@PreAuthorize("@pms.hasPermission('archives_temployeeproject_delempproinfo')" )
	public R deleteEmpPro(@RequestBody TEmployeeProject tEmployeeProject) {
		return tEmployeeProjectService.deleteEmpPro(tEmployeeProject);
	}

	/**
	 * 减项
	 * @param file
	 * @Author huyc
	 * @Date 2022-06-20
	 */
	@SneakyThrows
	@Operation(summary = "批量减项", description = "批量减项")
	@SysLog("批量减项" )
	@PostMapping("/batchDeleteEmpPro")
	@PreAuthorize("@pms.hasPermission('archives_temployeeproject_batchdelempproinfo')" )
	public R<List<ErrorMessage>> batchDeleteEmpPro(@RequestBody MultipartFile file) {
		return tEmployeeProjectService.batchDeleteEmpPro(file.getInputStream());
	}

	/**
	 * 批量新增项目档案信息
	 *
	 * @param file
	 * @param idAdd
	 * @return R<List>
	 * @Author huyc
	 * @Date 2022-06-20
	 **/
	@SneakyThrows
	@Operation(description = "批量新增项目档案信息 hasPermission('wxhr:employeeProjectinfo_importListAdd')")
	@SysLog("批量新增项目档案信息")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('wxhr:employeeProjectinfo_importListAdd')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file, @RequestParam("idAdd") String idAdd){
		return tEmployeeProjectService.importListAdd(file.getInputStream(), idAdd);
	}

	/**
	 * 批量更新项目档案信息
	 * @param file
	 * @return R<List>
	 * @Author huyc
	 * @Date 2022-06-21
	 **/
	@SneakyThrows
	@Operation(description = "批量更新项目档案信息 hasPermission('archives_project_importListUpdate')")
	@SysLog("批量更新项目档案信息")
	@PostMapping("/importListUpate")
	@PreAuthorize("@pms.hasPermission('archives_project_importListUpdate')")
	public R<List<ErrorMessage>> importListUpate(@RequestBody MultipartFile file) {
		return tEmployeeProjectService.importEmpProjectUpdate(file.getInputStream());
	}

	/**
	 * 通过id批量删除项目档案
	 * @return R
	 */
	@Operation(summary = "通过id批量删除项目档案", description = "通过id批量删除项目档案")
	@SysLog("通过id批量删除项目档案" )
	@PostMapping("/deletebyIds" )
	@PreAuthorize("@pms.hasPermission('archives_temployeeproject_batchdel')" )
	public R<List<ErrorMessage>> removeByIds(@RequestBody List<String> idList) {
		return tEmployeeProjectService.batchRemoveByIds(idList);
	}

	/**
	 * 批量导出项目档案信息
	 * @param projectDTO
	 * @return R<List>
	 * @Author huyc
	 * @Date 2022-06-21
	 **/
	@PostMapping("/export")
	public void export(HttpServletResponse response, @RequestBody  EmployeeProjectExportParamVO projectDTO) {
		YifuUser user = SecurityUtils.getUser();
		if (user != null && Common.isNotNull(user.getId())) {
			menuUtil.setAuthSql(user, projectDTO);
		}
		tEmployeeProjectService.listExportProject(response,projectDTO,projectDTO.getExportFields());
	}

	/**
	 * 通过项目编码查询项目信息
	 * @param empNo
	 * @return R
	 */
	@Operation(summary = "通过项目编码查询项目信息", description = "通过项目编码查询项目信息")
	@PostMapping("/getItemInfo" )
	public R getItemByEmpNo(@RequestParam String empIdcard,@RequestParam String empNo) {
		return tEmployeeProjectService.getItemByEmpNo(empIdcard,empNo);
	}

	/**
	 * 通过身份证获取对应所有项目信息
	 * @author fxj
	 * @param empIdCard
	 * @date 2022-06-29 17:45
	 * @return R
	 */
	@Operation(summary = "通过身份证获取对应所有项目信息", description = "通过身份证获取对应所有项目信息")
	@GetMapping("/getProjectByIdCard" )
	public R<Map<String,String>> getProjectByIdCard(@RequestParam String empIdCard) {
		return tEmployeeProjectService.getProjectByIdCard(empIdCard);
	}

	/**
	 * @param employeeProject
	 * @Description: 获取项目档案
	 * @Author: hgw
	 * @Date: 2022/7/21 15:10
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeProject>
	 **/
	@Inner
	@PostMapping("/inner/getByEmpIdAndDeptId")
	public TEmployeeProject getByEmpIdAndDeptId(@RequestBody TEmployeeProject employeeProject) {
		return  tEmployeeProjectService.getByEmpIdAndDeptId(employeeProject.getEmpId(), employeeProject.getDeptId());
	}

	/**
	 * @Description: 更新社保公积金状态
	 * @Author: fxj
	 * @Date: 2022/08/01 18:52
	 **/
	@Operation(summary = "更新社保公积金状态", description = "更新社保公积金状态")
	@Inner
	@PostMapping("/inner/updateProjectSocialFund")
	public Boolean updateProjectSocialFund(@RequestBody UpProjectSocialFundVo vo) {
		return tEmployeeProjectService.updateProjectSocialFund(vo);
	}

	/**
	 * @Description: 更新项目档案的商险状态
	 * @Author: hgw
	 * @Date: 2022-11-23 17:17:59
	 **/
	@Operation(summary = "更新项目档案的商险状态", description = "更新项目档案的商险状态")
	@Inner
	@PostMapping("/inner/updateEmployeeInsuranceStatus")
	public Boolean updateEmployeeInsuranceStatus(@RequestBody EmpStatusVo vo) {
		return tEmployeeProjectService.updateEmployeeInsuranceStatus(vo);
	}

	/**
	 * @Description: 更新人员档案、项目档案的商险状态
	 * @Author: hgw
	 * @Date: 2022-11-23 17:17:59
	 **/
	@Operation(summary = "更新人员档案、项目档案的薪资状态", description = "更新人员档案、项目档案的薪资状态")
	@Inner
	@PostMapping("/inner/updateEmployeeSalaryStatus")
	public Boolean updateEmployeeSalaryStatus(@RequestBody EmpStatusVo vo) {
		return tEmployeeProjectService.updateEmployeeSalaryStatus(vo);
	}

	/**
	 * 查询项目对应的数据
	 * @author hyc
	 * @param deptNoList
	 * @date 2022-12-12 17:45
	 * @return R
	 */
	@Operation(summary = "查询项目对应的数据", description = "查询项目对应的数据")
	@PostMapping("/selectSettleMentData")
	public R<List<TSettleDomainDataVo>> selectSettleMentData(@RequestBody List<String> deptNoList) {
		return tEmployeeProjectService.selectSettleMentData(deptNoList);
	}
}
