/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.salary.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TBankSet;
import com.yifu.cloud.plus.v1.yifu.salary.service.TBankSetService;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TBankSetSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 开户行配置表
 *
 * @author hgw
 * @date 2023-03-06 11:54:21
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tbankset")
@Tag(name = "开户行配置表管理")
public class TBankSetController {

	private final TBankSetService tBankSetService;


	/**
	 * 简单分页查询
	 *
	 * @param page     分页对象
	 * @param tBankSet 开户行配置表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TBankSet>> getTBankSetPage(Page<TBankSet> page, TBankSetSearchVo tBankSet) {
		return new R<>(tBankSetService.getTBankSetPage(page, tBankSet));
	}

	/**
	 * 不分页查询
	 *
	 * @param tBankSet 开户行配置表
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	public R<List<TBankSet>> getTBankSetNoPage(@RequestBody TBankSetSearchVo tBankSet) {
		return R.ok(tBankSetService.noPageDiy(tBankSet));
	}

	/**
	 * 通过id查询开户行配置表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('salary_tbankset_get')")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('salary_tbankset_get')")
	public R<TBankSet> getById(@PathVariable("id") String id) {
		return R.ok(tBankSetService.getById(id));
	}

	/**
	 * 新增开户行配置表
	 *
	 * @param tBankSet 开户行配置表
	 * @return R
	 */
	@Operation(summary = "新增开户行配置表", description = "新增开户行配置表：hasPermission('salary_tbankset_add')")
	@SysLog("新增开户行配置表")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('salary_tbankset_add')")
	public R<Boolean> save(@RequestBody TBankSet tBankSet) {
		if (Common.isEmpty(tBankSet.getBankName())) {
			return R.failed("开户行不可为空！");
		}
		List<TBankSet> tBankSetList = tBankSetService.getList(tBankSet.getBankName(), null);
		if (tBankSetList != null && !tBankSetList.isEmpty()) {
			return R.failed("已存在该开户行！");
		}
		return R.ok(tBankSetService.save(tBankSet));
	}

	/**
	 * 修改开户行配置表
	 *
	 * @param tBankSet 开户行配置表
	 * @return R
	 */
	@Operation(summary = "修改开户行配置表", description = "修改开户行配置表：hasPermission('salary_tbankset_edit')")
	@SysLog("修改开户行配置表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('salary_tbankset_edit')")
	public R<Boolean> updateById(@RequestBody TBankSet tBankSet) {
		List<TBankSet> tBankSetList = tBankSetService.getList(tBankSet.getBankName(), tBankSet.getId());
		if (tBankSetList != null && !tBankSetList.isEmpty()) {
			return R.failed("已存在该开户行！");
		}
		return R.ok(tBankSetService.updateById(tBankSet));
	}

	/**
	 * 通过id删除开户行配置表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除开户行配置表", description = "通过id删除开户行配置表：hasPermission('salary_tbankset_del')")
	@SysLog("通过id删除开户行配置表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('salary_tbankset_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		TBankSet bankSet = tBankSetService.getById(id);
		if (bankSet != null) {
			bankSet.setDeleteFlag(CommonConstants.STATUS_DEL);
			tBankSetService.updateById(bankSet);
			return R.ok();
		} else {
			return R.failed("未找到!");
		}
	}

	/**
	 * 开户行配置表 批量导入
	 *
	 * @author hgw
	 * @date 2023-03-06 11:54:21
	 **/
	@SneakyThrows
	@Operation(description = "批量新增开户行配置表 hasPermission('salary_tbankset-batch-import')")
	@SysLog("批量新增开户行配置表")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('salary_tbankset-batch-import')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file) {
		return tBankSetService.importDiy(file.getInputStream());
	}

	/**
	 * 开户行配置表 批量导出
	 *
	 * @author hgw
	 * @date 2023-03-06 11:54:21
	 **/
	@Operation(description = "导出开户行配置表 hasPermission('salary_tbankset-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('salary_tbankset-export')")
	public void export(HttpServletResponse response, @RequestBody TBankSetSearchVo searchVo) {
		tBankSetService.listExport(response, searchVo);
	}
}
