/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.social.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.social.entity.TSocialInfo;
import com.yifu.cloud.plus.v1.yifu.social.service.TSocialInfoService;
import com.yifu.cloud.plus.v1.yifu.social.vo.TSocialInfoSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 社保明细表
 *
 * @author fxj
 * @date 2022-07-15 11:38:05
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tsocialinfo" )
@Tag(name = "社保明细表管理")
public class TSocialInfoController {

    private final TSocialInfoService tSocialInfoService;

    
    /**
     * 简单分页查询
     * @param page 分页对象
     * @param tSocialInfo 社保明细表
     * @return
     */
    @Operation(description = "简单分页查询")
    @GetMapping("/page")
    public R<IPage<TSocialInfo>> getTSocialInfoPage(Page<TSocialInfo> page, TSocialInfoSearchVo tSocialInfo) {
        return  new R<>(tSocialInfoService.getTSocialInfoPage(page,tSocialInfo));
    }
    /**
     * 不分页查询
     * @param tSocialInfo 社保明细表
     * @return
     */
    @Operation(summary = "不分页查询", description = "不分页查询")
    @PostMapping("/noPage" )
    //@PreAuthorize("@pms.hasPermission('demo_tsocialinfo_get')" )
    public R<List<TSocialInfo>> getTSocialInfoNoPage(@RequestBody TSocialInfoSearchVo tSocialInfo) {
        return R.ok(tSocialInfoService.noPageDiy(tSocialInfo));
    }

    /**
     * 通过id查询社保明细表
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id查询", description = "通过id查询：hasPermission('demo_tsocialinfo_get')")
    @GetMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_tsocialinfo_get')" )
    public R<TSocialInfo> getById(@PathVariable("id" ) String id) {
        return R.ok(tSocialInfoService.getById(id));
    }

    /**
     * 新增社保明细表
     * @param tSocialInfo 社保明细表
     * @return R
     */
    @Operation(summary = "新增社保明细表", description = "新增社保明细表：hasPermission('demo_tsocialinfo_add')")
    @SysLog("新增社保明细表" )
    @PostMapping
    @PreAuthorize("@pms.hasPermission('demo_tsocialinfo_add')" )
    public R<Boolean> save(@RequestBody TSocialInfo tSocialInfo) {
        return R.ok(tSocialInfoService.save(tSocialInfo));
    }

    /**
     * 修改社保明细表
     * @param tSocialInfo 社保明细表
     * @return R
     */
    @Operation(summary = "修改社保明细表", description = "修改社保明细表：hasPermission('demo_tsocialinfo_edit')")
    @SysLog("修改社保明细表" )
    @PutMapping
    @PreAuthorize("@pms.hasPermission('demo_tsocialinfo_edit')" )
    public R<Boolean> updateById(@RequestBody TSocialInfo tSocialInfo) {
        return R.ok(tSocialInfoService.updateById(tSocialInfo));
    }

    /**
     * 通过id删除社保明细表
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id删除社保明细表", description = "通过id删除社保明细表：hasPermission('demo_tsocialinfo_del')")
    @SysLog("通过id删除社保明细表" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_tsocialinfo_del')" )
    public R<Boolean> removeById(@PathVariable String id) {
        return R.ok(tSocialInfoService.removeById(id));
    }

    /**
	 * 社保明细表 批量导入
	 *
     * @author fxj
     * @date 2022-07-15 11:38:05
	 **/
    @SneakyThrows
    @Operation(description = "批量新增社保明细表 hasPermission('demo_tsocialinfo-batch-import')")
    @SysLog("批量新增社保明细表")
    @PostMapping("/importListAdd")
    @PreAuthorize("@pms.hasPermission('demo_tsocialinfo-batch-import')")
    public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file){
        return tSocialInfoService.importDiy(file.getInputStream());
    }

    /**
     * 社保明细表 批量导出
     * @author fxj
     * @date 2022-07-15 11:38:05
     **/
    @Operation(description = "导出社保明细表 hasPermission('demo_tsocialinfo-export')")
    @PostMapping("/export")
    @PreAuthorize("@pms.hasPermission('demo_tsocialinfo-export')")
    public void export(HttpServletResponse response, @RequestBody TSocialInfoSearchVo searchVo) {
            tSocialInfoService.listExport(response,searchVo);
    }

	/**
	 * @Description: 获取社保，根据身份证与项目ID
	 * @Author: hgw
	 * @Date: 2022/8/1 11:27
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.social.entity.TSocialInfo>
	 **/
	@Operation(summary = "获取社保，根据身份证与项目ID", description = "获取社保，根据身份证与项目ID")
	@GetMapping("/getSocialByIdCardAndDeptId" )
	public R<TSocialInfo> getSocialByIdCardAndDeptId(@RequestParam String idCard
			, @RequestParam(required = false) String deptId) {
		TSocialInfo socialInfo;
		if (Common.isNotNull(deptId)) {
			socialInfo = tSocialInfoService.getOne(Wrappers.<TSocialInfo>query().lambda()
					.eq(TSocialInfo::getEmpIdcard, idCard)
					.eq(TSocialInfo::getDeleteFlag, CommonConstants.STATUS_NORMAL)
					.eq(TSocialInfo::getSettleDomain, deptId)
					.orderByDesc(TSocialInfo::getCreateTime).last(CommonConstants.LAST_ONE_SQL));
		} else {
			socialInfo = tSocialInfoService.getOne(Wrappers.<TSocialInfo>query().lambda()
					.eq(TSocialInfo::getEmpIdcard, idCard)
					.eq(TSocialInfo::getDeleteFlag, CommonConstants.STATUS_NORMAL)
					.orderByDesc(TSocialInfo::getCreateTime).last(CommonConstants.LAST_ONE_SQL));
		}
		return R.ok(socialInfo);
	}
}
