/*
 * Copyright (c) 2020 yifu4cloud Authors. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.yifu.cloud.plus.v1.yifu.admin.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.yifu.cloud.plus.v1.yifu.admin.api.entity.SysArea;
import com.yifu.cloud.plus.v1.yifu.admin.api.vo.AreaMap;
import com.yifu.cloud.plus.v1.yifu.admin.api.vo.AreaVo;
import com.yifu.cloud.plus.v1.yifu.admin.service.SysAreaService;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CacheConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.util.RedisUtil;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * 字典表 前端控制器
 * </p>
 *
 * @author lengleng
 * @since 2019-03-19
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/area")
@Tag(name = "区域管理模块")
public class AreaController {

	private final SysAreaService sysAreaService;

	/**
	 * @Description: 获取所有区域（id，名称）
	 * @Author: hgw
	 * @Date: 2022/6/22 19:33
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.Map < java.lang.Integer, java.lang.String>>
	 **/
	@Inner
	@PostMapping("/inner/getAreaToId")
	public R<Map<String, String>> getAreaToId() {
		return sysAreaService.getAreaToId();
	}

	/**
	 * @param
	 * @Description: 获取所有区域（名称，id）
	 * @Author: hgw
	 * @Date: 2022/6/22 19:34
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.Map < java.lang.String, java.util.Map < java.lang.String, java.lang.String>>>
	 **/
	@Inner
	@PostMapping("/inner/getAreaToName")
	public R<Map<String, String>> getAreaToName() {
		return sysAreaService.getAreaToName();
	}

	/**
	 * @param
	 * @Description: 获取所有区域list
	 * @Author: hgw
	 * @Date: 2022-7-6 16:42:39
	 **/
	@Inner
	@PostMapping("/inner/getSysAreaList")
	public List<SysArea> getSysAreaList() {
		return sysAreaService.list(Wrappers.<SysArea>query().lambda()
				.eq(SysArea::getStatus, CommonConstants.ZERO_INT).orderByAsc(SysArea::getParentId));
	}
	/**
	 * @param
	 * @Description: 获取所有区域list
	 * @Author: hyc
	 * @Date: 2022-7-21 19:45:24
	 **/
	@Inner
	@PostMapping("/inner/getAreaVo")
	public AreaVo getAreaVo() {
		List<SysArea> aList = sysAreaService.list(Wrappers.<SysArea>query().lambda()
				.eq(SysArea::getStatus, CommonConstants.ZERO_INT).orderByAsc(SysArea::getParentId));
		AreaVo vo = new AreaVo();
		vo.setSysAreaList(aList);
		return vo;
	}
	@Autowired
	private RedisUtil redisUtil;
	/**
	 * @param
	 * @Description: 获取所有区域list
	 * @Author: hyc
	 * @Date: 2022-7-21 19:45:24
	 **/
	@Inner
	@PostMapping("/inner/getAreaMap")
	public AreaMap getAreaMap() {
		Map<String, String> idLabelMap = (Map<String, String>) redisUtil.get(CacheConstants.AREA_LABEL_GROUP);
		Map<String, String> labelIdMap = (Map<String, String>) redisUtil.get(CacheConstants.AREA_VALUE_GROUP);
		if (Common.isEmpty(idLabelMap) || Common.isEmpty(labelIdMap)){
			initAreaRedis();
		}
		AreaMap map = new AreaMap();
		idLabelMap = (Map<String, String>) redisUtil.get(CacheConstants.AREA_LABEL_GROUP);
		labelIdMap = (Map<String, String>) redisUtil.get(CacheConstants.AREA_VALUE_GROUP);
		map.setIdLabelMap(idLabelMap);
		map.setLabelIdMap(labelIdMap);
		return map;
	}

	private void initAreaRedis() {
		List<SysArea> allAreas = sysAreaService.getAllArea();
		if (Common.isNotNull(allAreas)) {
			Map<String, SysArea> areaMap = new HashMap<>();
			Map<String, String> idLabelMap = new HashMap<>();
			for (SysArea area : allAreas) {
				areaMap.put(area.getId().toString(), area);
				idLabelMap.put(area.getId().toString(), area.getAreaName());
			}
			redisUtil.set(CacheConstants.AREA_LABEL_GROUP,idLabelMap);
			SysArea temp;
			Map<String, String> labelIdMap = new HashMap<>();
			for (SysArea area : allAreas) {
				if (Common.isNotNull(area.getParentId()) && area.getParentId().intValue() > 0) {
					temp = areaMap.get(area.getParentId().toString());
					labelIdMap.put(area.getAreaName()
							+ CommonConstants.DOWN_LINE_STRING
							+ (Common.isNotNull(temp) ? temp.getAreaName() : ""), area.getId().toString());
				} else {
					labelIdMap.put(area.getAreaName(), area.getId().toString());
				}
			}
			redisUtil.set(CacheConstants.AREA_VALUE_GROUP,labelIdMap);
		}
	}
	/**
	 * @Author fxj
	 * @Description 地区map
	 * @Date 14:56 2023/1/4
	 * @Param
	 * @return
	**/
	@Operation(description = "地区map")
	@GetMapping("/getAreaMap")
	public R<Map<Integer,SysArea>> getAreaMapUpms() {
		return new R<>(sysAreaService.getSysAreaMap());
	}

	/**
	 * @Author fxj
	 * @Description 区划树形结构
	 * @Date 15:09 2023/1/4
	 * @Param
	 * @return
	**/
	@Operation(description = "区划树形结构")
	@GetMapping("/tree")
	public R getTree() {
		return new R<>(sysAreaService.getAreaTree());
	}

	/**
	 * 通过id查询单条记录
	 *
	 * @param id
	 * @return R
	 */
	@Operation(description = "id查询")
	@GetMapping("/{id}")
	public R<SysArea> getById(@PathVariable("id") Integer id) {
		return new R<>(sysAreaService.getById(id));
	}
}
