/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TEmpContractAlert;
import com.yifu.cloud.plus.v1.yifu.archives.service.TEmpContractAlertService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.ContractAlertSearchVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.MenuUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 员工合同续签待办
 *
 * @author fxj
 * @date 2022-07-01 14:15:05
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tempcontractalert" )
@Tag(name = "员工合同续签待办管理")
public class TEmpContractAlertController {

    private final TEmpContractAlertService tEmpContractAlertService;
	private final MenuUtil menuUtil;

    /**
     * 分页查询
     * @param page 分页对象
     * @param searchVo 员工合同续签待办
     * @return
     */
    @Operation(summary = "分页查询", description = "分页查询")
    @GetMapping("/page" )
    //@PreAuthorize("@pms.hasPermission('demo_tempcontractalert_get')" )
    public R<IPage<TEmpContractAlert>> getTEmpContractAlertPage(Page page, ContractAlertSearchVo searchVo) {
		setAuth(searchVo);
        return R.ok(tEmpContractAlertService.pageDiy(page, searchVo));
    }
    /**
     * 不分页查询
     * @param searchVo 员工合同续签待办
     * @return
     */
    @Operation(summary = "不分页查询", description = "不分页查询")
    @PostMapping("/noPage" )
    //@PreAuthorize("@pms.hasPermission('demo_tempcontractalert_get')" )
    public R<List<TEmpContractAlert>> getTEmpContractAlertNoPage(@RequestBody ContractAlertSearchVo searchVo) {
		setAuth(searchVo);
        return R.ok(tEmpContractAlertService.listDiy(searchVo));
    }

    /**
     * 通过id查询员工合同续签待办
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id查询", description = "通过id查询：hasPermission('demo_tempcontractalert_get')")
    @GetMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_tempcontractalert_get')" )
    public R<TEmpContractAlert> getById(@PathVariable("id" ) String id) {
        return R.ok(tEmpContractAlertService.getById(id));
    }

    /**
     * 新增员工合同续签待办
     * @param tEmpContractAlert 员工合同续签待办
     * @return R
     */
    @Operation(summary = "新增员工合同续签待办", description = "新增员工合同续签待办：hasPermission('demo_tempcontractalert_add')")
    @SysLog("新增员工合同续签待办" )
    @PostMapping
    @PreAuthorize("@pms.hasPermission('demo_tempcontractalert_add')" )
    public R<Boolean> save(@RequestBody TEmpContractAlert tEmpContractAlert) {
        return R.ok(tEmpContractAlertService.save(tEmpContractAlert));
    }

    /**
     * 修改员工合同续签待办
     * @param tEmpContractAlert 员工合同续签待办
     * @return R
     */
    @Operation(summary = "修改员工合同续签待办", description = "修改员工合同续签待办：hasPermission('demo_tempcontractalert_edit')")
    @SysLog("修改员工合同续签待办" )
    @PutMapping
    @PreAuthorize("@pms.hasPermission('demo_tempcontractalert_edit')" )
    public R<Boolean> updateById(@RequestBody TEmpContractAlert tEmpContractAlert) {
        return R.ok(tEmpContractAlertService.updateById(tEmpContractAlert));
    }

    /**
     * 通过id删除员工合同续签待办
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id删除员工合同续签待办", description = "通过id删除员工合同续签待办：hasPermission('demo_tempcontractalert_del')")
    @SysLog("通过id删除员工合同续签待办" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_tempcontractalert_del')" )
    public R<Boolean> removeById(@PathVariable String id) {
        return R.ok(tEmpContractAlertService.removeById(id));
    }

	/**
	 * 批量导出合同信息
	 * @param searchVo
	 * @return R<List>
	 * @Author FXJ
	 * @Date 2022-07-1
	 **/
	@Operation(description = "导出")
	@PostMapping("/export")
	public void export(HttpServletResponse response, @RequestBody ContractAlertSearchVo searchVo) {
		setAuth(searchVo);
		tEmpContractAlertService.listExport(response,searchVo);
	}

	private void setAuth(ContractAlertSearchVo searchVo) {
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, searchVo);
		if (Common.isNotNull(searchVo.getAuthSql())) {
			if (searchVo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
				searchVo.setAuthSql(searchVo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "dept_id"));
			}
			if (searchVo.getAuthSql().contains(CommonConstants.A_CREATE_BY)) {
				searchVo.setAuthSql(searchVo.getAuthSql().replace(CommonConstants.A_CREATE_BY, "CREATE_BY"));
			}
		}
	}

	/**
	 * 定时任务生成合同续签代码信息
	 * @return R<List>
	 * @Author FXJ
	 * @Date 2022-07-4
	 **/
	@SysLog("定时任务生成合同续签代码信息")
	@PostMapping("/taskCreateContractAlert")
	@Inner
	public R taskCreateContractAlert() {
		return tEmpContractAlertService.taskCreateContractAlert();
	}
}
