/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.salary.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.MenuUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TSalaryEmployee;
import com.yifu.cloud.plus.v1.yifu.salary.service.TSalaryEmployeeService;
import com.yifu.cloud.plus.v1.yifu.salary.vo.SalaryEmployeeImportVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TSalaryEmployeeEkpUpdateVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TSalaryEmployeeExportVo;
import com.yifu.cloud.plus.v1.yifu.salary.vo.TSalaryEmployeeSearchVo;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 薪酬人员表
 *
 * @author hgw
 * @date 2022-08-05 11:40:14
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tsalaryemployee")
@Tag(name = "薪酬人员表管理")
public class TSalaryEmployeeController {

	private final TSalaryEmployeeService tSalaryEmployeeService;

	private final MenuUtil menuUtil;

	/**
	 * 简单分页查询
	 *
	 * @param page            分页对象
	 * @param tSalaryEmployee 薪酬人员表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TSalaryEmployee>> getTSalaryEmployeePage(Page<TSalaryEmployee> page, TSalaryEmployeeSearchVo tSalaryEmployee) {
		YifuUser user = SecurityUtils.getUser();
		if (user == null || Common.isEmpty(user.getId())) {
			return R.failed(CommonConstants.PLEASE_LOG_IN);
		}
		menuUtil.setAuthSql(user, tSalaryEmployee);
		return new R<>(tSalaryEmployeeService.getTSalaryEmployeePage(page, tSalaryEmployee));
	}

	/**
	 * 不分页查询
	 *
	 * @param tSalaryEmployee 薪酬人员表
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	//@PreAuthorize("@pms.hasPermission('salary_tsalaryemployee_get')" )
	public R<List<TSalaryEmployeeExportVo>> getTSalaryEmployeeNoPage(@RequestBody TSalaryEmployeeSearchVo tSalaryEmployee) {
		return R.ok(tSalaryEmployeeService.noPageDiy(tSalaryEmployee));
	}

	/**
	 * 通过id查询薪酬人员表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('salary_tsalaryemployee_get')")
	@GetMapping("/{id}")
	public R<TSalaryEmployee> getById(@PathVariable("id") String id) {
		return R.ok(tSalaryEmployeeService.getById(id));
	}

	/**
	 * 新增薪酬人员表
	 *
	 * @param tSalaryEmployee 薪酬人员表
	 * @return R
	 */
	@Operation(summary = "新增薪酬人员表", description = "新增薪酬人员表：hasPermission('salary_tsalaryemployee_add')")
	@SysLog("新增薪酬人员表")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('salary_tsalaryemployee_add')")
	public R<Boolean> save(@RequestBody TSalaryEmployee tSalaryEmployee) {
		return R.ok(tSalaryEmployeeService.save(tSalaryEmployee));
	}

	/**
	 * 修改薪酬人员表
	 *
	 * @param tSalaryEmployee 薪酬人员表
	 * @return R
	 */
	@Operation(summary = "修改薪酬人员表", description = "修改薪酬人员表：hasPermission('salary_tsalaryemployee_edit')")
	@SysLog("修改薪酬人员表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('salary_tsalaryemployee_edit')")
	public R<Boolean> updateById(@RequestBody TSalaryEmployee tSalaryEmployee) {
		return R.ok(tSalaryEmployeeService.updateById(tSalaryEmployee));
	}

	/**
	 * 通过id删除薪酬人员表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除薪酬人员表", description = "通过id删除薪酬人员表：hasPermission('salary_tsalaryemployee_del')")
	@SysLog("通过id删除薪酬人员表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('salary_tsalaryemployee_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tSalaryEmployeeService.removeById(id));
	}

	/**
	 * 薪酬人员表 批量导出
	 *
	 * @author fxj
	 * @date 2022-08-05 11:40:14
	 **/
	@Operation(description = "导出薪酬人员表 hasPermission('salary_tsalaryemployee-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('salary_tsalaryemployee-export')")
	public void export(HttpServletResponse response, @RequestBody TSalaryEmployeeSearchVo searchVo) {
		YifuUser user = SecurityUtils.getUser();
		if (user != null && Common.isNotNull(user.getId())) {
			menuUtil.setAuthSql(user, searchVo);
		}
		if (Common.isNotNull(searchVo.getAuthSql()) && searchVo.getAuthSql().contains(CommonConstants.A_CREATE_BY)) {
			searchVo.setAuthSql(searchVo.getAuthSql().replace(CommonConstants.A_CREATE_BY, "CREATE_BY"));
		}
		tSalaryEmployeeService.listExport(response, searchVo);
	}

	/**
	 * 批量更新-薪资人员信息
	 *
	 * @param file
	 * @return
	 * @Author hgw
	 * @Date 2022-10-10 17:17:11
	 **/
	@SneakyThrows
	@Operation(description = "批量更新-薪资人员信息 hasPermission('tsalaryemployee_batch_update')")
	@SysLog("批量更新-薪资人员信息")
	@PostMapping("/batchUpdateSalaryEmployee")
	@PreAuthorize("@pms.hasPermission('tsalaryemployee_batch_update')")
	public R<List<ErrorMessage>> batchUpdateSalaryEmployee(@RequestBody MultipartFile file) {
		return tSalaryEmployeeService.batchUpdateSalaryEmployee(file.getInputStream());
	}

	/**
	 * @param vo
	 * @Description: 批量新增或更新人员
	 * @Author: hgw
	 * @Date: 2023/1/17 15:35
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage>>
	 **/
	@Operation(description = "批量新增或更新人员")
	@SysLog("批量新增或更新人员")
	@PostMapping("/importEmployee")
	@PreAuthorize("@pms.hasPermission('tsalaryemployee_import_employee')")
	public R<List<ErrorMessage>> importEmployee(@RequestBody SalaryEmployeeImportVo vo) {
		return tSalaryEmployeeService.importEmployee(vo);
	}

	/**
	 * 同步ekp薪酬信息
	 * @author huyc
	 * @param vo
	 * @return {@link R}
	 */
	@Schema(description = "同步ekp薪酬人员信息")
	@PostMapping(value = "/updateEmployeeFromEkp")
	public R updateEmployeeFromEkp(TSalaryEmployeeEkpUpdateVo vo) {
		return tSalaryEmployeeService.updateEmployeeFromEkp(vo);
	}

}
