/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.archives.service.impl;

import cn.hutool.core.bean.BeanUtil;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.admin.api.vo.AreaMap;
import com.yifu.cloud.plus.v1.yifu.archives.entity.*;
import com.yifu.cloud.plus.v1.yifu.archives.mapper.*;
import com.yifu.cloud.plus.v1.yifu.archives.service.LGuaziOfferRecordService;
import com.yifu.cloud.plus.v1.yifu.archives.service.*;
import com.yifu.cloud.plus.v1.yifu.archives.service.TPreEmpDeclarationService;
import com.yifu.cloud.plus.v1.yifu.archives.util.IdCardUtil;
import com.yifu.cloud.plus.v1.yifu.archives.vo.*;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CacheConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.ExcelAttributeConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.SecurityConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.*;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.config.DaprUpmsProperties;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.HttpDaprUtil;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.UpmsDaprUtils;
import com.yifu.cloud.plus.v1.yifu.common.mybatis.base.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.AuthorityUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URL;
import java.net.URLEncoder;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.*;

/**
 * 瓜子档案表
 *
 * @author chenyx
 * @date 2025-06-13 15:19:03
 */
@Log4j2
@Service
@RequiredArgsConstructor
@EnableConfigurationProperties(DaprUpmsProperties.class)
public class TGzEmpInfoServiceImpl extends ServiceImpl<TGzEmpInfoMapper, TGzEmpInfo> implements TGzEmpInfoService {

	private final TGzOfferInfoMapper gzOfferInfoMapper;

	private final TGzEmpContactInfoMapper gzEmpContactInfoMapper;

	private final TGzEmpEducationMapper gzEmpEducationMapper;

	private final TGzEmpFamilyMapper gzEmpFamilyMapper;

	private final TGzEmpSkillMapper gzEmpSkillMapper;

	private final TGzEmpWorkMapper gzEmpWorkMapper;

	private final TAttaInfoService attaInfoService;

	// 拿区域
	private final DaprUpmsProperties daprUpmsProperties;

	@Autowired
	private OSSUtil ossUtil;

	private final LGuaziOfferRecordService lGuaziOfferRecordService;

	private final TPreEmpDeclarationService tPreEmpDeclarationService;

	@Autowired
	private UpmsDaprUtils upmsDaprUtils;

	// 项目
	private final TSettleDomainService tSettleDomainService;
	// 人员档案表
	private final TEmployeeInfoService tEmployeeInfoService;
	// 项目档案表
	private final TEmployeeProjectService tEmployeeProjectService;
	// 员工工作履历信息表
	private final TEmpWorkRecordingService tEmpWorkRecordingService;
	// 紧急联络人
	private final TEmpContactInfoService tEmpContactInfoService;
	// 员工学历信息表
	private final TEmpEducationService tEmpEducationService;
	// 员工家庭信息表
	private final TEmpFamilyService tEmpFamilyService;


	/**
	 * 瓜子档案表简单分页查询
	 *
	 * @param tGzEmpInfo 瓜子档案表
	 * @return
	 */
	@Override
	public IPage<TGzEmpInfo> getTGzEmpInfoPage(Page<TGzEmpInfo> page, TGzEmpInfoSearchVo tGzEmpInfo) {
		return baseMapper.getTGzEmpInfoPage(page, tGzEmpInfo);
	}

	/**
	 * 通过OfferId查询瓜子档案详情
	 *
	 * @param offerId OfferId
	 * @return R
	 */
	@Override
	public R<TGzEmpInfoSearchVo> getInfoByOfferId(Integer offerId) {
		TGzEmpInfoSearchVo gzEmpInfoVo = new TGzEmpInfoSearchVo();

		TGzOfferInfo gzOfferInfo = gzOfferInfoMapper.getInfoById(offerId);
		if(Common.isEmpty(gzOfferInfo)){
			return R.failed("数据异常，禁止操作");
		}


		TGzEmpInfo gzEmpInfo = baseMapper.getInfoByOfferId(offerId);

		// 还未生成档案，取offer里的信息
		if(Common.isEmpty(gzEmpInfo)){
			gzEmpInfoVo.setBizId(gzOfferInfo.getBizId());
			gzEmpInfoVo.setOfferId(offerId);
			gzEmpInfoVo.setName(gzOfferInfo.getName());
			gzEmpInfoVo.setNameAc(gzOfferInfo.getNameAc());
			gzEmpInfoVo.setNationalIdType(gzOfferInfo.getNationalIdType());
			gzEmpInfoVo.setNationalId(gzOfferInfo.getNationalId());
			// 根据身份证得出 出生日期和性别
			if(Common.isNotNull(gzEmpInfoVo.getNationalId()) && gzEmpInfoVo.getNationalId().length() > 14){
				String idCard = gzEmpInfoVo.getNationalId();
				gzEmpInfoVo.setBirthdate(IdCardUtil.getBirthdate(idCard));
				// 性别: F:女,M:男
				if (idCard.length() > 17) {
					if (Integer.parseInt(idCard.substring(16, 17)) % 2 == 1) {
						gzEmpInfoVo.setSex("M");
					} else {
						gzEmpInfoVo.setSex("F");
					}
				}
			}
			gzEmpInfoVo.setPhone(gzOfferInfo.getPhone());
			gzEmpInfoVo.setMail(gzOfferInfo.getEmail());
			// 地址类型: 系统存默认值“现住址”:1A
			gzEmpInfoVo.setAddressType("1A");

			// 默认所有附属信息给个空对象（前端要求）
			List<TGzEmpContactInfo> emergContactList = new ArrayList<>();
			TGzEmpContactInfo gzEmpContactInfo = new TGzEmpContactInfo();
			emergContactList.add(gzEmpContactInfo);
			gzEmpInfoVo.setEmergContactList(emergContactList);

			List<TGzEmpEducation> eduList = new ArrayList<>();
			TGzEmpEducation gzEmpEducation = new TGzEmpEducation();
			eduList.add(gzEmpEducation);
			gzEmpInfoVo.setEduExpList(eduList);

			List<TGzEmpWork> workList = new ArrayList<>();
			TGzEmpWork gzEmpWork = new TGzEmpWork();
			workList.add(gzEmpWork);
			gzEmpInfoVo.setWorkExpList(workList);

			List<TGzEmpFamily> familyList = new ArrayList<>();
			TGzEmpFamily gzEmpFamily = new TGzEmpFamily();
			familyList.add(gzEmpFamily);
			gzEmpInfoVo.setFamilyList(familyList);

			List<TGzEmpSkill> skillList = new ArrayList<>();
			TGzEmpSkill gzEmpSkill = new TGzEmpSkill();
			skillList.add(gzEmpSkill);
			gzEmpInfoVo.setSkillList(skillList);

		} else {
			BeanUtil.copyProperties(gzEmpInfo, gzEmpInfoVo);
			// 获取瓜子档案附属信息

			// 紧急联系人
			LambdaQueryWrapper<TGzEmpContactInfo> queryContactInfo = Wrappers.<TGzEmpContactInfo>lambdaQuery()
					.eq(TGzEmpContactInfo::getEmpId, gzEmpInfo.getId())
					.eq(TGzEmpContactInfo::getDelFlag, CommonConstants.ZERO_STRING)
					.eq(TGzEmpContactInfo::getPrimaryNid, "Y")
					.last(" limit 1 ");
			List<TGzEmpContactInfo> emergContactList = gzEmpContactInfoMapper.selectList(queryContactInfo);
			gzEmpInfoVo.setEmergContactList(emergContactList);
			// 教育经历
			LambdaQueryWrapper<TGzEmpEducation> queryEduInfo = Wrappers.<TGzEmpEducation>lambdaQuery()
					.eq(TGzEmpEducation::getEmpId, gzEmpInfo.getId())
					.eq(TGzEmpEducation::getDelFlag, CommonConstants.ZERO_STRING)
					.eq(TGzEmpEducation::getHighEduQualFlag, "Y")
					.last(" limit 1 ");
			List<TGzEmpEducation> eduList = gzEmpEducationMapper.selectList(queryEduInfo);
			gzEmpInfoVo.setEduExpList(eduList);

			// 工作经历列表
			LambdaQueryWrapper<TGzEmpWork> queryWorkInfo = Wrappers.<TGzEmpWork>lambdaQuery()
					.eq(TGzEmpWork::getEmpId, gzEmpInfo.getId())
					.eq(TGzEmpWork::getDelFlag, CommonConstants.ZERO_STRING)
					.orderByAsc(TGzEmpWork::getCreateTime).orderByAsc(TGzEmpWork::getId);
			List<TGzEmpWork> workList = gzEmpWorkMapper.selectList(queryWorkInfo);
			gzEmpInfoVo.setWorkExpList(workList);
			// 家庭成员列表
			LambdaQueryWrapper<TGzEmpFamily> queryFamilyInfo = Wrappers.<TGzEmpFamily>lambdaQuery()
					.eq(TGzEmpFamily::getEmpId, gzEmpInfo.getId())
					.eq(TGzEmpFamily::getDelFlag, CommonConstants.ZERO_STRING)
					.orderByAsc(TGzEmpFamily::getCreateTime).orderByAsc(TGzEmpFamily::getId);
			List<TGzEmpFamily> familyList = gzEmpFamilyMapper.selectList(queryFamilyInfo);
			gzEmpInfoVo.setFamilyList(familyList);
			// 技能证书列表
			LambdaQueryWrapper<TGzEmpSkill> querySkillInfo = Wrappers.<TGzEmpSkill>lambdaQuery()
					.eq(TGzEmpSkill::getEmpId, gzEmpInfo.getId())
					.eq(TGzEmpSkill::getDelFlag, CommonConstants.ZERO_STRING)
					.orderByAsc(TGzEmpSkill::getCreateTime).orderByAsc(TGzEmpSkill::getId);
			List<TGzEmpSkill> skillList = gzEmpSkillMapper.selectList(querySkillInfo);
			gzEmpInfoVo.setSkillList(skillList);

			// 附件
			List<TAttaInfo> attaInfos = attaInfoService.list(Wrappers.<TAttaInfo>query().lambda().eq(TAttaInfo::getDomainId, gzEmpInfo.getId()));
			if (Common.isNotNull(attaInfos)) {
				URL url;
				for (TAttaInfo atta : attaInfos) {
					url = ossUtil.getObjectUrl(null, atta.getAttaSrc());
					if (Common.isNotNull(url)) {
						atta.setAttaUrl(url.toString());
					}
				}
			}
			gzEmpInfoVo.setAttaList(attaInfos);
		}

		TPreEmpDeclaration declaration = tPreEmpDeclarationService.getTPreEmpDeclarationList();
		gzEmpInfoVo.setTPreEmpDeclaration(declaration);

		return R.ok(gzEmpInfoVo);
	}

	/**
	 * @Description: 瓜子档案表批量导出
	 * @Author: hgw
	 * @Date: 2025/6/19 10:03
	 * @return: void
	 **/
	@Override
	public void listExport(HttpServletResponse response, TGzEmpInfoSearchVo searchVo) {
		String prfName = "瓜子员工档案信息";
		String fileName = prfName + DateUtil.getThisTime() + ".xlsx";
		//获取要导出的列表
		List<TGzEmpInfoExportMain> list = new ArrayList<>();
		long count = baseMapper.getTGzEmpInfoCount(searchVo);

		ServletOutputStream out = null;
		try {
			out = response.getOutputStream();
			response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
			response.setCharacterEncoding("utf-8");
			response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName, CommonConstants.UTF8));
			Map<String,String> redisLabelMap = (Map<String, String>) RedisUtil.redis.opsForValue().get(CacheConstants.REDIS_DICT_LABLE);
			Map<String,String> idLabelMap = (Map<String, String>) RedisUtil.redis.opsForValue().get(CacheConstants.AREA_VALUE_ALL);
			if (Common.isEmpty(idLabelMap) || idLabelMap.isEmpty()) {
				R<AreaMap> areaListR = HttpDaprUtil.invokeMethodPost(daprUpmsProperties.getAppUrl(), daprUpmsProperties.getAppId()
						, "/area/inner/getAreaMapAll", "", AreaMap.class, SecurityConstants.FROM_IN);
				if (Common.isNotNull(areaListR)) {
					idLabelMap = areaListR.getData().getIdLabelMap();
				}
			}
			if (redisLabelMap != null && idLabelMap != null) {
				redisLabelMap.putAll(idLabelMap);
			}

			Map<String,String> nameAndDicTypeMap = new HashMap<>();
			ExcelUtil.getDictMapAndExp(TGzEmpInfoExportMain.class, nameAndDicTypeMap, redisLabelMap);
			ExcelUtil.getDictMapAndExp(TGzEmpInfoExportEducation.class, nameAndDicTypeMap, redisLabelMap);
			ExcelUtil.getDictMapAndExp(TGzEmpInfoExportFamily.class, nameAndDicTypeMap, redisLabelMap);
			// ExcelWriter excelWriter = EasyExcelFactory.write(out, TGzEmpInfoExportMain.class).registerConverter(new DictConverter(nameAndDicTypeMap, redisLabelMap)).build()
			ExcelWriter excelWriter = EasyExcelFactory.write(out).build();
			int index = 0;
			if (count > CommonConstants.ZERO_INT) {
				// 1：主表+紧急联系人
				WriteSheet writeSheet;
				int sheetIndex = 1;
				for (int i = 0; i <= count; i += CommonConstants.EXCEL_EXPORT_LIMIT) {
					// 获取实际记录
					searchVo.setLimitStart(i);
					searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
					list = baseMapper.getTGzEmpInfoList(searchVo);
					if (Common.isNotNull(list)) {
						writeSheet = EasyExcelFactory.writerSheet(index, "基础信息" + (sheetIndex++))
								.head(TGzEmpInfoExportMain.class)
								.registerConverter(new DictConverter(nameAndDicTypeMap, redisLabelMap)).build();
						excelWriter.write(list, writeSheet);
						index++;
					}
					if (Common.isNotNull(list)) {
						list.clear();
					}
				}

				// 2附属表：
				// 2.1教育经历
				count = baseMapper.getTGzEmpInfoCountEducation(searchVo);
				if (count > CommonConstants.ZERO_INT) {
					sheetIndex = 1;
					List<TGzEmpInfoExportEducation> listEducation;
					for (int i = 0; i <= count; i += CommonConstants.EXCEL_EXPORT_LIMIT) {
						// 获取实际记录
						searchVo.setLimitStart(i);
						searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
						listEducation = baseMapper.getTGzEmpInfoListEducation(searchVo);
						if (Common.isNotNull(listEducation)) {
							writeSheet = EasyExcelFactory.writerSheet(index,"教育经历" + (sheetIndex++)).head(TGzEmpInfoExportEducation.class)
									.registerConverter(new DictConverter(nameAndDicTypeMap, redisLabelMap)).build();
							excelWriter.write(listEducation, writeSheet);
							index++;
						}
						if (Common.isNotNull(listEducation)) {
							listEducation.clear();
						}
					}
				}
				// 2.2工作经历
				count = baseMapper.getTGzEmpInfoCountWork(searchVo);
				if (count > CommonConstants.ZERO_INT) {
					sheetIndex = 1;
					List<TGzEmpInfoExportWork> listWork;
					for (int i = 0; i <= count; i += CommonConstants.EXCEL_EXPORT_LIMIT) {
						// 获取实际记录
						searchVo.setLimitStart(i);
						searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
						listWork = baseMapper.getTGzEmpInfoListWork(searchVo);
						if (Common.isNotNull(listWork)) {
							writeSheet = EasyExcelFactory.writerSheet(index,"工作经历" + (sheetIndex++)).head(TGzEmpInfoExportWork.class).build();
							excelWriter.write(listWork, writeSheet);
							index++;
						}
						if (Common.isNotNull(listWork)) {
							listWork.clear();
						}
					}
				}
				// 2.3家庭成员
				count = baseMapper.getTGzEmpInfoCountFamily(searchVo);
				if (count > CommonConstants.ZERO_INT) {
					sheetIndex = 1;
					List<TGzEmpInfoExportFamily> listFamily;
					for (int i = 0; i <= count; i += CommonConstants.EXCEL_EXPORT_LIMIT) {
						// 获取实际记录
						searchVo.setLimitStart(i);
						searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
						listFamily = baseMapper.getTGzEmpInfoListFamily(searchVo);
						if (Common.isNotNull(listFamily)) {
							writeSheet = EasyExcelFactory.writerSheet(index,"家庭成员" + (sheetIndex++)).head(TGzEmpInfoExportFamily.class)
									.registerConverter(new DictConverter(nameAndDicTypeMap, redisLabelMap)).build();
							excelWriter.write(listFamily, writeSheet);
							index++;
						}
						if (Common.isNotNull(listFamily)) {
							listFamily.clear();
						}
					}
				}
				// 2.4技能证书
				count = baseMapper.getTGzEmpInfoCountSkill(searchVo);
				if (count > CommonConstants.ZERO_INT) {
					sheetIndex = 1;
					List<TGzEmpInfoExportSkill> listSkill;
					for (int i = 0; i <= count; i += CommonConstants.EXCEL_EXPORT_LIMIT) {
						// 获取实际记录
						searchVo.setLimitStart(i);
						searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
						listSkill = baseMapper.getTGzEmpInfoListSkill(searchVo);
						if (Common.isNotNull(listSkill)) {
							writeSheet = EasyExcelFactory.writerSheet(index,"技能证书" + (sheetIndex++)).head(TGzEmpInfoExportSkill.class).build();
							excelWriter.write(listSkill, writeSheet);
							index++;
						}
						if (Common.isNotNull(listSkill)) {
							listSkill.clear();
						}
					}
				}
			} else {
				WriteSheet writeSheet = EasyExcelFactory.writerSheet(prfName + index).build();
				excelWriter.write(list, writeSheet);
			}
			if (Common.isNotNull(list)) {
				list.clear();
			}
			out.flush();
			excelWriter.finish();
		} catch (Exception e) {
			log.error("执行异常", e);
		} finally {
			try {
				if (null != out) {
					out.close();
				}
			} catch (IOException e) {
				log.error("执行异常", e);
			}
		}
	}

	@Override
	public List<TGzEmpInfo> noPageDiy(TGzEmpInfoSearchVo searchVo) {
		LambdaQueryWrapper<TGzEmpInfo> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TGzEmpInfo::getId, idList);
		}
		if (searchVo.getLimitStart() >= 0 && searchVo.getLimitEnd() > 0) {
			wrapper.last(" limit " + searchVo.getLimitStart() + "," + searchVo.getLimitEnd());
		}
		wrapper.orderByDesc(BaseEntity::getCreateTime);
		return baseMapper.selectList(wrapper);
	}

	private Long noPageCountDiy(TGzEmpInfoSearchVo searchVo) {
		LambdaQueryWrapper<TGzEmpInfo> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TGzEmpInfo::getId, idList);
		}
		return baseMapper.selectCount(wrapper);
	}

	private LambdaQueryWrapper buildQueryWrapper(TGzEmpInfoSearchVo entity) {
		LambdaQueryWrapper<TGzEmpInfo> wrapper = Wrappers.lambdaQuery();

		if (Common.isNotNull(entity.getCreateName())) {
			wrapper.eq(TGzEmpInfo::getCreateName, entity.getCreateName());
		}
		return wrapper;
	}

	/**
	 * @Description: 瓜子保存新建用户
	 * @Author: hgw
	 * @Date: 2025/6/20 12:01
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser
	 **/
	private YifuUser getNewYifuUser() {
		Set<String> dbAuthsSet = new HashSet<>();
		Collection<? extends GrantedAuthority> authorities = AuthorityUtils
				.createAuthorityList(dbAuthsSet.toArray(new String[0]));
		return new YifuUser("3", 1L, "", "瓜子",
				"瓜子", "0", SecurityConstants.BCRYPT + "123456",
				"12345678911", true, true, true,
				true,
				"1", authorities, "1",
				null, null,
				null,null);
	}

	/**
	 * @Description: 保存瓜子档案表
	 * @Author: hgw
	 * @Date: 2025/6/19 18:14
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.Boolean>
	 **/
	@Override
	@Transactional
	public R<Boolean> saveTGzEmpInfo(TGzEmpInfoSearchVo tGzEmpInfoVo) {
		YifuUser user = getNewYifuUser();
		TGzEmpInfo tGzEmpInfo = tGzEmpInfoVo;
		Integer offerId = tGzEmpInfoVo.getOfferId();
		if (Common.isEmpty(offerId)) {
			return R.failed("请传参流程ID！");
		} else {
			TGzOfferInfo gzOfferInfo = gzOfferInfoMapper.getInfoById(offerId);
			if(gzOfferInfo == null || Common.isEmpty(gzOfferInfo) || Common.isEmpty(gzOfferInfo.getBizId())){
				return R.failed("未找到瓜子Offer数据，Offer的ID="+ offerId);
			}
			if (Common.isEmpty(gzOfferInfo.getOfferStatus()) || !"9".equals(gzOfferInfo.getOfferStatus())) {
				return R.failed("瓜子Offer状态不对（"+gzOfferInfo.getOfferStatus()+"），不可编辑，Offer的ID="+ offerId);
			}
			gzOfferInfo.setOfferStatus("12");
			gzOfferInfoMapper.updateById(gzOfferInfo);

			// 记录状态变更的offer操作日志 记录来源: 6：（系统触发）C端采集提交
			this.saveOfferStatusRecord(gzOfferInfo.getId(),CommonConstants.SIX_STRING,
					CommonConstants.NINE_STRING,gzOfferInfo.getOfferStatus(),"0","系统触发");
 		}
		String gzEmpId = tGzEmpInfoVo.getId();

		if (Common.isEmpty(gzEmpId)) {
			tGzEmpInfo.setCreateBy(user.getId());
			tGzEmpInfo.setCreateName(user.getNickname());
			tGzEmpInfo.setCreateTime(LocalDateTime.now());
			this.save(tGzEmpInfo);
			gzEmpId = tGzEmpInfo.getId();
		} else {
			tGzEmpInfo.setUpdateBy(user.getId());
			tGzEmpInfo.setUpdateTime(LocalDateTime.now());
			this.updateById(tGzEmpInfo);
		}
		// 获取瓜子档案附属信息：先删除后增加：
		// 紧急联系人
		gzEmpContactInfoMapper.deleteByEmpId(gzEmpId);
		List<TGzEmpContactInfo> emergContactList = tGzEmpInfoVo.getEmergContactList();
		if (emergContactList != null && !emergContactList.isEmpty()) {
			for (TGzEmpContactInfo a : emergContactList) {
				a.setId(null);
				a.setEmpId(gzEmpId);
				a.setCreateBy(user.getId());
				a.setCreateName(user.getNickname());
				a.setCreateTime(LocalDateTime.now());
				gzEmpContactInfoMapper.insert(a);
			}
		}
		// 教育经历
		gzEmpEducationMapper.deleteByEmpId(gzEmpId);
		List<TGzEmpEducation> eduExpList = tGzEmpInfoVo.getEduExpList();
		if (eduExpList != null && !eduExpList.isEmpty()) {
			for (TGzEmpEducation a : eduExpList) {
				a.setId(null);
				a.setEmpId(gzEmpId);
				a.setCreateBy(user.getId());
				a.setCreateName(user.getNickname());
				a.setCreateTime(LocalDateTime.now());
				gzEmpEducationMapper.insert(a);
			}
		}

		// 工作经历列表
		gzEmpWorkMapper.deleteByEmpId(gzEmpId);
		List<TGzEmpWork> workList = tGzEmpInfoVo.getWorkExpList();
		if (workList != null && !workList.isEmpty()) {
			for (TGzEmpWork a : workList) {
				a.setId(null);
				a.setEmpId(gzEmpId);
				a.setCreateBy(user.getId());
				a.setCreateName(user.getNickname());
				a.setCreateTime(LocalDateTime.now());
				gzEmpWorkMapper.insert(a);
			}
		}

		// 家庭成员列表
		gzEmpFamilyMapper.deleteByEmpId(gzEmpId);
		List<TGzEmpFamily> familyList = tGzEmpInfoVo.getFamilyList();
		if (familyList != null && !familyList.isEmpty()) {
			for (TGzEmpFamily a : familyList) {
				a.setId(null);
				a.setEmpId(gzEmpId);
				a.setCreateBy(user.getId());
				a.setCreateName(user.getNickname());
				a.setCreateTime(LocalDateTime.now());
				gzEmpFamilyMapper.insert(a);
			}
		}

		// 技能证书列表
		gzEmpSkillMapper.deleteByEmpId(gzEmpId);
		List<TGzEmpSkill> skillList = tGzEmpInfoVo.getSkillList();
		if (skillList != null && !skillList.isEmpty()) {
			for (TGzEmpSkill a : skillList) {
				a.setId(null);
				a.setEmpId(gzEmpId);
				a.setCreateBy(user.getId());
				a.setCreateName(user.getNickname());
				a.setCreateTime(LocalDateTime.now());
				gzEmpSkillMapper.insert(a);
			}
		}

		// 附件
		// * 41（瓜子申明签名）；42（瓜子确认签名）；43（瓜子pdf文件)
		//	 * CONTRACT（合同） NID_A（证件正面） NID_B（证件反面） BANK_FLOW（银行流水）
		//	 * EDUQ（学历） DEGRE（学位） PHOTO（头像） BANK_A（银行卡正面）
		//	 * BANK_B（银行卡反面） LEAVE_PRO（离职证明） EXAM_RPT（体检报告）
		attaInfoService.deleteByDomainId(gzEmpId);
		List<TAttaInfo> attaList = tGzEmpInfoVo.getAttaList();
		if (attaList != null && !attaList.isEmpty()) {
			for (TAttaInfo a : attaList) {
				a.setId(null);
				a.setDomainId(gzEmpId);
				a.setCreateBy(user.getId());
				a.setCreateName(user.getNickname());
				a.setCreateTime(LocalDateTime.now());
			}
			attaInfoService.saveBatch(attaList);
		}

		return R.ok();
	}

	/**
	 * 记录单个状态变更的offer操作日志
	 * @param offerId offerId
	 * @param recordFrom 记录来源
	 * @param beforeStatus 变更前状态
	 * @param nowStatus 当前状态
	 * @param userId 触发人ID
	 * @param nickname 触发人名称
	 */
	private void saveOfferStatusRecord(Integer offerId, String recordFrom,
									   String beforeStatus, String nowStatus, String userId, String nickname){
		// 记录offer操作日志
		LGuaziOfferRecordVo lGuaziOfferRecordVo = new LGuaziOfferRecordVo();
		// 存变更前后的状态到 状态数组里
		List<String> changeStatusList = new ArrayList<>();
		changeStatusList.add(beforeStatus);
		changeStatusList.add(nowStatus);

		if(changeStatusList.size() > 0){
			lGuaziOfferRecordVo.setOfferId(offerId);
			// 记录来源
			lGuaziOfferRecordVo.setRecordFrom(recordFrom);
			lGuaziOfferRecordVo.setChangeStatusList(changeStatusList);
			lGuaziOfferRecordVo.setCreateBy(userId);
			lGuaziOfferRecordVo.setCreateName(nickname);
			lGuaziOfferRecordService.saveRecordOne(lGuaziOfferRecordVo);
		}
	}

	/**
	 * @Description: 保存到人员档案
	 * @Author: hgw
	 * @Date: 2025/6/20 16:44
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Override
	public R<String> saveEmpInfoByGz(TGzOfferInfo offerInfo) {
		TSettleDomainSelectVo dept = tSettleDomainService.getSettleDomainVoByNo(CommonConstants.GZ_DEPT_NO);
		if (dept == null || Common.isEmpty(dept.getId())) {
			return R.failed("未找到瓜子项目！项目编码："+CommonConstants.GZ_DEPT_NO);
		}
		String deptId = dept.getId();
		if (Common.isEmpty(offerInfo.getId())) {
			return R.failed("请传参Offer的ID！");
		}
		YifuUser user = getNewYifuUser();
		// 保存到人员档案
		// 先判断信息是否存在：
		TGzEmpInfo gzEmpInfo = baseMapper.getInfoByOfferId(offerInfo.getId());
		if (gzEmpInfo == null || Common.isEmpty(gzEmpInfo.getNationalId())) {
			return R.failed("未找到瓜子档案信息！");
		}
		String gzEmpId = gzEmpInfo.getId();
		String empIdCard = gzEmpInfo.getNationalId();

		// 翻译字典
		Map<String, Map<String,String>> dictMap = new HashMap<>();
		R<Map<String, Map<String,String>>> relDictMapR = upmsDaprUtils.getRelValueMapByOutDictType();
		if (null != relDictMapR
				&& relDictMapR.getCode() == CommonConstants.SUCCESS
				&& Common.isNotNull(relDictMapR.getData())) {
			dictMap = relDictMapR.getData();
		}
		List<TGzEmpWork> workList = gzEmpWorkMapper.getListByEmpId(gzEmpId);

		TGzEmpEducation education = gzEmpEducationMapper.getOneByEmpId(gzEmpId);

		// 是否为首份工作 0是/1否
		String firstWorkFlag = (workList == null || workList.isEmpty()) ? CommonConstants.ZERO_STRING : CommonConstants.ONE_STRING;
				// 1：档案
		TEmployeeInfo emp = this.saveEmployeeInfo(gzEmpInfo, user, empIdCard, dictMap, firstWorkFlag, education);
		String empId = emp.getId();
		// 2：项目档案
		this.saveEmployeeProject(emp, user, empIdCard, deptId, dept);

		// 3:其他附属信息与附件
		// 3.1 工作：
		if (workList != null && !workList.isEmpty()) {
			// 先删除其他信息
			tEmpWorkRecordingService.deleteWorkRecordByEmpId(empId);
			TEmpWorkRecording info;
			for (TGzEmpWork gz : workList) {
				info = new TEmpWorkRecording();
				info.setWorkUnit(gz.getWorkUnit());
				info.setWorkDepart(gz.getWorkDept());
				info.setWorkJob(gz.getWorkJob());
				try {
					// 修改后代码：
					if (gz.getWorkStartDate() != null) {
						info.setStartDate(gz.getWorkStartDate().toInstant()
								.atZone(ZoneId.systemDefault())
								.toLocalDateTime());
					}
					if (gz.getWorkEndDate() != null) {
						info.setEndDate(gz.getWorkEndDate().toInstant()
								.atZone(ZoneId.systemDefault())
								.toLocalDateTime());
					}
				} catch (Exception e) {
					return R.failed("工作的开始结束日期转化失败！");
				}
				info.setDeleteFlag(CommonConstants.STATUS_NORMAL);
				info.setEmpId(empId);
				info.setEmpCode(emp.getEmpCode());
				info.setEmpName(emp.getEmpName());
				info.setEmpIdcard(emp.getEmpIdcard());
				info.setSettleDomain(deptId);
				if (Common.isEmpty(info.getCreateBy())) {
					info.setCreateBy(user.getId());
					info.setCreateName(user.getNickname());
					info.setCreateTime(LocalDateTime.now());
				}
				tEmpWorkRecordingService.save(info);
			}
		}
		// 家庭
		List<TGzEmpFamily> familyList = gzEmpFamilyMapper.getListByEmpId(gzEmpId);
		if (familyList != null && !familyList.isEmpty()) {
			// 先删除其他信息
			tEmpFamilyService.deleteByEmpId(empId);
			TEmpFamily info;
			for (TGzEmpFamily gz : familyList) {
				info = new TEmpFamily();

				info.setRelationshipSelf(getDictValue(dictMap, gz.getFamilyRelation(), ExcelAttributeConstants.GZ_FAMILY_RELATION));
				info.setWorkUnit(gz.getFamilyCompany());
				info.setFamilyName(gz.getFamilyName());
				info.setContractTel(gz.getFamilyPhone());

				info.setDeleteFlag(CommonConstants.STATUS_NORMAL);
				info.setEmpId(empId);
				info.setEmpCode(emp.getEmpCode());
				info.setEmpName(emp.getEmpName());
				info.setEmpIdcard(emp.getEmpIdcard());
				info.setSettleDomain(deptId);
				if (Common.isEmpty(info.getCreateBy())) {
					info.setCreateBy(user.getId());
					info.setCreateName(user.getNickname());
					info.setCreateTime(LocalDateTime.now());
				}
				tEmpFamilyService.save(info);
			}
		}
		// 学历
		if (education != null && Common.isNotNull(education.getId())) {
			TEmpEducation info = tEmpEducationService.getByEmpId(empId);
			if (info == null) {
				info = new TEmpEducation();
			}
			info.setEducationName(getDictValue(dictMap, education.getEduQual(), ExcelAttributeConstants.GZ_EDU_QUAL));
			info.setSchool(education.getSchool());
			info.setMajor(education.getMajor());
			info.setEntryDate(education.getEduStartDate());
			info.setGradutionDate(education.getEduEndDate());
			info.setHighIdentification(CommonConstants.ZERO_STRING);

			info.setDeleteFlag(CommonConstants.STATUS_NORMAL);
			info.setEmpId(empId);
			info.setEmpCode(emp.getEmpCode());
			info.setEmpName(emp.getEmpName());
			info.setEmpIdcard(emp.getEmpIdcard());
			info.setSettleDomain(deptId);
			if (Common.isEmpty(info.getCreateBy())) {
				info.setCreateBy(user.getId());
				info.setCreateName(user.getNickname());
				info.setCreateTime(LocalDateTime.now());
			}
			// 将所有的最高学历改为非最高
			if (Common.isNotNull(empIdCard)) {
				tEmpEducationService.updateEducationToNoByIdCard(empIdCard);
			}
			tEmpEducationService.saveOrUpdate(info);
		}
		// 紧急
		TGzEmpContactInfo contactInfo = gzEmpContactInfoMapper.getOneByEmpId(gzEmpId);
		if (contactInfo != null && Common.isNotNull(contactInfo.getId())) {
			TEmpContactInfo info = tEmpContactInfoService.getByEmpId(empId);
			if (info == null) {
				info = new TEmpContactInfo();
			}
			info.setRelationType(getDictValue(dictMap, contactInfo.getEmergRelation(), ExcelAttributeConstants.GZ_EMERG_RELATION));
			info.setEmpName(contactInfo.getEmergName());
			info.setTel(contactInfo.getEmergPhone());
			info.setAddress(contactInfo.getEmergAddress());

			info.setEmpId(empId);
			info.setEmpIdcard(emp.getEmpIdcard());
			if (Common.isEmpty(info.getCreateBy())) {
				info.setCreateBy(user.getId());
				info.setCreateName(user.getNickname());
				info.setCreateTime(LocalDateTime.now());
			}
			tEmpContactInfoService.saveOrUpdate(info);
		}
		// 附件
		List<TAttaInfo> attaList = attaInfoService.getTAttaInfoListByDoMainId(gzEmpId);
		if (attaList != null) {
			// 先删除其他信息
			attaInfoService.deleteByDomainIdAndOther(empId);
			TAttaInfo attaEmp;
			List<TAttaInfo> attaEmpList = new ArrayList<>();
			String idStr = "id";
			for (TAttaInfo atta : attaList) {
				attaEmp = new TAttaInfo();
				BeanUtil.copyProperties(atta, attaEmp, idStr);
				if (Common.isEmpty(attaEmp.getAttaName())) {
					attaEmp.setAttaName(CommonConstants.CENTER_SPLIT_LINE_STRING);
				}
				attaEmp.setDomainId(empId);
				attaEmp.setCreateBy(user.getId());
				attaEmp.setCreateName(user.getNickname());
				attaEmp.setCreateTime(LocalDateTime.now());
				attaEmpList.add(attaEmp);
			}
			attaInfoService.saveBatch(attaEmpList);
		}
		return R.ok();
	}

	private TEmployeeInfo saveEmployeeInfo(TGzEmpInfo gzEmpInfo, YifuUser user, String empIdCard
			, Map<String, Map<String,String>> dictMap, String firstWorkFlag, TGzEmpEducation education) {
		// 查找档案、项目档案，无则新建
		TEmployeeInfo emp = tEmployeeInfoService.getInfoByIdcard(empIdCard);
		if (emp == null) {
			emp = tEmployeeInfoService.getInfoByDelete(empIdCard);
			if (emp != null) {
				emp.setCreateTime(LocalDateTime.now());
			}
		} else if (CommonConstants.ONE_INT == emp.getFileStatus()) {
			emp.setCreateTime(LocalDateTime.now());
		}
		boolean isNew = false;
		if (emp == null) {
			isNew = true;
			emp = new TEmployeeInfo();
		}
		// 转化瓜子属性
		this.setEmpInfoByGz(gzEmpInfo, emp, dictMap, firstWorkFlag, education);
		// 新员工，塞基本属性
		if (isNew) {
			tEmployeeInfoService.setBaseData(emp, CommonConstants.NINE_STRING, user);
		}
		if (Common.isEmpty(emp.getEmpCode())) {
			emp.setEmpCode(tEmployeeInfoService.getCode());
		}
		emp.setUpdateBy(user.getId());
		emp.setUpdateTime(LocalDateTime.now());
		emp.setFileStatus(CommonConstants.ZERO_INT);
		emp.setDeleteFlag(CommonConstants.STATUS_NORMAL);
		tEmployeeInfoService.saveOrUpdate(emp);
		return emp;
	}

	private void saveEmployeeProject(TEmployeeInfo emp, YifuUser user, String empIdCard, String deptId, TSettleDomainSelectVo dept) {
		// 查找档案、项目档案，无则新建
		boolean isNew;
		TEmployeeProject project = tEmployeeProjectService.getByEmpIdCardAndDeptId(empIdCard, deptId);
		if (project == null) {
			project = tEmployeeProjectService.getByEmpIdCardAndDeptIdAndDelete(empIdCard, deptId);
			if (project != null) {
				project.setCreateTime(LocalDateTime.now());
				project.setProjectSource(CommonConstants.NINE_STRING);
				project.setDeleteFlag(CommonConstants.STATUS_NORMAL);
			}
		} else if (CommonConstants.ONE_INT == project.getProjectStatus()) {
			project.setCreateTime(LocalDateTime.now());
		}
		isNew = false;
		if (project == null) {
			isNew = true;
			project = new TEmployeeProject();
		}
		project.setProjectStatus(CommonConstants.ZERO_INT);
		if (Common.isEmpty(project.getStatus())) {
			project.setStatus(CommonConstants.ZERO_INT);
		}
		if (Common.isEmpty(project.getEmpId()) && emp != null) {
			project.setEmpId(emp.getId());
		}
		if (isNew) {
			// 新项目档案塞基本属性
			this.setNewEmpProjectBaseData(project, emp, dept, user);
		}
		tEmployeeProjectService.saveOrUpdate(project);
	}

	/**
	 * @Description: 新员工塞基本项目档案信息
	 * @Author: hgw
	 * @Date: 2024/6/19 14:26
	 * @return: void
	 **/
	private void setNewEmpProjectBaseData(TEmployeeProject insTEmployeePro, TEmployeeInfo tEmployeeInfo, TSettleDomain tSettleDomain, YifuUser user) {
		// 新增档案
		insTEmployeePro.setId(String.valueOf(UUID.randomUUID()).replace("-", ""));
		insTEmployeePro.setDeleteFlag(CommonConstants.STATUS_NORMAL);
		insTEmployeePro.setStatus(CommonConstants.ZERO_INT);

		//获取项目和单位信息
		if (Common.isNotNull(tSettleDomain)) {
			insTEmployeePro.setDeptId(tSettleDomain.getId());
			insTEmployeePro.setDeptName(tSettleDomain.getDepartName());
			insTEmployeePro.setBusinessPrimaryType(tSettleDomain.getBusinessPrimaryType());
			insTEmployeePro.setBusinessSecondType(tSettleDomain.getBusinessSecondType());
			insTEmployeePro.setBusinessThirdType(tSettleDomain.getBusinessThirdType());
			insTEmployeePro.setUnitId(tSettleDomain.getCustomerId());
			insTEmployeePro.setUnitNo(tSettleDomain.getCustomerNo());
			insTEmployeePro.setUnitName(tSettleDomain.getCustomerName());
			String empNO = tEmployeeProjectService.getEmpNo(insTEmployeePro.getDeptNo());
			insTEmployeePro.setEmpNo(empNO);
		}
		//员工编码生成规则
		if (tEmployeeInfo != null) {
			insTEmployeePro.setEmpId(tEmployeeInfo.getId());
			insTEmployeePro.setEmpName(tEmployeeInfo.getEmpName());
			insTEmployeePro.setEmpCode(tEmployeeInfo.getEmpCode());
		}
		insTEmployeePro.setCreateBy(user.getId());
		insTEmployeePro.setCreateTime(LocalDateTime.now());
		insTEmployeePro.setCreateName(user.getNickname());
		insTEmployeePro.setProjectSource(CommonConstants.NINE_STRING);
	}

	/**
	 * @Description: 转化瓜子属性
	 * @Author: hgw
	 * @Date: 2025/6/20 16:17
	 * @return: void
	 **/
	private void setEmpInfoByGz(TGzEmpInfo gzEmpInfo, TEmployeeInfo emp, Map<String, Map<String,String>> dictMap
			, String firstWorkFlag, TGzEmpEducation education) {

		// 按gzEmpInfo实体类顺序转化：
		emp.setEmpName(gzEmpInfo.getName());
		emp.setIdProvince(gzEmpInfo.getNativePlaceAreaId());
		emp.setIdCity(null);
		emp.setIdTown(null);
		// 民族
		emp.setEmpNational(getDictValue(dictMap, gzEmpInfo.getEthnicGrpCd(), ExcelAttributeConstants.GZ_ETHNIC_GRP_CD));
		// 婚姻状态
		emp.setEmpMarriStatus(getDictValue(dictMap, gzEmpInfo.getMarStatus(), ExcelAttributeConstants.GZ_MAR_STATUS));
		// 政治面貌
		emp.setPoliticalStatus(getDictValue(dictMap, gzEmpInfo.getPoliticalSta(), ExcelAttributeConstants.GZ_POLITICAL_STA));
		emp.setValidityStart(gzEmpInfo.getNidStartDate());
		emp.setValidityEnd(gzEmpInfo.getNidExpiryDate());
		emp.setEmpPhone(gzEmpInfo.getPhone());
		emp.setEmpEmail(gzEmpInfo.getMail());
		if (Common.isNotNull(gzEmpInfo.getStateAreaId())) {
			emp.setContactProvince(String.valueOf(gzEmpInfo.getStateAreaId()));
			if (Common.isNotNull(gzEmpInfo.getCityAreaId())) {
				emp.setContactCity(String.valueOf(gzEmpInfo.getCityAreaId()));
				if (Common.isNotNull(gzEmpInfo.getCountyAreaId())) {
					emp.setContactTown(String.valueOf(gzEmpInfo.getCountyAreaId()));
				} else {
					emp.setContactTown(null);
				}
			} else {
				emp.setContactCity(null);
				emp.setContactTown(null);
			}
		}
		emp.setContactAddress(gzEmpInfo.getAddress());
		// 户口性质
		emp.setEmpRegisType(getDictValue(dictMap, gzEmpInfo.getHukouType(), ExcelAttributeConstants.GZ_HUKOU_TYPE));

		emp.setFileProvince(gzEmpInfo.getHukouProvinceAreaId());
		emp.setFileCity(gzEmpInfo.getHukouCityAreaId());
		emp.setFileTown(gzEmpInfo.getHukouCountyAreaId());
		emp.setBankName(getDictValue(dictMap, gzEmpInfo.getBankCd(), ExcelAttributeConstants.GZ_BANK));
		emp.setBankSubName(gzEmpInfo.getBankName());
		if (Common.isNotNull(gzEmpInfo.getAccountProvinceAreaId())) {
			emp.setBankProvince(String.valueOf(gzEmpInfo.getAccountProvinceAreaId()));
		}
		if (Common.isNotNull(gzEmpInfo.getAccountCityAreaId())) {
			emp.setBankCity(String.valueOf(gzEmpInfo.getAccountCityAreaId()));
		}
		emp.setBankNo(gzEmpInfo.getAccountEcId());
		// 瓜子默认外包 2025-6-24 10:26:24 倩倩确认的
		emp.setEmpNatrue(CommonConstants.ZERO_STRING);

		if (education != null && Common.isNotNull(education.getId())) {
			emp.setHignEducation(getDictValue(dictMap, education.getEduQual(), ExcelAttributeConstants.GZ_EDU_QUAL));
			if ("1E".equals(education.getEduQual())) {
				emp.setIsCollege(CommonConstants.ZERO_INT);
			} else {
				emp.setIsCollege(CommonConstants.ONE_INT);
			}
			emp.setSchool(education.getSchool());
			emp.setMajor(education.getMajor());
			emp.setAdmissionDate(education.getEduStartDate());
			emp.setGradutionDate(education.getEduEndDate());
		}
		// 按照是否有工作经历来判断：无，则是首份工作
		emp.setFirstWorkFlag(firstWorkFlag);
		// 2025-6-24 10:38:06 倩倩：因为 瓜子的附属信息：技能证书 不同步MVP，因此这里默认否
		emp.setHaveQualification(CommonConstants.ONE_STRING);


	}

	/**
	 * @param dictMap 总字典Map
	 * @param gzValue 瓜子字段值
	 * @param dictType 字典类型
	 * @Description: 通用获取字典值的方法
	 * @Author: hgw
	 * @Date: 2025/6/23 17:56
	 * @return: java.lang.String
	 **/
	private String getDictValue(Map<String, Map<String, String>> dictMap, String gzValue, String dictType) {
		if (Common.isNotNull(gzValue)) {
			Map<String, String> dictValueMap = dictMap.get(dictType);
			if (dictValueMap != null && dictValueMap.get(gzValue) != null) {
				return dictValueMap.get(gzValue);
			}
		}
		return null;
	}

}
