/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.pig4cloud.plugin.excel.vo.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeInfo;
import com.yifu.cloud.plus.v1.yifu.archives.service.TEmployeeInfoService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.*;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CacheConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.ResultConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.redis.RedisDistributedLock;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.config.DaprUpmsProperties;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.MenuUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * 人员档案表
 *
 * @author hgw
 * @date 2022-06-20 09:55:06
 */
@EnableConfigurationProperties({DaprUpmsProperties.class})
@RestController
@RequiredArgsConstructor
@RequestMapping("/temployeeinfo")
@Tag(name = "人员档案表管理")
public class TEmployeeInfoController {

	private final TEmployeeInfoService tEmployeeInfoService;

	private final MenuUtil menuUtil;

	/**
	 * 分页查询
	 *
	 * @param page          分页对象
	 * @param tEmployeeInfo 人员档案表
	 * @return
	 */
	@Operation(summary = "分页查询", description = "分页查询")
	@GetMapping("/page")
	public R<IPage<TEmployeeInfo>> getTEmployeeInfoPage(Page<TEmployeeInfo> page, TEmployeeInfo tEmployeeInfo) {
		YifuUser user = SecurityUtils.getUser();
		if (user == null || Common.isEmpty(user.getId())) {
			return R.failed(CommonConstants.PLEASE_LOG_IN);
		}
		if (Common.isEmpty(tEmployeeInfo.getMId())) {
			tEmployeeInfo.setMId("1536922631616278529");
		}
		menuUtil.setAuthSql(user, tEmployeeInfo);
		if (Common.isNotNull(tEmployeeInfo.getAuthSql()) && tEmployeeInfo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
			tEmployeeInfo.setAuthSql(tEmployeeInfo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "b.DEPT_ID"));
		}
		return R.ok(tEmployeeInfoService.getPage(page, tEmployeeInfo));
	}


	/**
	 * 分页查询
	 *
	 * @param page          分页对象
	 * @param tEmployeeInfo 人员档案表
	 * @return
	 */
	@Operation(summary = "分页查询", description = "分页查询")
	@GetMapping("/pageAsso")
	public R<IPage<TEmployeeInfo>> getTEmployeeInfoPagepageAsso(Page<TEmployeeInfo> page, TEmployeeInfo tEmployeeInfo) {
		return R.ok(tEmployeeInfoService.getPageAsso(page, tEmployeeInfo));
	}

	/**
	 * 分页查询
	 *
	 * @param page          分页对象
	 * @param tEmployeeInfo 人员档案表
	 * @return
	 */
	@Operation(summary = "离职库分页查询", description = "离职库分页查询")
	@GetMapping("/getLeavePage")
	public R<IPage<TEmployeeInfo>> getLeavePage(Page<TEmployeeInfo> page, TEmployeeInfo tEmployeeInfo) {
		YifuUser user = SecurityUtils.getUser();
		if (user == null || Common.isEmpty(user.getId())) {
			return R.failed(CommonConstants.PLEASE_LOG_IN);
		}
		if (Common.isEmpty(tEmployeeInfo.getMId())) {
			tEmployeeInfo.setMId("1536952884128591873");
		}
		menuUtil.setAuthSql(user, tEmployeeInfo);
		if (Common.isNotNull(tEmployeeInfo.getAuthSql()) && tEmployeeInfo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
			tEmployeeInfo.setAuthSql(tEmployeeInfo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "b.DEPT_ID"));
			if (tEmployeeInfo.getAuthSql().contains("1=2  or ")) {
				tEmployeeInfo.setAuthSql(tEmployeeInfo.getAuthSql().replace("1=2  or ", ""));
			}
		}
		return R.ok(tEmployeeInfoService.getLeavePage(page, tEmployeeInfo));
	}


	/**
	 * 通过id查询人员档案表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询")
	@GetMapping("/{id}")
	public R<TEmployeeInfo> getById(@PathVariable("id") String id) {
		return tEmployeeInfoService.getByIdAssociate(id);
	}

	/**
	 * @param empId
	 * @Description: 内部接口获取档案基本信息
	 * @Author: hgw
	 * @Date: 2022/7/21 14:47
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeInfo>
	 **/
	@Inner
	@PostMapping("/inner/getByEmpId")
	public TEmployeeInfo getByEmpId(@RequestBody String empId) {
		return tEmployeeInfoService.getById(empId);
	}

	/**
	 * @param idcard
	 * @Description: 内部接口获取档案基本信息
	 * @Author: hgw
	 * @Date: 2022/7/21 14:47
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeInfo>
	 **/
	@Inner
	@PostMapping("/inner/getTEmployeeInfoByIdcard")
	public TEmployeeInfo getTEmployeeInfoByIdcard(@RequestBody String idcard) {
		return tEmployeeInfoService.getInfoByIdcard(idcard);
	}

	/**
	 * @param empName   姓名
	 * @param empIdCard 身份证
	 * @Description: 新增人员档案前的校验
	 * @Author: hgw
	 * @Date: 2022/6/21 11:53
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.Boolean>
	 **/
	@Operation(summary = "新增人员档案前的校验", description = "新增人员档案前的校验")
	@SysLog("新增人员档案前的校验")
	@GetMapping("/checkIdCard")
	//@PreAuthorize("@pms.hasPermission('temployeeinfo_add')")
	public R<TEmployeeInfo> checkIdCard(@RequestParam String empName, @RequestParam String empIdCard) {
		TEmployeeInfo employeeInfo = new TEmployeeInfo();
		employeeInfo.setEmpName(empName);
		employeeInfo.setEmpIdcard(empIdCard);
		employeeInfo.setDeleteFlag(CommonConstants.STATUS_NORMAL);
		return tEmployeeInfoService.checkIdCard(employeeInfo);
	}

	/**
	 * 新增人员档案表
	 *
	 * @param tEmployeeInfo 人员档案表
	 * @return R
	 */
	@Operation(summary = "新增人员档案表（附带项目档案、学历）", description = "新增人员档案表")
	@SysLog("新增人员档案表（附带项目档案、学历）")
	@PostMapping("/addNewEmployee")
	//@PreAuthorize("@pms.hasPermission('temployeeinfo_add')")
	public R<String> addNewEmployee(@Valid @RequestBody TEmployeeInfo tEmployeeInfo) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		// 获取redis分布式事务锁
		String key = CacheConstants.EMP_ADD_LOCK + CommonConstants.DOWN_LINE_STRING + user.getId();
		String requestId;
		try {
			requestId = RedisDistributedLock.getLock(key,"10");
		} catch (Exception e) {
			throw new RuntimeException(ResultConstants.NO_GETLOCK_DATA+CommonConstants.DOWN_LINE_STRING+e.getMessage());
		}
		try {
			if (Common.isNotNull(requestId)) {
				return tEmployeeInfoService.addNewEmployee(tEmployeeInfo);
			} else {
				return R.failed(ResultConstants.NO_GETLOCK_DATA);
			}
		}finally {
			//主动释放锁
			RedisDistributedLock.unlock(key, requestId);
		}

	}

	/**
	 * 修改人员档案表
	 *
	 * @param tEmployeeInfo 人员档案表
	 * @return R
	 */
	@Operation(summary = "修改人员档案表", description = "修改人员档案表")
	@SysLog("修改人员档案表")
	@PostMapping("/updateEmployee")
	@PreAuthorize("@pms.hasPermission('temployeeinfo_edit')")
	public R<Boolean> updateEmployee(@RequestBody TEmployeeInfo tEmployeeInfo) {
		return tEmployeeInfoService.updateEmployee(tEmployeeInfo);
	}

	/**
	 * @param id
	 * @Description: 通过id删除人员档案表
	 * @Author: hgw
	 * @Date: 2022/6/21 19:42
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "通过id删除人员档案表", description = "通过id删除人员档案表")
	@SysLog("通过id删除人员档案表")
	@GetMapping("/deleteEmployee")
	@PreAuthorize("@pms.hasPermission('temployeeinfo_del')")
	public R<String> deleteEmployee(@RequestParam String id) {
		return tEmployeeInfoService.deleteEmployee(id);
	}

	/**
	 * @param idList
	 * @Description: 通过ids删除人员档案表
	 * @Author: hgw
	 * @Date: 2022/6/21 19:42
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "通过idList批量删除人员档案表", description = "通过ids批量删除人员档案表")
	@SysLog("通过idList批量删除人员档案表")
	@PostMapping("/batchDeleteEmployee")
	@PreAuthorize("@pms.hasPermission('temployeeinfo_batch_del')")
	public R<List<ErrorMessageVO>> batchDeleteEmployee(@RequestBody List<String> idList) {
		return tEmployeeInfoService.batchDeleteEmployee(idList);
	}

	/**
	 * @param id
	 * @Description: 通过id减档人员档案表
	 * @Author: hgw
	 * @Date: 2022/6/21 19:42
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "通过id减档人员档案表temployeeinfo_minus", description = "通过id减档人员档案表")
	@SysLog("通过id减档人员档案表")
	@GetMapping("/minusEmployee")
	@PreAuthorize("@pms.hasPermission('temployeeinfo_minus')")
	public R<String> minusEmployee(@RequestParam String id, @RequestParam String leaveReason
			, @RequestParam(required = false) String leaveRemark) {
		return tEmployeeInfoService.minusEmployee(id, leaveReason, leaveRemark);
	}

	/**
	 * @Description: 通过ids批量减档人员档案表
	 * @Author: hgw
	 * @Date: 2022/6/21 19:42
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@SneakyThrows
	@Operation(summary = "批量减档人员档案表temployeeinfo_batch_minus", description = "批量减档人员档案表")
	@SysLog("批量减档人员档案表")
	@PostMapping("/batchMinusEmployee")
	@PreAuthorize("@pms.hasPermission('temployeeinfo_batch_minus')")
	public R<List<ErrorMessageVO>> batchMinusEmployee(@RequestBody MultipartFile file) {
		return tEmployeeInfoService.batchMinusEmployee(file.getInputStream());
	}

	/**
	 * @param id         离职档案id
	 * @param projectIds 需要复项的项目ids
	 * @Description: 档案复档
	 * @Author: hgw
	 * @Date: 2022/6/23 17:40
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "档案复档", description = "档案复档 projectIds：项目档案ids")
	@SysLog("档案复档")
	@GetMapping("/restoreEmployee")
	public R<String> restoreEmployee(@RequestParam String id, String projectIds) {
		return tEmployeeInfoService.restoreEmployee(id, projectIds);
	}

	/**
	 * @Description: 批量导入人员档案
	 * @Author: hgw
	 * @Date: 2022/6/22 11:57
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.pig4cloud.plugin.excel.vo.ErrorMessage>>
	 **/
	@SneakyThrows
	@Operation(summary = "批量导入人员档案", description = "批量导入人员档案 hasPermission('employee_batch_import') 额外属性：isCanAdd：0不可以复档复项；1允许复档复项")
	@SysLog("批量导入人员档案")
	@PostMapping("/batchImportEmployee")
	@PreAuthorize("@pms.hasPermission('employee_batch_import')")
	public R<List<ErrorMessage>> batchImportEmployee(@RequestBody MultipartFile file, @RequestParam Integer isCanAdd) {
		return tEmployeeInfoService.batchImportEmployee(file.getInputStream(), isCanAdd);
	}

	/**
	 * @Description: 批量更新人员档案
	 * @Author: hgw
	 * @Date: 2022/6/22 20:29
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.pig4cloud.plugin.excel.vo.ErrorMessage>>
	 **/
	@SneakyThrows
	@Operation(summary = "批量更新人员档案", description = "批量更新人员档案 hasPermission('employee_batch_update')")
	@SysLog("批量更新人员档案")
	@PostMapping("/batchUpdateEmployee")
	@PreAuthorize("@pms.hasPermission('employee_batch_update')")
	public R<List<ErrorMessage>> batchUpdateEmployee(@RequestBody MultipartFile file,
													 @RequestParam(name = "exportFields", required = true) String[] exportFields) {
		return tEmployeeInfoService.batchUpdateEmployee(file.getInputStream(),exportFields);
	}

	/**
	 * @Description: 批量更新人员和银行信息
	 * @Author: hgw
	 * @Date: 2024-10-16 17:41:04
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.pig4cloud.plugin.excel.vo.ErrorMessage>>
	 **/
	@SneakyThrows
	@Operation(summary = "批量更新人员和银行信息", description = "批量更新人员和银行信息 hasPermission('employee_batch_update_bank')")
	@SysLog("批量更新人员和银行信息")
	@PostMapping("/batchUpdateEmployeeAndBank")
	// @PreAuthorize("@pms.hasPermission('employee_batch_update_bank')")
	public R<List<PreUploadEmpImportTwoVo>> batchUpdateEmployeeAndBank(MultipartFile file, PreUploadImportParamVo vo) throws Exception {
		return tEmployeeInfoService.batchUpdateEmployeeAndBank(file, vo);
	}

	/**
	 * 批量导出人员档案信息
	 *
	 * @Author hgw
	 * @Date 2022-6-24 09:36:06
	 **/
	@Operation(summary = "批量导出人员档案信息", description = "批量导出人员档案信息 hasPermission('employee_export_employee')")
	@SysLog("批量导出人员档案信息")
	// @ResponseExcel
	@PostMapping("/exportEmployee")
	public void exportEmployee(@RequestBody(required = false) EmployeeInfoSearchVo searchVo, HttpServletResponse response) {
		YifuUser user = SecurityUtils.getUser();
		if (user != null && Common.isNotNull(user.getId())) {
			menuUtil.setAuthSql(user, searchVo);
			if (Common.isNotNull(searchVo.getAuthSql()) && searchVo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
				searchVo.setAuthSql(searchVo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "b.DEPT_ID"));
			}
		}
		tEmployeeInfoService.exportEmployee(searchVo, response);
	}

	/**
	 * 批量导出离职档案信息
	 *
	 * @Author hgw
	 * @Date 2022-6-24 09:36:06
	 **/
	@Operation(summary = "批量导出离职档案信息", description = "批量导出离职档案信息 hasPermission('employee_export_employee_leave')")
	@SysLog("批量导出离职档案信息")
	// @ResponseExcel
	@PostMapping("/exportLeaveEmployee")
	public void exportLeaveEmployee(@RequestBody(required = false) EmployeeInfoSearchVo employeeInfo, HttpServletResponse response) {
		YifuUser user = SecurityUtils.getUser();
		if (user != null && Common.isNotNull(user.getId())) {
			menuUtil.setAuthSql(user, employeeInfo);
			if (Common.isNotNull(employeeInfo.getAuthSql()) && employeeInfo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
				employeeInfo.setAuthSql(employeeInfo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "b.DEPT_ID"));
			}
		}
		tEmployeeInfoService.exportLeaveEmployee(employeeInfo, response);
	}

	/**
	 * 通过身份和编码查询档案 项目档案 合同信息
	 *
	 * @Author hgw
	 * @Date 2022-6-24 09:36:06
	 **/
	@Operation(description = "通过身份和编码查询档案 项目档案 合同信息")
	@SysLog("通过身份和编码查询档案 项目档案 合同信息")
	@Inner
	@PostMapping("/getDispatchEmpVo")
	public Map<String, DispatchEmpVo> getDispatchEmpVo(@RequestBody(required = true)DispatchCheckVo checkVo) {
		return  tEmployeeInfoService.getDispatchEmpVo(checkVo);
	}

	/**
	 * @Author fxj
	 * @Description 新增档案|项目档案|合同信息
	 * @Date 21:05 2022/7/20
	 * @Param
	 * @return
	**/
	@Operation(description = "新增档案|项目档案|合同信息")
	@SysLog("新增档案|项目档案|合同信息")
	@Inner
	@PostMapping("/addDispatchInfo")
	public EmpDispatchAddVo addDispatchInfo(@RequestBody(required = true) EmpDispatchAddVo addVo) {
		return  tEmployeeInfoService.addDispatchInfo(addVo);
	}
	/**
	 * @Description: 派单审核通过 更新档案 合同  项目档案审核状态
	 * @Author: fxj
	 * @Date: 2022/08/01 18:52
	 **/
	@Operation(summary = "通过EMP_ID查询人员档案变更日志", description = "通过EMP_ID查询人员档案变更日志")
	@Inner
	@PostMapping("/inner/updateEmpInfo")
	public Boolean updateEmpInfo(@RequestBody UpdateEmpVo vo) throws IllegalAccessException {
		return tEmployeeInfoService.updateEmpInfo(vo);
	}

	/**
	 * @Description: 查验手机号码是否存在
	 * @Author: fxj
	 * @Date: 2022/08/01 18:52
	 **/
	@Operation(summary = "通过mobile查询是否已存在", description = "通过mobile查询是否已存在")
	@Inner
	@PostMapping("/inner/checkMobile")
	public Boolean checkMobile(@RequestBody String mobile) {
		return tEmployeeInfoService.checkMobile(mobile);
	}

	/**
	 * @Description: 更新人员档案的商险状态
	 * @Author: hgw
	 * @Date: 2022-11-23 17:17:59
	 **/
	@Operation(summary = "更新人员档案的商险状态", description = "更新人员档案的商险状态")
	@Inner
	@PostMapping("/inner/updateEmployeeInfoInsuranceStatus")
	public Boolean updateEmployeeInsuranceStatus(@RequestBody EmpStatusVo vo) {
		return tEmployeeInfoService.updateEmployeeInfoInsuranceStatus(vo);
	}

	/**
	 * @Description: 更新人员档案的薪资状态
	 * @Author: hgw
	 * @Date: 2022-11-23 17:17:59
	 **/
	@Operation(summary = "更新人员档案的薪资状态", description = "更新人员档案的薪资状态")
	@Inner
	@PostMapping("/inner/updateEmployeeInfoSalaryStatus")
	public Boolean updateEmployeeInfoSalaryStatus(@RequestBody EmpStatusVo vo) {
		return tEmployeeInfoService.updateEmployeeInfoSalaryStatus(vo);
	}

	/**
	 * @Description: 每月1号更新人员档案、项目档案的薪资状态为2否
	 * @Author: hgw
	 * @Date: 2022/11/24 15:00
	 * @return: void
	 **/
	@Operation(summary = "每月1号更新人员档案、项目档案的薪资状态为2否", description = "每月1号更新人员档案、项目档案的薪资状态为2否")
	@Inner
	@PostMapping("/inner/everyMonthUpdateSalaryStatus")
	public void everyMonthUpdateSalaryStatus() {
		tEmployeeInfoService.everyMonthUpdateSalaryStatus();
	}

	/**
	 * @Description: 每天更新人员档案、项目档案的过期合同为2不在用
	 * @Author: hgw
	 * @Date: 2022/11/24 15:00
	 * @return: void
	 **/
	@Operation(summary = "每天更新人员档案、项目档案的过期合同为2不在用", description = "每天更新人员档案、项目档案的过期合同为2不在用")
	@Inner
	@PostMapping("/inner/everyDayUpdateEmployeeContractStatus")
	public void everyDayUpdateEmployeeContractStatus() {
		tEmployeeInfoService.everyDayUpdateEmployeeContractStatus();
	}

	/**
	 * @param idCard
	 * @Description: HCM获取人员信息
	 * @Author: hgw
	 * @Date: 2023/11/23 11:42
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<com.yifu.cloud.plus.v1.yifu.archives.vo.HcmEmployeeInfoVO>
	 **/
	@Operation(description = "HCM获取人员信息")
	@SysLog("HCM获取人员信息--HCM调用接口")
	@GetMapping("/getHcmEmpInfo")
	public R<HcmEmployeeInfoVO> getHcmEmpInfo(@RequestParam String idCard) {
		return tEmployeeInfoService.getHcmEmpInfo(idCard);
	}

	@Operation(description = "测试异常数据使用接口")
	@SysLog("测试异常数据使用接口")
	@PostMapping("/getListByDeptIdByLimit")
	public List<TEmployeeInfo> getListByDeptIdByLimit(@RequestParam String deptId,@RequestParam Integer limit1, @RequestParam Integer limit2) {
		return tEmployeeInfoService.getListByDeptIdByLimit(deptId, limit1,limit2);
	}

	/**
	 * @param idCardList
	 * @Description: HCM获取人员信息列表
	 * @Author: hgw
	 * @Date: 2023/11/23 11:43
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.yifu.cloud.plus.v1.yifu.archives.vo.HcmEmployeeInfoVO>>
	 **/
	@Operation(description = "HCM获取人员信息列表")
	@SysLog("HCM获取人员信息列表--HCM调用接口")
	@PostMapping("/getHcmEmpInfoList")
	public R<List<HcmEmployeeInfoVO>> getHcmEmpInfoList(@RequestBody List<String> idCardList) {
		return tEmployeeInfoService.getHcmEmpInfoList(idCardList);
	}

	/**
	 * @Author fxj
	 * @Description 每年1月1号刷新在职和离职人员的年龄信息  公式：当前年-出生年
	 * @Date 17:29 2024/8/21
	**/
	@Operation(summary = "每年1月1号刷新在职和离职人员的年龄信息", description = "每年1月1号刷新在职和离职人员的年龄信息")
	@Inner
	@PostMapping("/inner/updateEmployeeAgeTask")
	public void updateEmployeeAgeTask() {
		tEmployeeInfoService.updateEmployeeAgeTask();
	}

	/**
	 * @Author fxj
	 * @Description 人员档案附件信息导出
	 * @Date 15:15 2024/9/10
	 * @Param
	 * @return
	**/
	@Operation(summary = "导出人员档案附件信息ZIP包", description = "导出人员档案附件信息ZIP包")
	@SysLog("导出人员档案附件信息ZIP包")
	@PostMapping("/exportEmployeeFile/{id}")
	public void exportEmployeeFile(@PathVariable("id") String id, HttpServletResponse response) throws Exception{
		tEmployeeInfoService.exportEmployeeFile(id, response);
	}

	/**
	 * @param empName   姓名
	 * @param empIdCard 身份证
	 * @Description: 按姓名身份证查询档案-包括离职库
	 * @Author: hgw
	 * @Date: 2025-4-23 15:34:26
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "按姓名身份证查询档案-包括离职库", description = "按姓名身份证查询档案-包括离职库")
	@SysLog("按姓名身份证查询档案-包括离职库")
	@GetMapping("/getByEmpNameAndCard")
	public R<TEmployeeInfo> getByEmpNameAndCard(@RequestParam String empName, @RequestParam String empIdCard) {
		return tEmployeeInfoService.getByEmpNameAndCard(empName, empIdCard);
	}

	/**
	 * @param empPhoneOld 手机号-原先
	 * @param empPhoneNew 手机号-新
	 * @Description: 更新档案手机号-包括离职库
	 * @Author: hgw
	 * @Date: 2025-4-23 15:34:26
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "更新档案手机号-包括离职库", description = "更新档案手机号-包括离职库")
	@SysLog("更新档案手机号-包括离职库")
	@GetMapping("/updateEmpPhone")
	public R<String> updateEmpPhone(@RequestParam String id, @RequestParam String empPhoneOld, @RequestParam String empPhoneNew) {
		return tEmployeeInfoService.updateEmpPhone(id, empPhoneOld, empPhoneNew);
	}
	/**
	 * 远程接口获取非在职档案（无档案或已离职）
	 */
	@Inner
	@PostMapping("/inner/getEmpOnlineMap")
	public Set<String> getCustomerUserMap(@RequestBody List<String> empIdCardList) {
		return tEmployeeInfoService.getEmpOnlineMap(empIdCardList);
	}
}
