/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.entity.SysAutoDictItem;
import com.yifu.cloud.plus.v1.yifu.archives.service.SysAutoDictItemService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.SysAutoDictItemSearchVo;
import com.yifu.cloud.plus.v1.yifu.archives.vo.SysAutoDictItemVo;
import com.yifu.cloud.plus.v1.yifu.archives.vo.TSettleDomainSelectVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 字典项
 *
 * @author fxj
 * @date 2025-03-13 15:15:26
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/sysautodictitem" )
@Tag(name = "字典项管理")
public class SysAutoDictItemController {

    private final  SysAutoDictItemService sysAutoDictItemService;


    /**
     * 简单分页查询
     * @param page 分页对象
     * @param sysAutoDictItem 字典项
     * @return
     */
    @Operation(description = "简单分页查询")
    @GetMapping("/page")
    public R<IPage<SysAutoDictItem>> getSysAutoDictItemPage(Page<SysAutoDictItem> page, SysAutoDictItemSearchVo sysAutoDictItem) {
        return  new R<>(sysAutoDictItemService.getSysAutoDictItemPage(page,sysAutoDictItem));
    }
    /**
     * 不分页查询
     * @param sysAutoDictItem 字典项
     * @return
     */
    @Operation(summary = "不分页查询", description = "不分页查询")
    @PostMapping("/noPage" )
    //@PreAuthorize("@pms.hasPermission('demo_sysautodictitem_get')" )
    public R<List<SysAutoDictItem>> getSysAutoDictItemNoPage(@RequestBody SysAutoDictItemSearchVo sysAutoDictItem) {
        return R.ok(sysAutoDictItemService.noPageDiy(sysAutoDictItem));
    }

    /**
     * 通过id查询字典项
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id查询", description = "通过id查询：hasPermission('demo_sysautodictitem_get')")
    @GetMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_sysautodictitem_get')" )
    public R<SysAutoDictItem> getById(@PathVariable("id" ) Long id) {
        return R.ok(sysAutoDictItemService.getById(id));
    }

    /**
     * 新增字典项
     * @param sysAutoDictItem 字典项
     * @return R
     */
    @Operation(summary = "新增字典项", description = "新增字典项：hasPermission('demo_sysautodictitem_add')")
    @SysLog("新增字典项" )
    @PostMapping
    @PreAuthorize("@pms.hasPermission('demo_sysautodictitem_add')" )
    public R<Boolean> save(@RequestBody SysAutoDictItem sysAutoDictItem) {
        return R.ok(sysAutoDictItemService.save(sysAutoDictItem));
    }

    /**
     * 修改字典项
     * @param sysAutoDictItem 字典项
     * @return R
     */
    @Operation(summary = "修改字典项", description = "修改字典项：hasPermission('demo_sysautodictitem_edit')")
    @SysLog("修改字典项" )
    @PutMapping
    @PreAuthorize("@pms.hasPermission('demo_sysautodictitem_edit')" )
    public R<Boolean> updateById(@RequestBody SysAutoDictItem sysAutoDictItem) {
        return R.ok(sysAutoDictItemService.updateById(sysAutoDictItem));
    }

    /**
     * 通过id删除字典项
     * @param id id
     * @return R
     */
    @Operation(summary = "通过id删除字典项", description = "通过id删除字典项：hasPermission('demo_sysautodictitem_del')")
    @SysLog("通过id删除字典项" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('demo_sysautodictitem_del')" )
    public R<Boolean> removeById(@PathVariable Long id) {
        return R.ok(sysAutoDictItemService.removeById(id));
    }

    /**
	 * 字典项 批量导入
	 *
     * @author fxj
     * @date 2025-03-13 15:15:26
	 **/
    @SneakyThrows
    @Operation(description = "批量新增字典项 hasPermission('demo_sysautodictitem-batch-import')")
    @SysLog("批量新增字典项")
    @PostMapping("/importListAdd")
    @PreAuthorize("@pms.hasPermission('demo_sysautodictitem-batch-import')")
    public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file){
        return sysAutoDictItemService.importDiy(file.getInputStream());
    }

    /**
     * 字典项 批量导出
     * @author fxj
     * @date 2025-03-13 15:15:26
     **/
    @Operation(description = "导出字典项 hasPermission('demo_sysautodictitem-export')")
    @PostMapping("/export")
    @PreAuthorize("@pms.hasPermission('demo_sysautodictitem-export')")
    public void export(HttpServletResponse response, @RequestBody SysAutoDictItemSearchVo searchVo) {
            sysAutoDictItemService.listExport(response,searchVo);
    }

	/**
	 * 获取项目配置下的字典列表
	 * @param sysAutoDictItem 字典项
	 * @return R<List<SysAutoDictItem>>
	 * @author chenyuxi
	 */
	@Operation(summary = "获取项目配置下的字典列表")
	@GetMapping("/selectByDeptNo" )
	public R<List<SysAutoDictItem>> selectByDeptNo(SysAutoDictItemSearchVo sysAutoDictItem) {
		if(Common.isEmpty(sysAutoDictItem.getDeptNo())){
			return R.failed("项目编号不能为空");
		}
		return R.ok(sysAutoDictItemService.selectByDeptNo(sysAutoDictItem));
	}

	@Inner
	@PostMapping("/getSettleDomainSelectVoById")
	public SysAutoDictItemVo getInnerSettleDomainSelectVoByUserId(@RequestBody SysAutoDictItemVo dictItemVo) {
		SysAutoDictItemVo settleDomainSelectVo = new SysAutoDictItemVo();
		LambdaQueryWrapper<SysAutoDictItem> dictQuery = Wrappers.<SysAutoDictItem>lambdaQuery()
				.eq(SysAutoDictItem::getDeptNo, dictItemVo.getDeptNo())
				.eq(SysAutoDictItem::getDelFlag, CommonConstants.NOT_DELETE_FLAG);

		List<SysAutoDictItem> list = sysAutoDictItemService.list(dictQuery);
		settleDomainSelectVo.setSysAutoDictItemList(list);
		return settleDomainSelectVo;
	}
}
