/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.archives.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ArrayUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.archives.entity.*;
import com.yifu.cloud.plus.v1.yifu.archives.mapper.*;
import com.yifu.cloud.plus.v1.yifu.archives.service.TAutoMainRelService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.*;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.ServiceNameConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.*;
import com.yifu.cloud.plus.v1.yifu.common.core.util.equator.HrEquator;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.ArchivesDaprUtil;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.serializer.SerializerFeature.*;

/**
 * 项目配置表主表
 *
 * @author fxj
 * @date 2025-03-13 15:15:26
 */
@Log4j2
@Service
public class TAutoMainRelServiceImpl extends ServiceImpl<TAutoMainRelMapper, TAutoMainRel> implements TAutoMainRelService {

	@Autowired
	private TAutoEmpRuleInfoMapper autoEmpRuleInfoMapper;

	@Autowired
	private TAutoRuleLogMapper autoRuleLogMapper;

	@Autowired
	private SysAutoDictItemMapper autoDictItemMapper;

	@Autowired
	private ArchivesDaprUtil archivesDaprUtil;

	@Autowired
	private SysAutoDictMapper autoDictMapper;

	public static final String itemsLabel = "label,description,disable";

	public static final String itemsLabelRepeat = "同一项目岗位名称不可重复";

    /**
     * 项目配置表主表简单分页查询
     * @param tAutoMainRel 项目配置表主表
     * @return
     */
    @Override
    public IPage<TAutoMainRel> getTAutoMainRelPage(Page<TAutoMainRel> page, TAutoMainRelSearchVo tAutoMainRel){
        return baseMapper.getTAutoMainRelPage(page,tAutoMainRel);
    }

    /**
     * 项目配置表主表批量导出
     * @param searchVo 项目配置表主表
     * @return
     */
    @Override
    public void listExport(HttpServletResponse response, TAutoMainRelSearchVo searchVo){
        String fileName = "项目配置表批量导出" + DateUtil.getThisTime() + ".xlsx";
        //获取要导出的列表
        List<TAutoMainRelExportVo> list = new ArrayList<>();
        long count = noPageCountDiy(searchVo);
        ServletOutputStream out = null;
        try {
            out = response.getOutputStream();
            response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
            response.setCharacterEncoding("utf-8");
            response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName  , CommonConstants.UTF8));
            // 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
            ExcelWriter  excelWriter = EasyExcel.write(out, TAutoMainRelExportVo.class).build();
            int index = 0;
            if (count >  CommonConstants.ZERO_INT){
                for (int i = 0; i <= count; ) {
                    // 获取实际记录
                    searchVo.setLimitStart(i);
                    searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
                    list = noPageDiy(searchVo);
                    if (Common.isNotNull(list)){
                        ExcelUtil<TAutoMainRelExportVo> util = new ExcelUtil<>(TAutoMainRelExportVo.class);
                        for (TAutoMainRelExportVo vo:list){
                            util.convertEntity(vo,null,null,null);
                        }
                    }
                    if (Common.isNotNull(list)){
                        WriteSheet writeSheet = EasyExcel.writerSheet("项目配置表主表"+index).build();
                        excelWriter.write(list,writeSheet);
                        index++;
                    }
                    i = i + CommonConstants.EXCEL_EXPORT_LIMIT;
                    if (Common.isNotNull(list)){
                        list.clear();
                    }
                }
            }else {
                WriteSheet writeSheet = EasyExcel.writerSheet("项目配置表主表"+index).build();
                excelWriter.write(list,writeSheet);
            }
            if (Common.isNotNull(list)){
                list.clear();
            }
            out.flush();
            excelWriter.finish();
        }catch (Exception e){
            log.error("执行异常" ,e);
        }finally {
            try {
                if (null != out) {
                    out.close();
                }
            } catch (IOException e) {
                log.error("执行异常", e);
            }
        }
    }

    @Override
    public List<TAutoMainRelExportVo> noPageDiy(TAutoMainRelSearchVo searchVo) {

        return baseMapper.noPageDiy(searchVo);
    }

	private  Long noPageCountDiy(TAutoMainRelSearchVo searchVo) {
        return baseMapper.noPageCountDiy(searchVo);
    }

    private LambdaQueryWrapper buildQueryWrapper(TAutoMainRelSearchVo entity){
        LambdaQueryWrapper<TAutoMainRel> wrapper = Wrappers.lambdaQuery();
        if (ArrayUtil.isNotEmpty(entity.getCreateTimes())) {
            wrapper.ge(TAutoMainRel::getCreateTime, entity.getCreateTimes()[0])
                    .le(TAutoMainRel::getCreateTime,
                            entity.getCreateTimes()[1]);
        }
        if (Common.isNotNull(entity.getCreateName())){
            wrapper.eq(TAutoMainRel::getCreateName,entity.getCreateName());
        }
        return wrapper;
    }

    @Override
    public R<List<ErrorMessage>> importDiy(InputStream inputStream) {
        return  R.ok();
    }
    private void importTAutoMainRel(List<TAutoMainRelVo> excelVOList, List<ErrorMessage> errorMessageList) {
        // 个性化校验逻辑
        ErrorMessage errorMsg;
        // 执行数据插入操作 组装
        for (int i = 0; i < excelVOList.size(); i++) {
            TAutoMainRelVo excel = excelVOList.get(i);
            // 数据合法情况 TODO

            // 插入
            insertExcel(excel);
            errorMessageList.add(new ErrorMessage(excel.getRowIndex(),CommonConstants.SAVE_SUCCESS));
        }
    }
    /**
     * 插入excel bad record
     */
    private void insertExcel(TAutoMainRelVo excel) {
        TAutoMainRel insert = new TAutoMainRel();
        BeanUtil.copyProperties(excel, insert);
        this.save(insert);
    }

	@Override
	public R<TAutoMainRelDetailVo> getByIdAsso(String id) {
		TAutoMainRelDetailVo detailVo = new TAutoMainRelDetailVo();
		//项目配置主表
		TAutoMainRel mainRel = baseMapper.selectById(id);
		if (Common.isEmpty(mainRel)){
			return R.failed(CommonConstants.PARAM_IS_NOT_ERROR);
		}
		detailVo.setAutoMainRel(mainRel);
		//档案管理规则配置表
		detailVo.setAutoEmpRuleInfo(autoEmpRuleInfoMapper.selectOne(Wrappers.<TAutoEmpRuleInfo>query().lambda()
				.eq(TAutoEmpRuleInfo::getMainId,id).last(CommonConstants.LAST_ONE_SQL)));
		//项目配置日志表
		detailVo.setAutoRuleLogs(autoRuleLogMapper.selectList(Wrappers.<TAutoRuleLog>query().lambda()
				.eq(TAutoRuleLog::getMainId,id)));
		//岗位信息表
		detailVo.setAutoDictItems(autoDictItemMapper.selectList(Wrappers.<SysAutoDictItem>query().lambda()
				.eq(SysAutoDictItem::getDeptNo,mainRel.getDeptNo())
				.eq(SysAutoDictItem::getDisable,CommonConstants.ZERO_STRING)));
		//项目信息表
		detailVo.setSettleDomainVo(archivesDaprUtil.selectDeptByNo(mainRel.getDeptNo()));
		return R.ok(detailVo);
	}
	/**
	 * @Author fxj
	 * @Description 规则保存
	 * @Date 10:24 2025/3/14
	**/
	@Override
	public R<Boolean> saveAsso(TAutoMainRelAddVo entity) {
		YifuUser user  = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		if (Common.isEmpty(entity)){
			return R.failed(CommonConstants.PARAM_IS_NOT_ERROR);
		}
		TAutoMainRel autoMainRel = entity.getAutoMainRel();
		if (Common.isEmpty(autoMainRel)
				|| Common.isEmpty(autoMainRel.getDeptId())
				|| Common.isEmpty(autoMainRel.getDeptNo())){
			return R.failed(CommonConstants.PARAM_IS_NOT_ERROR);
		}
		TAutoMainRel existRel = baseMapper.selectOne(Wrappers.<TAutoMainRel>query().lambda()
				.eq(TAutoMainRel::getDeptNo,autoMainRel.getDeptNo()).last(CommonConstants.LAST_ONE_SQL));
		if (!Common.isEmpty(existRel)){
			return R.failed("已存在对应项目规则配置！");
		}
		SysAutoDict autoDict = autoDictMapper.selectOne(Wrappers.<SysAutoDict>query().lambda()
				.eq(SysAutoDict::getType,CommonConstants.POST_TYPE).last(CommonConstants.LAST_ONE_SQL));
		if (Common.isEmpty(autoDict)){
			return R.failed(CommonConstants.INIT_DATA_GET_FAIL);
		}
		TAutoEmpRuleInfo autoEmpRuleInfo = entity.getAutoEmpRuleInfo();
		if (Common.isNotNull(autoEmpRuleInfo)){
			ExcelUtil<TAutoEmpRuleInfo> excelUtil = new ExcelUtil<>(TAutoEmpRuleInfo.class);
			ErrorMessage errorMessage =  excelUtil.checkEntity(autoEmpRuleInfo,0);
			if (Common.isNotNull(errorMessage)){
				return R.failed(errorMessage.getMessage());
			}
		}
		//新增岗位字典
		List<SysAutoDictItem> autoDictItems = entity.getAutoDictItems();
		Set<String> repeatItems = new HashSet<>();
		for (SysAutoDictItem item:autoDictItems) {
			if (Common.isNotNull(repeatItems.contains(item.getLabel()))) {
				return R.failed(itemsLabelRepeat);
			}
		}
		//新增主表
		autoMainRel.setRuleUpdatePerson(user.getNickname());
		autoMainRel.setRuleUpdateTime(DateUtil.getCurrentDateTime());
		int res = baseMapper.insert(autoMainRel);
		if (res <= CommonConstants.ZERO_INT){
			return R.failed(CommonConstants.SAVE_FAILED);
		}

		if (Common.isNotNull(autoDictItems)) {
			for (SysAutoDictItem sysAutoDictItem:autoDictItems){
				//初始化字典信息
				initDictItem(sysAutoDictItem, user, autoDict, autoMainRel);
				sysAutoDictItem.setCreateTime(LocalDateTime.now());
				sysAutoDictItem.setCreateBy(user.getId());
				sysAutoDictItem.setCreateName(user.getNickname());
				autoDictItemMapper.insert(sysAutoDictItem);
			}
		}
		//新增档案规则
		if (Common.isNotNull(autoEmpRuleInfo)){
			autoEmpRuleInfo.setDeptNo(autoMainRel.getDeptNo());
			autoEmpRuleInfo.setDeptId(autoMainRel.getDeptId());
			autoEmpRuleInfo.setMainId(autoMainRel.getId());
			autoEmpRuleInfoMapper.insert(autoEmpRuleInfo);
		}
		return R.ok();
	}
	/**
	 * @Author fxj
	 * @Description 规则修改
	 * @Date 10:24 2025/3/14
	**/
	@Override
	public R<Boolean> updateAsso(TAutoMainRelAddVo entity) {
		YifuUser user  = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return R.failed(CommonConstants.USER_FAIL);
		}
		if (Common.isEmpty(entity)){
			return R.failed(CommonConstants.PARAM_IS_NOT_ERROR);
		}
		TAutoMainRel autoMainRel = entity.getAutoMainRel();
		if (Common.isEmpty(autoMainRel) || Common.isEmpty(autoMainRel.getId())){
			return R.failed(CommonConstants.PARAM_IS_NOT_ERROR);
		}
		TAutoMainRel autoMainRelOld = baseMapper.selectById(autoMainRel.getId());
		if (!Common.isNotNull(autoMainRelOld)){
			return   R.failed(CommonConstants.PARAM_IS_NOT_ERROR);
		}
		SysAutoDict autoDict = autoDictMapper.selectOne(Wrappers.<SysAutoDict>query().lambda()
				.eq(SysAutoDict::getType,CommonConstants.POST_TYPE).last(CommonConstants.LAST_ONE_SQL));
		if (Common.isEmpty(autoDict)){
			return R.failed(CommonConstants.INIT_DATA_GET_FAIL);
		}
		TAutoEmpRuleInfo autoEmpRuleNew = entity.getAutoEmpRuleInfo();
		TAutoEmpRuleInfo autoEmpRuleOld = null;
		if (Common.isNotNull(autoEmpRuleNew)){
			if (!Common.isNotNull(autoEmpRuleNew.getId())){
				return R.failed(CommonConstants.PARAM_IS_NOT_ERROR);
			}
			autoEmpRuleOld = autoEmpRuleInfoMapper.selectById(autoEmpRuleNew.getId());
			ExcelUtil<TAutoEmpRuleInfo> excelUtil = new ExcelUtil<>(TAutoEmpRuleInfo.class);
			ErrorMessage errorMessage =  excelUtil.checkEntity(autoEmpRuleNew,0);
			if (Common.isNotNull(errorMessage)){
				return R.failed(errorMessage.getMessage());
			}
		}
		//更新岗位字典
		List<SysAutoDictItem> autoDictItems = entity.getAutoDictItems();
		Map<String,SysAutoDictItem> repeatItems = new HashMap<>();
		//更新岗位字典
		List<SysAutoDictItem> oldAutoDictItems = autoDictItemMapper.selectList(Wrappers.<SysAutoDictItem>query()
				.lambda().eq(SysAutoDictItem::getDeptNo,autoMainRel.getDeptNo())
				.eq(SysAutoDictItem::getDisable,CommonConstants.ZERO_STRING));
		if (Common.isNotNull(oldAutoDictItems)){
			repeatItems = oldAutoDictItems.stream().collect(Collectors.toMap(SysAutoDictItem::getLabel,v->v));
		}
		//检验岗位字典是否重复
		R<Boolean> checkRes = checkItemRepeat(autoDictItems, repeatItems);
		if (checkRes != null){
			return checkRes;
		}
		//更新主表
		autoMainRel.setRuleUpdatePerson(user.getNickname());
		autoMainRel.setRuleUpdateTime(DateUtil.getCurrentDateTime());
		baseMapper.updateById(autoMainRel);
		Map<String,String> diffKeyMap = new HashMap<>();
		Map<String,Object> oldMap = new HashMap<>();
		Map<String,Object> newMap = new HashMap<>();
		if (!autoMainRel.getPostFlag().equals(autoMainRelOld.getPostFlag())){
			oldMap.put("oldRuleRel",autoMainRelOld);
			newMap.put("newRuleRel",autoMainRel);
			diffKeyMap.put("ruleRel","postFlag");
		}

		//处理岗位字典数据
		dictItemHandle(user, autoMainRel, autoDict, diffKeyMap, autoDictItems,oldAutoDictItems);
		oldMap.put("oldItems",oldAutoDictItems);
		newMap.put("newItems",autoDictItems);
		//更新档案管理规则
		if (Common.isNotNull(autoEmpRuleNew)){
			oldMap.put("oldEmpRule",autoEmpRuleOld);
			newMap.put("newEmpRule",autoEmpRuleNew);
			List<String> ignoreFields = new ArrayList<>();
			ignoreFields.add("mainId");
			ignoreFields.add("createBy");
			ignoreFields.add("createName");
			ignoreFields.add("createTime");
			ignoreFields.add("deptId");
			String differenceKey = HrEquator.comparisonValueIgnoreField(autoEmpRuleOld, autoEmpRuleNew,ignoreFields);
			if (!Common.isEmpty(differenceKey)){
				diffKeyMap.put("empRule",differenceKey);
			}
			autoEmpRuleNew.setDeptNo(autoMainRel.getDeptNo());
			autoEmpRuleNew.setDeptId(autoMainRel.getDeptId());
			autoEmpRuleNew.setMainId(autoMainRel.getId());
			autoEmpRuleInfoMapper.updateById(autoEmpRuleNew);
		}
		//插入变更日志
		insertLog(autoMainRel, diffKeyMap, oldMap, newMap);
		return R.ok(true,CommonConstants.UPDATE_SUCCESS);
	}

	private R<Boolean> checkItemRepeat(List<SysAutoDictItem> autoDictItems, Map<String, SysAutoDictItem> repeatItems) {
		SysAutoDictItem checkItem;
		for (SysAutoDictItem item: autoDictItems) {
			if (Common.isNotNull(repeatItems)) {
				checkItem = repeatItems.get(item.getLabel());
				if (null != checkItem
						&& item.getLabel().equals(checkItem.getLabel())
						&& CommonConstants.ZERO_STRING.equals(checkItem.getDisable())
						&& (Common.isEmpty(checkItem.getId()) || !checkItem.getId().equals(item.getId()))
						){
					return R.failed(itemsLabelRepeat);
				}
				repeatItems.put(item.getLabel(),item);
			}
		}
		return null;
	}

	private void dictItemHandle(YifuUser user,
								TAutoMainRel autoMainRel,
								SysAutoDict autoDict,
								Map<String, String> diffKeyMap,
								List<SysAutoDictItem> autoDictItems,
								List<SysAutoDictItem> oldAutoDictItems) {

		if (Common.isNotNull(autoDictItems)){
			Map<Long, SysAutoDictItem> oldItemMap = null;
			if (Common.isNotNull(oldAutoDictItems)){
				oldItemMap = oldAutoDictItems.stream().collect(Collectors.toMap(SysAutoDictItem::getId, v->v));
			}
			SysAutoDictItem oldItem;
			String diffKey;
			StringBuilder difkeys = new StringBuilder("");
			List<String> ignore = new ArrayList<>();
			ignore.add("value,sortOrder");
			for(SysAutoDictItem sysAutoDictItem : autoDictItems){
				initDictItem(sysAutoDictItem, user, autoDict, autoMainRel);
				if (Common.isNotNull(sysAutoDictItem.getId())){
					oldItem = null==oldItemMap?null:oldItemMap.get(sysAutoDictItem.getId());
					if (Common.isNotNull(oldItem)){

						diffKey = HrEquator.comparisonValueIgnoreField(oldItem, sysAutoDictItem,ignore);
						if (Common.isNotNull(diffKey)){
							if (Common.isEmpty(difkeys.toString())){
								difkeys.append(diffKey);
							}else {
								difkeys.append(CommonConstants.COMMA_STRING);
								difkeys.append(diffKey);
							}
						}
					}
					autoDictItemMapper.updateById(sysAutoDictItem);
				}else {
					if (Common.isEmpty(difkeys.toString())){
						difkeys.append(itemsLabel);
					}else {
						difkeys.append(CommonConstants.COMMA_STRING);
						difkeys.append(itemsLabel);
					}
					sysAutoDictItem.setCreateTime(LocalDateTime.now());
					sysAutoDictItem.setCreateBy(user.getId());
					sysAutoDictItem.setCreateName(user.getNickname());
					autoDictItemMapper.insert(sysAutoDictItem);
				}
			}
			if (!Common.isEmpty(difkeys.toString())){
				String[]  keys = difkeys.toString().split(CommonConstants.COMMA_STRING);
				String keysStr = Arrays.stream(keys).distinct().collect(Collectors.joining(CommonConstants.COMMA_STRING));
				diffKeyMap.put("postType",keysStr);
			}
		}
	}

	/**
	 * @Author fxj
	 * @Description 插入变更日志
	 * @Date 15:43 2025/3/14
	**/
	private void insertLog(TAutoMainRel autoMainRel, Map<String, String> diffKeyMap, Map<String, Object> oldMap, Map<String, Object> newMap) {
		try{
			//比较记录不影响业务逻辑，用try套住
			//如果有差异保存差异
			if(!Common.isEmpty(diffKeyMap)){
				TAutoRuleLog log = new TAutoRuleLog();
				log.setType(0);
				log.setMainId(autoMainRel.getId());
				log.setCreateTime(LocalDateTime.now());
				log.setOldInfo(JSON.toJSONString(oldMap,features));
				log.setNewInfo(JSON.toJSONString(newMap,features));
				log.setDifferenceInfo(JSON.toJSONString(diffKeyMap,features));
				autoRuleLogMapper.insert(log);
			}
		}catch (Exception e){
			log.error(JSON.toJSON(oldMap)+"插入修改记录报错>>>",e);
		}
	}
	/**
	 * @Author fxj
	 * @Description 初始化字典项信息公共方法
	 * @Date 15:43 2025/3/14
	**/
	private void initDictItem(SysAutoDictItem sysAutoDictItem, YifuUser user, SysAutoDict autoDict, TAutoMainRel autoMainRel) {
		sysAutoDictItem.setUpdateTime(LocalDateTime.now());
		sysAutoDictItem.setUpdateBy(user.getId());
		sysAutoDictItem.setDelFlag(CommonConstants.ZERO_STRING);
		sysAutoDictItem.setClientId(ServiceNameConstants.SERVICE_CLIENT_ID_MVP);
		sysAutoDictItem.setDictId(autoDict.getId());
		sysAutoDictItem.setType(CommonConstants.POST_TYPE);
		if (Common.isEmpty(sysAutoDictItem.getDisable())){
			sysAutoDictItem.setDisable(CommonConstants.ZERO_STRING);
		}
		sysAutoDictItem.setDeptNo(autoMainRel.getDeptNo());
	}

	private SerializerFeature[] features = new SerializerFeature[] {
			WriteMapNullValue, WriteNullNumberAsZero, WriteNullListAsEmpty,
			WriteNullStringAsEmpty,WriteDateUseDateFormat
	};


	@Override
	public R<TAutoMainRelDetailVo> getProjectRule(String deptId, String deptNo) {
		if (Common.isEmpty(deptId) && Common.isEmpty(deptNo)){
			return R.failed(CommonConstants.PARAM_IS_NOT_ERROR);
		}
		LambdaQueryWrapper<TAutoMainRel> wrapper = buildQueryWrapper(deptId,deptNo);
		TAutoMainRel rel = baseMapper.selectOne(wrapper);
		if (Common.isNotNull(rel) && Common.isNotNull(rel.getId())){
			return this.getByIdAsso(rel.getId());
		}
		return R.ok();
	}
	private LambdaQueryWrapper buildQueryWrapper(String deptId, String deptNo){
		LambdaQueryWrapper<TAutoMainRel> wrapper = Wrappers.lambdaQuery();
		if (ArrayUtil.isNotEmpty(deptId)) {
			wrapper.ge(TAutoMainRel::getDeptId, deptId);
		}
		if (Common.isNotNull(deptNo)){
			wrapper.eq(TAutoMainRel::getDeptNo,deptNo);
		}
		return wrapper;
	}
}
