/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeLog;
import com.yifu.cloud.plus.v1.yifu.archives.service.TEmployeeLogService;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;


/**
 * 档案变更日志表
 *
 * @author hgw
 * @date 2022-06-20 09:55:06
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/temployeelog")
@Tag(name = "档案变更日志表管理")
public class TEmployeeLogController {

	private final TEmployeeLogService tEmployeeLogService;

	/**
	 * 分页查询
	 *
	 * @param page         分页对象
	 * @param tEmployeeLog 档案变更日志表
	 * @return
	 */
	@Operation(summary = "分页查询", description = "分页查询")
	@GetMapping("/page")
	@PreAuthorize("@pms.hasPermission('demo_temployeelog_get')")
	public R<IPage<TEmployeeLog>> getTEmployeeLogPage(Page<TEmployeeLog> page, TEmployeeLog tEmployeeLog) {
		return R.ok(tEmployeeLogService.page(page, Wrappers.query(tEmployeeLog).orderByDesc(CommonConstants.CREATE_TIME)));
	}


	/**
	 * 通过id查询档案变更日志表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('demo_temployeelog_get')")
	public R<TEmployeeLog> getById(@PathVariable("id") String id) {
		return R.ok(tEmployeeLogService.getById(id));
	}

	/**
	 * 新增档案变更日志表
	 *
	 * @param tEmployeeLog 档案变更日志表
	 * @return R
	 */
	@Operation(summary = "新增档案变更日志表", description = "新增档案变更日志表")
	@SysLog("新增档案变更日志表")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('demo_temployeelog_add')")
	public R<Boolean> save(@RequestBody TEmployeeLog tEmployeeLog) {
		return R.ok(tEmployeeLogService.save(tEmployeeLog));
	}

	/**
	 * 修改档案变更日志表
	 *
	 * @param tEmployeeLog 档案变更日志表
	 * @return R
	 */
	@Operation(summary = "修改档案变更日志表", description = "修改档案变更日志表")
	@SysLog("修改档案变更日志表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('demo_temployeelog_edit')")
	public R<Boolean> updateById(@RequestBody TEmployeeLog tEmployeeLog) {
		return R.ok(tEmployeeLogService.updateById(tEmployeeLog));
	}

	/**
	 * 通过id删除档案变更日志表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除档案变更日志表", description = "通过id删除档案变更日志表")
	@SysLog("通过id删除档案变更日志表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('demo_temployeelog_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tEmployeeLogService.removeById(id));
	}

	/**
	 * @Description: 通过EMP_ID查询人员档案变更日志
	 * @Author: hgw
	 * @Date: 2022/6/28 16:45
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeLog>>
	 **/
	@Operation(summary = "通过EMP_ID查询人员档案变更日志", description = "通过EMP_ID查询人员档案变更日志")
	@GetMapping("/getByEmpId")
	public R<List<TEmployeeLog>> getByEmpId(@RequestParam String empId) {
		return R.ok(tEmployeeLogService.list(Wrappers.<TEmployeeLog>query().eq("EMP_ID", empId)
				.eq("TYPE", CommonConstants.ZERO_INT).orderByDesc(CommonConstants.CREATE_TIME)));
	}

	/**
	 * @Description: 通过PROJECT_ID查询项目档案变更日志
	 * @Author: hgw
	 * @Date: 2022/6/28 16:45
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeLog>>
	 **/
	@Operation(summary = "通过PROJECT_ID查询项目档案变更日志", description = "通过PROJECT_ID查询项目档案变更日志")
	@GetMapping("/getByProjectId")
	public R<List<TEmployeeLog>> getByProjectId(@RequestParam String projectId) {
		return R.ok(tEmployeeLogService.list(Wrappers.<TEmployeeLog>query().eq("PROJECT_ID", projectId)
				.eq("TYPE", CommonConstants.ONE_INT).orderByDesc(CommonConstants.CREATE_TIME)));
	}
}
