/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.salary.service.impl;

import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.ResultConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.DateUtil;
import com.yifu.cloud.plus.v1.yifu.ekp.vo.EkpSalaryParamVo;
import com.yifu.cloud.plus.v1.yifu.insurances.vo.EkpSocialViewVo;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TDeptSee;
import com.yifu.cloud.plus.v1.yifu.salary.entity.THaveSalaryNosocial;
import com.yifu.cloud.plus.v1.yifu.salary.entity.TSalaryAccount;
import com.yifu.cloud.plus.v1.yifu.salary.mapper.SysConfigLimitMapper;
import com.yifu.cloud.plus.v1.yifu.salary.mapper.TDeptSeeMapper;
import com.yifu.cloud.plus.v1.yifu.salary.mapper.TSalaryAccountMapper;
import com.yifu.cloud.plus.v1.yifu.salary.service.TSalaryAccountService;
import com.yifu.cloud.plus.v1.yifu.salary.vo.*;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.apache.ibatis.annotations.Param;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * 工资报账主表（工资条）
 *
 * @author hgw
 * @date 2022-08-05 11:40:14
 */
@Log4j2
@RequiredArgsConstructor
@Service
public class TSalaryAccountServiceImpl extends ServiceImpl<TSalaryAccountMapper, TSalaryAccount> implements TSalaryAccountService {
	@Autowired
	private TDeptSeeMapper deptSeeMapper;

	private AtomicInteger atomicInteger = new AtomicInteger(0);
	//初始化附件上传队列上限值
	private  int maxLimit = 1;

	private final SysConfigLimitMapper configLimitMapper;
	/**
	 * 工资报账主表（工资条）简单分页查询
	 *
	 * @param tSalaryAccount 工资报账主表（工资条）
	 * @return
	 */
	@Override
	public IPage<TSalaryAccount> getTSalaryAccountPage(Page<TSalaryAccount> page, TSalaryAccountSearchVo tSalaryAccount) {
		// 不可查询项目合集 fxj 2024-10-23 优化
		List<TDeptSee> depts = deptSeeMapper.selectList(Wrappers.<TDeptSee>query().lambda().eq(TDeptSee::getCanSee,CommonConstants.ONE_INT));
		List<String> deptNos = null;
		if (Common.isNotNull(depts)){
			deptNos = depts.stream().map(TDeptSee::getDeptNo).collect(Collectors.toList());
		}
		return baseMapper.getTSalaryAccountPage(page, tSalaryAccount,deptNos);
	}

	/**
	 * @Author fxj
	 * @Description
	 * @Date 15:27 2024/10/23
	 * @Param type 0 报账批量导出  1 报账审核人导出
	 * @return
	**/
	@Override
	public void exportAsso(HttpServletResponse response, TSalaryAccountSearchVo searchVo,int type) {
		maxLimit = configLimitMapper.getSysConfigLimitByKey("DOWN_LOAD_MAX_LIMIT");
		if (atomicInteger.incrementAndGet() <= maxLimit){
			try {
				if (CommonConstants.ZERO_INT == type){
					listExport(response, searchVo);
				}else if (CommonConstants.ONE_INT == type){
					auditExport(response, searchVo);
				}
			}catch (Exception e){
				log.error("导出异常：",e);
			}finally {
				atomicInteger.decrementAndGet();
			}
		}else {
			// 前面做了+1（atomicInteger.incrementAndGet()）这里也要-1
			atomicInteger.decrementAndGet();
			log.error("超出阈值："+ ResultConstants.FILE_DOWN_LOAD_DATA);
			exportError(response, ResultConstants.FILE_DOWN_LOAD_DATA);
		}

	}
	/**
	 * 工资报账主表（工资条）批量导出
	 *
	 * @return
	 */
	@Override
	public void listExport(HttpServletResponse response, TSalaryAccountSearchVo searchVo) {
		String fileName = "员工报账查询批量导出" + DateUtil.getThisTime() + CommonConstants.XLSX;
		//获取要导出的列表
		List<SalaryAccountExportVo> list = new ArrayList<>();
		// 不可查询项目合集 fxj 2024-10-23 优化
		List<TDeptSee> depts = deptSeeMapper.selectList(Wrappers.<TDeptSee>query().lambda().eq(TDeptSee::getCanSee,CommonConstants.ONE_INT));
		List<String> deptNos = null;
		if (Common.isNotNull(depts)){
			deptNos = depts.stream().map(TDeptSee::getDeptNo).collect(Collectors.toList());
		}
		long count = noPageCountDiyLast(searchVo,deptNos);
		ServletOutputStream out = null;
		try {
			out = response.getOutputStream();
			response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
			response.setCharacterEncoding("utf-8");
			response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName, "UTF-8"));
			// 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
			ExcelWriter excelWriter = EasyExcelFactory.write(out, TSalaryAccount.class).includeColumnFiledNames(searchVo.getExportFields()).build();
			int index = 0;
			WriteSheet writeSheet = null;
			if (count > CommonConstants.ZERO_INT) {
				for (int i = 0; i <= count; i += CommonConstants.EXCEL_EXPORT_LIMIT_1) {
					// 获取实际记录
					searchVo.setLimitStart(i);
					searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT_1);
					list = noPageDiyLast(searchVo,deptNos);
					if (Common.isNotNull(list)) {
						if (i%CommonConstants.EXCEL_EXPORT_LIMIT==0){
							writeSheet = EasyExcelFactory.writerSheet("工资报账主表（工资条）" + index).build();
							index++;
						}
						excelWriter.write(list, writeSheet);
						list.clear();
					}
				}
			} else {
				writeSheet = EasyExcelFactory.writerSheet("工资报账主表（工资条）" + index).build();
				excelWriter.write(list, writeSheet);
			}
			if (Common.isNotNull(list)) {
				list.clear();
			}
			out.flush();
			excelWriter.finish();
		} catch (Exception e) {
			log.error("执行异常", e);
		} finally {
			try {
				if (null != out) {
					out.close();
				}
			} catch (IOException e) {
				log.error("执行异常", e);
			}
		}
	}

	/**
	 * 审核人导出
	 *
	 * @return
	 */
	@Override
	public void auditExport(HttpServletResponse response, TSalaryAccountSearchVo searchVo) {
		String fileName = "员工报账导出" + DateUtil.getThisTime() + CommonConstants.XLSX;
		//获取要导出的列表
		List<TSalaryAccountExportVo> list = new ArrayList<>();
		long count = baseMapper.auditExportCountDiy(searchVo, searchVo.getIdList());
		ServletOutputStream out = null;
		try {
			out = response.getOutputStream();
			response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
			response.setCharacterEncoding("utf-8");
			response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName, "UTF-8"));
			// 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
			ExcelWriter excelWriter = EasyExcelFactory.write(out, TSalaryAccountExportVo.class).includeColumnFiledNames(searchVo.getExportFields()).build();
			int index = 0;
			WriteSheet writeSheet = null;
			if (count > CommonConstants.ZERO_INT) {
				Map<String,TSalaryAccountItemExportVo> itemExportVoMap = new HashMap<>();
				List<String> idList = null;
				List<String> idStandardList = null;
				List<TAuditExportStandardVo> standardItems = null;
				Map<String,TAuditExportStandardVo> itemStandardMap = new HashMap<>();
				Map<String,TAuditExportItemVo> itemAnnualBonusMap = new HashMap<>();
				Map<String,TAuditExportItemVo> itemPersionTaxMap = new HashMap<>();
				Map<String,TAuditExportItemVo> itemAnnuityMap = new HashMap<>();
				Map<String,TAuditExportItemVo> itemSubsidyMap = new HashMap<>();
				List<TSalaryAccountItemExportVo> items = null;
				List<TAuditExportItemVo> annualBonusItems;
				List<TAuditExportItemVo> exemptionPersionTaxItems;
				List<TAuditExportItemVo> enterpriseAnnuityItems;
				List<TAuditExportItemVo> phoneSubsidyItems;
				for (int i = 0; i <= count; i += CommonConstants.EXCEL_EXPORT_LIMIT_1) {
					// 获取实际记录
					searchVo.setLimitStart(i);
					searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT_1);
					list = baseMapper.auditExportDiy(searchVo, searchVo.getIdList());
					if (Common.isNotNull(list)) {
						//先把数据封装完成
						idList = list.stream().map(TSalaryAccountExportVo::getId).collect(Collectors.toList());
						idStandardList = list.stream().map(TSalaryAccountExportVo::getSalaryFormId).collect(Collectors.toList());
						//初始化明细数据MAP

						if (Common.isNotNull(idList)){
							items = baseMapper.auditExportSelectItems(idList);
							if (Common.isNotNull(items)){
								itemExportVoMap = items.stream().collect(Collectors.toMap(TSalaryAccountItemExportVo::getSalaryAccountId,vo -> vo,(k1,k2) -> k1));
							}
							items.clear();
							annualBonusItems = baseMapper.auditExportAnnualBonus(idList);
							if (Common.isNotNull(annualBonusItems)){
								itemAnnualBonusMap = annualBonusItems.stream().collect(Collectors.toMap(TAuditExportItemVo::getSalaryAccountId,vo -> vo,(k1,k2) -> k1));
							}
							annualBonusItems.clear();
							exemptionPersionTaxItems = baseMapper.auditExportExemptionPersionTax(idList);
							if (Common.isNotNull(exemptionPersionTaxItems)){
								itemPersionTaxMap = exemptionPersionTaxItems.stream().collect(Collectors.toMap(TAuditExportItemVo::getSalaryAccountId,vo -> vo,(k1,k2) -> k1));
							}
							exemptionPersionTaxItems.clear();
							enterpriseAnnuityItems = baseMapper.auditExportEnterpriseAnnuity(idList);
							if (Common.isNotNull(enterpriseAnnuityItems)){
								itemAnnuityMap = enterpriseAnnuityItems.stream().collect(Collectors.toMap(TAuditExportItemVo::getSalaryAccountId,vo -> vo,(k1,k2) -> k1));
							}
							enterpriseAnnuityItems.clear();
							phoneSubsidyItems = baseMapper.auditExportPhoneSubsidy(idList);
							if (Common.isNotNull(phoneSubsidyItems)){
								itemSubsidyMap = phoneSubsidyItems.stream().collect(Collectors.toMap(TAuditExportItemVo::getSalaryAccountId,vo -> vo,(k1,k2) -> k1));
							}
							phoneSubsidyItems.clear();
						}
						if (Common.isNotNull(idStandardList)){
							standardItems = baseMapper.auditExportStandard(idStandardList);
							if (Common.isNotNull(standardItems)){
								itemStandardMap = standardItems.stream().collect(Collectors.toMap(TAuditExportStandardVo::getSalaryFormId,vo -> vo,(k1,k2) -> k1));
								standardItems.clear();
							}
						}
						//封装信息
						initResult(list, itemExportVoMap,itemStandardMap,itemAnnualBonusMap,itemPersionTaxMap,itemAnnuityMap,itemSubsidyMap);
						if (i%CommonConstants.EXCEL_EXPORT_LIMIT==0){
							writeSheet = EasyExcelFactory.writerSheet("工资报账" + index).build();
							index++;
						}
						excelWriter.write(list, writeSheet);
						list.clear();
					}
					clearMap(itemExportVoMap, idList, idStandardList, itemStandardMap, itemAnnualBonusMap, itemPersionTaxMap, itemAnnuityMap, itemSubsidyMap);
				}
			} else {
				writeSheet = EasyExcelFactory.writerSheet("工资报账" + index).build();
				excelWriter.write(list, writeSheet);
			}
			if (Common.isNotNull(list)) {
				list.clear();
			}
			out.flush();
			excelWriter.finish();
		} catch (Exception e) {
			log.error("执行异常", e);
		} finally {

			try {
				if (null != out) {
					out.close();
				}
			} catch (IOException e) {
				log.error("执行异常", e);
			}
		}
	}

	private void clearMap(Map<String, TSalaryAccountItemExportVo> itemExportVoMap, List<String> idList, List<String> idStandardList, Map<String, TAuditExportStandardVo> itemStandardMap, Map<String, TAuditExportItemVo> itemAnnualBonusMap, Map<String, TAuditExportItemVo> itemPersionTaxMap, Map<String, TAuditExportItemVo> itemAnnuityMap, Map<String, TAuditExportItemVo> itemSubsidyMap) {
		if (Common.isNotNull(idList)){
			idList.clear();
		}
		if (Common.isNotNull(idStandardList)){
			idStandardList.clear();
		}
		if (Common.isNotNull(itemExportVoMap)){
			itemExportVoMap.clear();
		}
		if (Common.isNotNull(itemStandardMap)){
			itemStandardMap.clear();
		}
		if (Common.isNotNull(itemAnnualBonusMap)){
			itemAnnualBonusMap.clear();
		}
		if (Common.isNotNull(itemPersionTaxMap)){
			itemPersionTaxMap.clear();
		}
		if (Common.isNotNull(itemAnnuityMap)){
			itemAnnuityMap.clear();
		}
		if (Common.isNotNull(itemSubsidyMap)){
			itemSubsidyMap.clear();
		}
	}

	private void initResult(List<TSalaryAccountExportVo> list,
							Map<String, TSalaryAccountItemExportVo> itemExportVoMap,
							Map<String,TAuditExportStandardVo> itemStandardMap,
							Map<String,TAuditExportItemVo> itemAnnualBonusMap,
							Map<String,TAuditExportItemVo>  itemPersionTaxMap,
							Map<String,TAuditExportItemVo>  itemAnnuityMap,
							Map<String,TAuditExportItemVo> itemSubsidyMap) {
		TSalaryAccountItemExportVo voTemp;
		TAuditExportStandardVo standardVo;
		TAuditExportItemVo itemVo;
		for (TSalaryAccountExportVo vo: list){
			voTemp = itemExportVoMap.get(vo.getId());
			if (Common.isNotNull(voTemp)){
				vo.setPdeductionMoney(voTemp.getPdeductionMoney());
				vo.setPdeductionDetail(voTemp.getPdeductionDetail());
			}
			standardVo = itemStandardMap.get(vo.getSalaryFormId());
			if (Common.isNotNull(standardVo)){
				vo.setStatus(standardVo.getStatus());
				vo.setBusinessPrimaryType(standardVo.getBusinessPrimaryTyp());
				vo.setBusinessSecondType(standardVo.getBusinessSecondType());
				vo.setBusinessThirdType(standardVo.getBusinessThirdType());
			}
			itemVo = itemAnnualBonusMap.get(vo.getId());
			if (Common.isNotNull(itemVo)){
				vo.setAnnualBonus(itemVo.getAnnualBonus());
			}
			itemVo = itemPersionTaxMap.get(vo.getId());
			if (Common.isNotNull(itemVo)){
				vo.setExemptionPersionTax(itemVo.getExemptionPersionTax());
			}
			itemVo = itemAnnuityMap.get(vo.getId());
			if (Common.isNotNull(itemVo)){
				vo.setEnterpriseAnnuity(itemVo.getEnterpriseAnnuity());
			}
			itemVo = itemSubsidyMap.get(vo.getId());
			if (Common.isNotNull(itemVo)){
				vo.setPhoneSubsidy(itemVo.getPhoneSubsidy());
			}
		}
	}

	/**
	 * @Description: 根据工资id，返回报账明细（字段较少且有计算，其他地方勿用）
	 * @Author: hgw
	 * @Date: 2022/9/2 17:07
	 * @return: java.util.List<com.yifu.cloud.plus.v1.yifu.salary.entity.TSalaryAccount>
	 **/
	@Override
	public List<TSalaryAccount> getListByIncome(String salaryId) {
		return baseMapper.getListByIncome(salaryId);
	}

	@Override
	public List<TSalaryAccount> noPageDiy(TSalaryAccountSearchVo searchVo) {
		return baseMapper.noPageDiy(searchVo,searchVo.getIdList());
	}

	@Override
	public List<SalaryAccountExportVo> noPageDiyLast(TSalaryAccountSearchVo searchVo,List<String> deptNos) {
		return baseMapper.noPageDiyLast(searchVo,searchVo.getIdList(),deptNos);
	}

	@Override
	public List<TSalaryAccount> getAccountListBySalaryId(TSalaryAccountSearchVo searchVo) {
		return baseMapper.getAccountListBySalaryId(searchVo);
	}

	/**
	 * @param applyNo
	 * @Description: 导出报账专用hgw
	 * @Author: hgw
	 * @Date: 2024/4/18 11:00
	 * @return: java.util.List<com.yifu.cloud.plus.v1.yifu.salary.entity.TSalaryAccount>
	 **/
	@Override
	public List<TSalaryAccountExportByEkpVo> getAccountListByApplyNo(String applyNo) {
		return baseMapper.getAccountListByApplyNo(applyNo);
	}

	private int noPageCountDiy(TSalaryAccountSearchVo searchVo) {
		return baseMapper.noPageCountDiy(searchVo,searchVo.getIdList());
	}

	private int noPageCountDiyLast(TSalaryAccountSearchVo searchVo,List<String> deptNos) {
		return baseMapper.noPageCountDiyLast(searchVo,searchVo.getIdList(),deptNos);
	}

	@Override
	public Map<String, Integer> getAccountCheckMap(String settleId, String settleMonth, String salaryType) {
		List<AccountCheckVo> list;
		if (CommonConstants.ZERO_STRING.equals(salaryType)) {
			list = baseMapper.getAccountCheckList(settleId, settleMonth);
		} else {
			list = baseMapper.getAccountCheckListLabor(settleId, settleMonth);
		}
		Map<String, Integer> map = new HashMap<>();
		if (list != null && !list.isEmpty()) {
			for (AccountCheckVo str : list) {
				map.put(str.getTitle() + CommonConstants.DOWN_LINE_STRING + str.getRelaySalary().setScale(2, BigDecimal.ROUND_HALF_UP), 1);
			}
		}
		return map;
	}

	@Override
	public Map<String, BigDecimal> getAccountSpecialMap(String invoiceTitle, String unitId, String salaryMonth) {
		List<AccountCheckVo> list = baseMapper.getAccountSpecialList(invoiceTitle, unitId, salaryMonth);
		Map<String, BigDecimal> map = new HashMap<>();
		if (list != null && !list.isEmpty()) {
			for (AccountCheckVo str : list) {
				map.put(str.getTitle(), str.getRelaySalary());
			}
		}
		return map;
	}

	/**
	 * @param empIdCard
	 * @param nowYear
	 * @Description: 获取当前年最小计税月
	 * @Author: hgw
	 * @Date: 2022/1/19 17:57
	 * @return: java.lang.String
	 **/
	@Override
	public String getMinTaxMonthByNowYear(String empIdCard, int nowYear) {
		return baseMapper.getMinTaxMonthByNowYear(empIdCard, nowYear);
	}

	@Override
	public Map<String, String> getMinTaxMonthByNowYearAndList(List<String> empIdCardList, int nowYear) {
		Map<String, String> taxMonthMap = new HashMap<>();
		if (empIdCardList != null && !empIdCardList.isEmpty()) {
			List<TSalaryAccount> list = baseMapper.getMinTaxMonthByNowYearAndList(empIdCardList, nowYear);
			if (list != null && !list.isEmpty()) {
				for (TSalaryAccount a : list) {
					taxMonthMap.put(a.getEmpIdcard(), a.getTaxMonth());
				}
			}
		}
		return taxMonthMap;
	}

	/**
	 * @Author fxj
	 * @Description 获取有工资无社保数据
	 * @Date 17:22 2022/8/16
	 * @Param
	 * @return
	**/
	@Override
	public List<THaveSalaryNosocial> getLastMonthTHaveSalaryNosocial(String month) {
		return baseMapper.getLastMonthTHaveSalaryNosocial(month,
				DateUtil.getYearAndMonth(month,-CommonConstants.ONE_INT),
				DateUtil.getYearAndMonth(month,-CommonConstants.TWO_INT));
	}

	@Override
	public List<String> getAccountYearCheckMap() {
		return baseMapper.getAccountYearCheckList();
	}

	@Override
	public List<EkpSalaryParamVo> getEkpSalaryParamList(String salaryId) {
		return baseMapper.getEkpSalaryParamList(salaryId);
	}

	@Override
	public void backSalaryBySalaryId(String salaryId) {
		baseMapper.backSalaryBySalaryId(salaryId);
	}

	@Override
	public Boolean updateSalarySettleStatus(List<EkpSocialViewVo> viewVoList) {
		TSalaryAccount account;
		try {
			if (!viewVoList.isEmpty()) {
				for (EkpSocialViewVo viewVo : viewVoList) {
					//根据明细id更新结算状态
					account = baseMapper.selectById(viewVo.getId());
					if (Common.isNotNull(account)) {
						if (CommonConstants.ZERO_STRING.equals(viewVo.getPayFlag())) {
							account.setIncomeSettleFlag(viewVo.getIncomeSettleFlag());
							account.setIncomeCollectFlag(viewVo.getIncomeCollectFlag());
						} else {
							account.setPaySettleFlag(viewVo.getPaySettleFlag());
							account.setPayCollectFlag(viewVo.getPayCollectFlag());
						}
						baseMapper.updateById(account);
					}
				}
			}
		} catch (Exception e) {
			log.error("更新薪资结算状态失败", e);
			return false;
		}
		return true;
	}

	/**
	 * @param page 分页
	 * @param settlementFormVo 结算单Vo
	 * @Description: B端薪酬第一个统计列表
	 * @Author: hgw
	 * @Date: 2020/8/21 11:38
	 * @return: com.baomidou.mybatisplus.core.metadata.IPage<com.yifu.cloud.v1.hrms.api.vo.SettlementFormVo>
	 **/
	@Override
	public IPage<SettlementFormVo> getSettlementFormVoPage(Page<SettlementFormVo> page
			, SettlementFormVo settlementFormVo, List<String> settleDepartIdList){
		return baseMapper.getSettlementFormVoPage(page,settlementFormVo, settleDepartIdList);
	}

	/**
	 * @param salaryFormId 结算单id
	 * @Description: 工资详情-上面的统计
	 * @Author: hgw
	 * @Date: 2020-8-21 17:55:24
	 * @return: com.baomidou.mybatisplus.core.metadata.IPage<com.yifu.cloud.v1.hrms.api.vo.SettlementFormVo>
	 **/
	@Override
	public SettlementFormVo getSettlementFormVoBySalaryFormId(String salaryFormId) {
		return baseMapper.getSettlementFormVoBySalaryFormId(salaryFormId);
	}

	/**
	 * @param page
	 * @param salaryAccountAndItemVo
	 * @Description:  获取报账以及薪资
	 * @Author: hgw
	 * @Date: 2020/8/21 15:32
	 * @return: com.baomidou.mybatisplus.core.metadata.IPage<com.yifu.cloud.v1.hrms.api.vo.SalaryAccountAndItemVo>
	 **/
	@Override
	public IPage<SalaryAccountAndItemVo> getSalaryAccountAndItemVoPage(Page<SalaryAccountAndItemVo> page
			, SalaryAccountAndItemVo salaryAccountAndItemVo) {
		return baseMapper.getSalaryAccountAndItemVoPage(page, salaryAccountAndItemVo);
	}

	/**
	 * @param idList
	 * @Description: 根据idList获取报账信息
	 * @Author: hgw
	 * @Date: 2023/2/2 17:41
	 * @return: java.util.List<com.yifu.cloud.plus.v1.yifu.salary.entity.TSalaryAccount>
	 **/
	@Override
	public List<TSalaryAccount> getListByIdList(List<String> idList) {
		return baseMapper.getListByIdList(idList);
	}

	@Override
	public List<String> getAccountIdForProve(String startMouth, String endMouth, String emIdCard, String settleDepartId) {
		return baseMapper.getAccountIdForProve(startMouth, endMouth, emIdCard, settleDepartId);
	}
	@Override
	public List<String> getAccountMonthForProve(String startMouth, String endMouth, String emIdCard, String settleDepartId) {
		return baseMapper.getAccountMonthForProve(startMouth, endMouth, emIdCard, settleDepartId);
	}

	@Override
	public TSalaryAccountSumVo getAccountSumBySalaryId(String salaryId) {
		return baseMapper.getAccountSumBySalaryId(salaryId);
	}

	@Override
	public List<TSalaryAccount> allYearExport(TSalaryAccountSearchVo tSalaryAccount) {
		return baseMapper.allYearExport(tSalaryAccount);
	}

	@Override
	public List<TSalaryTypeThreeExportEkpVo> getAccountThreeListByApplyNo(String applyNo) {
		return baseMapper.getAccountThreeListByApplyNo(applyNo);
	}

	@Override
	public List<TSalaryTypeFourExportEkpVo> getAccountFourListByApplyNo(String applyNo) {
		return baseMapper.getAccountFourListByApplyNo(applyNo);
	}

	@Override
	public void exportError(HttpServletResponse response, String errorMessage) {
		String fileName = "导出异常" + DateUtil.getThisTime() + CommonConstants.XLSX;
		//获取要导出的列表
		List<ExportErrorVo> list = new ArrayList<>();
		list.add(new ExportErrorVo(errorMessage));
		long count = 0;
		ServletOutputStream out = null;
		try {
			out = response.getOutputStream();
			response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
			response.setCharacterEncoding("utf-8");
			response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName, "UTF-8"));
			// 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
			ExcelWriter excelWriter = EasyExcelFactory.write(out, ExportErrorVo.class).build();
			int index = 0;
			WriteSheet writeSheet = EasyExcelFactory.writerSheet("异常信息" + index).build();
			excelWriter.write(list, writeSheet);
			if (Common.isNotNull(list)) {
				list.clear();
			}
			out.flush();
			excelWriter.finish();
		} catch (Exception e) {
			log.error("执行异常", e);
		} finally {
			try {
				if (null != out) {
					out.close();
				}
			} catch (IOException e) {
				log.error("执行异常", e);
			}
		}
	}

	@Override
	public List<TSalaryAccount> getIdCardListBySalaryId(String salaryId) {
		return baseMapper.getIdCardListBySalaryId(salaryId);
	}

}
