/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TCompanyInfo;
import com.yifu.cloud.plus.v1.yifu.archives.service.TCompanyInfoService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.TCompanyInfoSearchVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.time.LocalDateTime;
import java.util.List;


/**
 * 收入证明 中的 单位信息维护表
 *
 * @author fxj
 * @date 2023-08-23 17:31:05
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/tcompanyinfo")
@Tag(name = "收入证明 中的 单位信息维护表管理")
public class TCompanyInfoController {

	private final TCompanyInfoService tCompanyInfoService;


	/**
	 * 简单分页查询
	 *
	 * @param page         分页对象
	 * @param tCompanyInfo 收入证明 中的 单位信息维护表
	 * @return
	 */
	@Operation(description = "简单分页查询")
	@GetMapping("/page")
	public R<IPage<TCompanyInfo>> getTCompanyInfoPage(Page<TCompanyInfo> page, TCompanyInfoSearchVo tCompanyInfo) {
		return new R<>(tCompanyInfoService.getTCompanyInfoPage(page, tCompanyInfo));
	}

	/**
	 * 不分页查询
	 *
	 * @param tCompanyInfo 收入证明 中的 单位信息维护表
	 * @return
	 */
	@Operation(summary = "不分页查询", description = "不分页查询")
	@PostMapping("/noPage")
	//@PreAuthorize("@pms.hasPermission('demo_tcompanyinfo_get')" )
	public R<List<TCompanyInfo>> getTCompanyInfoNoPage(@RequestBody TCompanyInfoSearchVo tCompanyInfo) {
		return R.ok(tCompanyInfoService.noPageDiy(tCompanyInfo));
	}

	/**
	 * 通过id查询收入证明 中的 单位信息维护表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询：hasPermission('demo_tcompanyinfo_get')")
	@GetMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('tcompanyinfo_get')")
	public R<TCompanyInfo> getById(@PathVariable("id") String id) {
		return R.ok(tCompanyInfoService.getById(id));
	}

	/**
	 * 新增收入证明 中的 单位信息维护表
	 *
	 * @param tCompanyInfo 收入证明 中的 单位信息维护表
	 * @return R
	 */
	@Operation(summary = "新增收入证明 中的 单位信息维护表", description = "新增收入证明 中的 单位信息维护表：hasPermission('tcompanyinfo_add')")
	@SysLog("新增收入证明 中的 单位信息维护表")
	@PostMapping
	@PreAuthorize("@pms.hasPermission('tcompanyinfo_add')")
	public R<Boolean> save(@RequestBody TCompanyInfo tCompanyInfo) {
		TCompanyInfo exits = tCompanyInfoService.getOne(Wrappers.<TCompanyInfo>query().lambda()
				.eq(TCompanyInfo::getDeleteFlag,CommonConstants.ZERO_STRING)
				.eq(TCompanyInfo::getCompanyName,tCompanyInfo.getCompanyName()).last(CommonConstants.LAST_ONE_SQL));
		if (Common.isNotNull(exits)){
			return R.failed("已存在相同公司名称的有效数据！");
		}
		tCompanyInfo.setCreateName(SecurityUtils.getUser().getNickname());
		tCompanyInfo.setCreateBy(SecurityUtils.getUser().getId());
		tCompanyInfo.setCreateTime(LocalDateTime.now());
		tCompanyInfo.setUpdateBy(SecurityUtils.getUser().getId());
		tCompanyInfo.setUpdateTime(LocalDateTime.now());
		return R.ok(tCompanyInfoService.save(tCompanyInfo));
	}
	/**
	 * 修改收入证明 中的 单位信息维护表
	 *
	 * @param tCompanyInfo 收入证明 中的 单位信息维护表
	 * @return R
	 */
	@Operation(summary = "修改收入证明 中的 单位信息维护表", description = "修改收入证明 中的 单位信息维护表：hasPermission('tcompanyinfo_edit')")
	@SysLog("修改收入证明 中的 单位信息维护表")
	@PutMapping
	@PreAuthorize("@pms.hasPermission('tcompanyinfo_edit')")
	public R<Boolean> updateById(@RequestBody TCompanyInfo tCompanyInfo) {
		TCompanyInfo exits = tCompanyInfoService.getOne(Wrappers.<TCompanyInfo>query().lambda()
				.eq(TCompanyInfo::getDeleteFlag,CommonConstants.ZERO_STRING)
				.ne(TCompanyInfo::getId,tCompanyInfo.getId())
				.eq(TCompanyInfo::getCompanyName,tCompanyInfo.getCompanyName()).last(CommonConstants.LAST_ONE_SQL));
		if (Common.isNotNull(exits)){
			return R.failed("已存在相同公司名称的有效数据！");
		}
		tCompanyInfo.setCreateTime(LocalDateTime.now());
		tCompanyInfo.setUpdateBy(SecurityUtils.getUser().getId());
		tCompanyInfo.setUpdateTime(LocalDateTime.now());
		return R.ok(tCompanyInfoService.updateById(tCompanyInfo));
	}

	/**
	 * 通过id删除收入证明 中的 单位信息维护表
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id删除收入证明 中的 单位信息维护表", description = "通过id删除收入证明 中的 单位信息维护表：hasPermission('tcompanyinfo_del')")
	@SysLog("通过id删除收入证明 中的 单位信息维护表")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('tcompanyinfo_del')")
	public R<Boolean> removeById(@PathVariable String id) {
		return R.ok(tCompanyInfoService.removeById(id));
	}

	/**
	 * 收入证明 中的 单位信息维护表 批量导入
	 *
	 * @author fxj
	 * @date 2023-08-23 17:31:05
	 **/
	@SneakyThrows
	@Operation(description = "批量新增收入证明 中的 单位信息维护表 hasPermission('tcompanyinfo-batch-import')")
	@SysLog("批量新增收入证明 中的 单位信息维护表")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('tcompanyinfo-batch-import')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file) {
		return tCompanyInfoService.importDiy(file.getInputStream());
	}

	/**
	 * 收入证明 中的 单位信息维护表 批量导出
	 *
	 * @author fxj
	 * @date 2023-08-23 17:31:05
	 **/
	@Operation(description = "导出收入证明 中的 单位信息维护表 hasPermission('tcompanyinfo-export')")
	@PostMapping("/export")
	@PreAuthorize("@pms.hasPermission('tcompanyinfo-export')")
	public void export(HttpServletResponse response, @RequestBody TCompanyInfoSearchVo searchVo) {
		tCompanyInfoService.listExport(response, searchVo);
	}
}
