/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.lock.annotation.Lock4j;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TEmpFamily;
import com.yifu.cloud.plus.v1.yifu.archives.service.TEmpFamilyService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.FamilySearchVO;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.util.RedisUtil;
import com.yifu.cloud.plus.v1.yifu.common.dapr.config.DaprCheckProperties;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.ArchivesDaprUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.util.Arrays;
import java.util.List;


/**
 * 员工家庭信息表
 *
 * @author fxj
 * @date 2022-06-18 09:39:46
 */
@EnableConfigurationProperties(DaprCheckProperties.class)
@Log4j2
@RestController
@RequiredArgsConstructor
@RequestMapping("/tempfamily" )
@Tag(name = "员工家庭信息表管理")
public class TEmpFamilyController {
    private final TEmpFamilyService tEmpFamilyService;
	@Autowired
	private final ArchivesDaprUtil archivesDaprUtil;
	int  i = 0;
    /**
     * 分页查询
     * @param page 分页对象
     * @param tEmpFamily 员工家庭信息表
     * @return
     */
    @Operation(description = "分页查询")
    @GetMapping("/page" )
    public R<IPage<TEmpFamily>> getTEmpFamilyPage(Page page, FamilySearchVO tEmpFamily) {
		return R.ok(tEmpFamilyService.pageDiy(page,tEmpFamily));
    }
	/**
	 * 不分页查询
	 * @param family 员工家庭信息表
	 * @return
	 */
	@Operation(description = "不分页查询")
	@PostMapping("/noPage" )
	public R<List<TEmpFamily>> getTEmpFamilyNoPage(@RequestBody FamilySearchVO family) {
		return R.ok(tEmpFamilyService.getTEmpFamilyNoPage(family));
	}

    /**
     * 通过id查询员工家庭信息表
     * @param id id
     * @return R
     */
    @Operation(description = "通过id查询hasPermission('archives-family-detail')")
    @GetMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('archives-family-detail')" )
    public R<TEmpFamily> getById(@PathVariable("id" ) String id) {
        return R.ok(tEmpFamilyService.getById(id));
    }

    /**
     * 新增员工家庭信息表  // TODO 新增变更档案学历最高学历信息
     * @param tEmpFamily 员工家庭信息表
     * @return R
     */
    @Operation(description = "新增员工家庭信息表hasPermission('archives-family-add')")
    @SysLog("新增员工家庭信息表" )
    @PostMapping
    @PreAuthorize("@pms.hasPermission('archives-family-add')" )
    public R<Boolean> save(@RequestBody TEmpFamily tEmpFamily) {
		String checkRes = tEmpFamilyService.checkRepeat(tEmpFamily.getEmpIdcard(),tEmpFamily.getFamilyName(),null);
		if (Common.isNotNull(checkRes)){
			return R.failed(checkRes);
		}
		tEmpFamily.setDeleteFlag(CommonConstants.ZERO_STRING);
        return R.ok(tEmpFamilyService.save(tEmpFamily));
    }

    /**
     * 修改员工家庭信息表
     * @param tEmpFamily 员工家庭信息表
     * @return R
     */
    @Operation(description = "修改员工家庭信息表hasPermission('archives-family-edit')")
    @SysLog("修改员工家庭信息表" )
    @PutMapping
    @PreAuthorize("@pms.hasPermission('archives-family-edit')" )
    public R<Boolean> updateById(@RequestBody TEmpFamily tEmpFamily) {
		String checkRes = tEmpFamilyService.checkRepeat(tEmpFamily.getEmpIdcard(),tEmpFamily.getFamilyName(),tEmpFamily.getId());
		if (Common.isNotNull(checkRes)){
			return R.failed(checkRes);
		}
        return R.ok(tEmpFamilyService.updateById(tEmpFamily));
    }

    /**
     * 通过id删除员工家庭信息表
     * @param id id
     * @return R
     */
    @Operation(description = "通过id删除员工家庭信息表hasPermission('archives-family-delete')")
    @SysLog("通过id删除员工家庭信息表" )
    @DeleteMapping("/{id}" )
    @PreAuthorize("@pms.hasPermission('archives-family-delete')" )
    public R<Boolean> removeById(@PathVariable String id) {
        return R.ok(tEmpFamilyService.removeById(id));
    }


	/**
	 * 通过id删除员工家庭信息表
	 * @param ids
	 * @return R
	 */
	@Operation(description = "通过id删除员工家庭信息表hasPermission('archives-family-batch-delete')")
	@SysLog("通过id删除员工家庭信息表" )
	@DeleteMapping("/removeByIds" )
	@PreAuthorize("@pms.hasPermission('archives-family-batch-delete')" )
	public R<Boolean> removeByIds(@RequestBody String[] ids) {
		List<String> idList = Arrays.asList(ids);
		return tEmpFamilyService.removeByIdsDiy(idList);
	}

	/**
	 * 批量新增员工家庭信息
	 *
	 * @param file
	 * @return
	 * @Author fxj
	 * @Date 2019-08-16
	 **/
	@SneakyThrows
	@Operation(description = "批量新增员工家庭信息 hasPermission('archives-family-batch-import')")
	@SysLog("批量新增员工家庭信息")
	@PostMapping("/importListAdd")
	@PreAuthorize("@pms.hasPermission('archives-family-batch-import')")
	public R<List<ErrorMessage>> importListAdd(@RequestBody MultipartFile file){
		return tEmpFamilyService.importEmpFamilyDiy(file.getInputStream());
	}
	@Autowired
	private RedisUtil redisUtil;
	@Lock4j(keys = {"#tEmpFamily.mId"}, expire = 60000, acquireTimeout = 1000)
	@Operation(description = "lockTest")
	@GetMapping("/lockTest" )
	public R<String> lockTest(Page page, FamilySearchVO tEmpFamily) {
		Object obj = redisUtil.get("lockTest");
		int lockValue;
		if (Common.isNotNull(obj)){
			lockValue = (int)obj ;
		}else {
			lockValue = 1;
		}
		lockValue = lockValue +1;
		redisUtil.set("lockTest",lockValue);
		return R.ok(CommonConstants.SUCCESS_MSG_PREFIX,String.valueOf(lockValue));
	}
}
