/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */

package com.yifu.cloud.plus.v1.yifu.archives.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yifu.cloud.plus.v1.yifu.archives.constants.ArchivesConstants;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeContractInfo;
import com.yifu.cloud.plus.v1.yifu.archives.service.TEmployeeContractInfoService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.EmpContractAuditVo;
import com.yifu.cloud.plus.v1.yifu.archives.vo.ErrorVO;
import com.yifu.cloud.plus.v1.yifu.archives.vo.TEmployeeContractSearchVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.ErrorMessage;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.dapr.util.MenuUtil;
import com.yifu.cloud.plus.v1.yifu.common.log.annotation.SysLog;
import com.yifu.cloud.plus.v1.yifu.common.security.annotation.Inner;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * 员工合同
 *
 * @author hgw
 * @date 2022-06-29 20:45:22
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/temployeecontractinfo")
@Tag(name = "员工合同管理")
public class TEmployeeContractInfoController {

	private final TEmployeeContractInfoService tEmployeeContractInfoService;
	private final MenuUtil menuUtil;
	/**
	 * 申请、归档分页查询
	 *
	 * @param page                  分页对象
	 * @param tEmployeeContractInfo 员工合同
	 * @return
	 */
	@Operation(summary = "申请分页查询（全部的数据（后面加上权限：自己创建的））", description = "申请分页查询")
	@GetMapping("/applyPage")
	public R<IPage<TEmployeeContractInfo>> getApplyPage(Page<TEmployeeContractInfo> page, TEmployeeContractInfo tEmployeeContractInfo) {
		this.setAuth(tEmployeeContractInfo);
		return new R<>(tEmployeeContractInfoService.getTEmployeeContractInfoPage(page, tEmployeeContractInfo));
	}

	@Operation(summary = "归档分页查询（审核通过的数据（后面加上权限：自己创建的））", description = "归档分页查询")
	@GetMapping("/filingPage")
	public R<IPage<TEmployeeContractInfo>> getFilingPage(Page<TEmployeeContractInfo> page, TEmployeeContractInfo tEmployeeContractInfo) {
		tEmployeeContractInfo.setIsFilePage(CommonConstants.ONE_INT);
		tEmployeeContractInfo.setAuditStatus(CommonConstants.dingleDigitIntArray[2]);
		this.setAuth(tEmployeeContractInfo);
		return new R<>(tEmployeeContractInfoService.getTEmployeeContractInfoPage(page, tEmployeeContractInfo));
	}

	/**
	 * @param tEmployeeContractInfo
	 * @Description: 加入权限
	 * @Author: hgw
	 * @Date: 2022/9/16 16:39
	 * @return: void
	 **/
	private void setAuth(TEmployeeContractInfo tEmployeeContractInfo) {
		YifuUser user = SecurityUtils.getUser();
		menuUtil.setAuthSql(user, tEmployeeContractInfo);
		if (Common.isNotNull(tEmployeeContractInfo.getAuthSql()) && tEmployeeContractInfo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
			tEmployeeContractInfo.setAuthSql(tEmployeeContractInfo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "a.SETTLE_DOMAIN"));
		}
	}

	/**
	 * 合同审核分页查询
	 *
	 * @param page                  分页对象
	 * @param tEmployeeContractInfo 员工合同
	 * @return
	 */
	@Operation(summary = "合同审核分页查询（待审核的数据）", description = "合同审核分页查询")
	@GetMapping("/auditPage")
	public R<IPage<TEmployeeContractInfo>> getAuditPage(Page<TEmployeeContractInfo> page, TEmployeeContractInfo tEmployeeContractInfo) {
		tEmployeeContractInfo.setAuditStatus(CommonConstants.ONE_INT);
		this.setAuth(tEmployeeContractInfo);
		return new R<>(tEmployeeContractInfoService.getTEmployeeContractInfoPage(page, tEmployeeContractInfo));
	}

	/**
	 * 分页查询
	 *
	 * @param page                  分页对象
	 * @param tEmployeeContractInfo 员工合同
	 * @return
	 */
	@Operation(summary = "合同查询（全部的数据）", description = "合同查询")
	@GetMapping("/page")
	public R<IPage<TEmployeeContractInfo>> gettemployeecontractinfoPage(Page<TEmployeeContractInfo> page, TEmployeeContractInfo tEmployeeContractInfo) {
		this.setAuth(tEmployeeContractInfo);
		return new R<>(tEmployeeContractInfoService.getTEmployeeContractInfoPage(page, tEmployeeContractInfo));
	}

	/**
	 * @param tEmployeeContractInfo
	 * @Description: 根据参数返回list
	 * @Author: hgw
	 * @Date: 2022/7/4 16:00
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.util.List < com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeContractInfo>>
	 **/
	@Operation(summary = "根据参数返回list", description = "根据参数返回list")
	@PostMapping("/getListByParam")
	public R<List<TEmployeeContractInfo>> getListByParam(@RequestBody TEmployeeContractInfo tEmployeeContractInfo) {
		return new R<>(tEmployeeContractInfoService.getTEmployeeContractList(tEmployeeContractInfo));
	}

	/**
	 * 通过id查询员工合同
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id查询", description = "通过id查询")
	@GetMapping("/{id}")
	public R<TEmployeeContractInfo> getById(@PathVariable("id") String id) {
		return R.ok(tEmployeeContractInfoService.getById(id));
	}

	/**
	 * 新增员工合同
	 *
	 * @param tEmployeeContractInfo 员工合同
	 * @return R
	 */
	@Operation(summary = "新增-员工合同auditStatus审核状态，前端传参（0：保存为待提交；1：提交为待审核）temployeecontractinfo_add附件传参attaList", description = "新增员工合同：hasPermission('temployeecontractinfo_add')")
	@SysLog("新增员工合同")
	@PostMapping("/saveNewContract")
	@PreAuthorize("@pms.hasPermission('temployeecontractinfo_add')")
	public R<List<ErrorMessage>> saveNewContract(@RequestBody TEmployeeContractInfo tEmployeeContractInfo) {
		return tEmployeeContractInfoService.saveNewContract(tEmployeeContractInfo);
	}

	/**
	 * 修改员工合同
	 *
	 * @param tEmployeeContractInfo 员工合同
	 * @return R
	 */
	@Operation(summary = "编辑-员工合同auditStatus审核状态，前端传参（0：保存为待提交；1：提交为待审核）temployeecontractinfo_edit编辑的附件，前端操作，要传参attaList", description = "修改员工合同：hasPermission('temployeecontractinfo_edit')")
	@SysLog("修改员工合同")
	@PostMapping("/updateContract")
	@PreAuthorize("@pms.hasPermission('temployeecontractinfo_edit')")
	public R<List<ErrorMessage>> updateContract(@RequestBody TEmployeeContractInfo tEmployeeContractInfo) {
		return tEmployeeContractInfoService.updateContract(tEmployeeContractInfo);
	}

	/**
	 * 通过id删除员工合同
	 *
	 * @param id id
	 * @return R
	 */
	@Operation(summary = "通过id假删除员工合同temployeecontractinfo_del", description = "通过id删除员工合同：hasPermission('temployeecontractinfo_del')")
	@SysLog("通过id假删除员工合同")
	@DeleteMapping("/{id}")
	@PreAuthorize("@pms.hasPermission('temployeecontractinfo_del')")
	public R<String> removeById(@PathVariable String id) {
		return tEmployeeContractInfoService.deleteContract(id);
	}

	/**
	 * 通过idList删除员工合同
	 *
	 * @param idList idList
	 * @return R
	 */
	@Operation(summary = "通过idList假删除员工合同temployeecontractinfo_del", description = "通过id删除员工合同：hasPermission('temployeecontractinfo_del')")
	@SysLog("通过idList假删除员工合同")
	@DeleteMapping("/batchDeleteContract")
	@PreAuthorize("@pms.hasPermission('temployeecontractinfo_del')")
	public R<List<ErrorVO>> batchDeleteContract(@RequestBody List<String> idList) {
		return tEmployeeContractInfoService.batchDeleteContract(idList);
	}

	/**
	 * 审核员工合同
	 * 审核通过后，变为在用，项目内其他的合同不再用
	 *
	 * @param tEmployeeContractInfo 员工合同
	 * @return R
	 */
	@Operation(summary = "审核-员工合同（主键：ID；审核意见auditStatus（2审核通过；3审核不通过）；审核说明auditRemark；）权限temployeecontractinfo_audit", description = "审核-员工合同")
	@SysLog("审核员工合同")
	@PostMapping("/auditContract")
	@PreAuthorize("@pms.hasPermission('temployeecontractinfo_audit')")
	public R<String> auditContract(@RequestBody TEmployeeContractInfo tEmployeeContractInfo) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return  R.failed(CommonConstants.USER_FAIL);
		}
		if (Common.isEmpty(tEmployeeContractInfo.getAuditStatus())) {
			return R.failed(ArchivesConstants.CONTRACT_AUDIT_STATUS_NOT_EMPTY);
		}
		if (Common.isNotNull(tEmployeeContractInfo.getAuditRemark()) && tEmployeeContractInfo.getAuditRemark().length() > 200) {
			return R.failed(ArchivesConstants.CONTRACT_AUDIT_REMARK_TOO_LONG);
		}
		return tEmployeeContractInfoService.auditContract(tEmployeeContractInfo,user);
	}
	/**
	 * @Author fxj
	 * @Description 批量审核员工合同
	 * @Date 16:21 2022/12/12
	**/
	@Operation(summary = "审核-员工合同（主键：ID；审核意见auditStatus（2审核通过；3审核不通过）；审核说明auditRemark；）权限temployeecontractinfo_audit", description = "审核-员工合同")
	@SysLog("批量审核员工合同")
	@PostMapping("/auditContractBatch")
	@PreAuthorize("@pms.hasPermission('temployeecontractinfo_audit')")
	public R<List<ErrorMessage>> auditContractBatch(@RequestBody EmpContractAuditVo contractAuditVo) {
		YifuUser user = SecurityUtils.getUser();
		if (Common.isEmpty(user)){
			return  R.failed(CommonConstants.USER_FAIL);
		}
		if (Common.isEmpty(contractAuditVo.getIds())) {
			return R.failed(CommonConstants.PARAM_IS_NOT_ERROR);
		}
		if (Common.isEmpty(contractAuditVo.getAuditStatus())) {
			return R.failed(ArchivesConstants.CONTRACT_AUDIT_STATUS_NOT_EMPTY);
		}
		if (Common.isNotNull(contractAuditVo.getAuditRemark()) && contractAuditVo.getAuditRemark().length() > 200) {
			return R.failed(ArchivesConstants.CONTRACT_AUDIT_REMARK_TOO_LONG);
		}
		return tEmployeeContractInfoService.auditContractBatch(contractAuditVo,user);
	}
	/**
	 * 审核员工合同
	 * 审核通过后，变为在用，项目内其他的合同不再用
	 *
	 * @param tEmployeeContractInfo 员工合同
	 * @return R
	 */
	@Operation(summary = "归档-员工合同（主键：ID；合同编号contractNo；档案柜号fileCabinetNo；附件List<String> attaList）权限temployeecontractinfo_filing", description = "归档-员工合同")
	@SysLog("归档员工合同")
	@PostMapping("/filingContract")
	@PreAuthorize("@pms.hasPermission('temployeecontractinfo_filing')")
	public R<String> filingContract(@RequestBody TEmployeeContractInfo tEmployeeContractInfo) {
		return tEmployeeContractInfoService.filingContract(tEmployeeContractInfo);
	}

	/**
	 * 批量申请-合同
	 *
	 * @param file
	 * @return
	 * @Author hgw
	 * @Date 2022-7-1 18:45:22
	 **/
	@SneakyThrows
	@Operation(description = "批量申请-合同 hasPermission('temployeecontractinfo_batch_import')")
	@SysLog("批量申请-合同")
	@PostMapping("/batchImport")
	@PreAuthorize("@pms.hasPermission('temployeecontractinfo_batch_import')")
	public R<List<ErrorMessage>> batchImport(@RequestBody MultipartFile file) {
		return tEmployeeContractInfoService.batchImport(file.getInputStream());
	}

	/**
	 * 批量更新-合同
	 *
	 * @param file
	 * @return
	 * @Author hgw
	 * @Date 2022-7-1 18:45:22
	 **/
	@SneakyThrows
	@Operation(description = "批量更新-合同 hasPermission('temployeecontractinfo_batch_update')")
	@SysLog("批量更新-合同")
	@PostMapping("/batchUpdate")
	@PreAuthorize("@pms.hasPermission('temployeecontractinfo_batch_update')")
	public R<List<ErrorMessage>> batchUpdate(@RequestBody MultipartFile file) {
		return tEmployeeContractInfoService.batchUpdate(file.getInputStream());
	}

	/**
	 * 批量导出合同
	 *
	 * @Author hgw
	 * @Date 2022-6-24 09:36:06
	 **/
	@Operation(summary = "批量导出合同employee_export_contract", description = "批量导出合同 hasPermission('employee_export_contract')")
	@SysLog("批量导出合同")
	@PostMapping("/exportContractInfo")
	public void exportContractInfo(@RequestBody(required = false) TEmployeeContractInfo contractInfo, HttpServletResponse response) {
		YifuUser user = SecurityUtils.getUser();
		if (user != null && !Common.isEmpty(user.getId())) {
			menuUtil.setAuthSql(user, contractInfo);
			if (Common.isNotNull(contractInfo.getAuthSql()) && contractInfo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
				contractInfo.setAuthSql(contractInfo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "a.SETTLE_DOMAIN"));
			}
		}
		tEmployeeContractInfoService.exportContractInfo(contractInfo, response);
	}

	/**
	 * 批量导出人员档案信息
	 *
	 * @Author hgw
	 * @Date 2022-6-24 09:36:06
	 **/
	@Operation(summary = "批量导出合同[历史]employee_export_contract_history", description = "批量导出合同[历史]hasPermission('employee_export_contract_history')")
	@SysLog("批量导出合同[历史]")
	@PostMapping("/exportContractInfoHistory")
	public void exportContractInfoHistory(@RequestBody(required = false) TEmployeeContractInfo contractInfo, HttpServletResponse response) {
		YifuUser user = SecurityUtils.getUser();
		if (user != null && !Common.isEmpty(user.getId())) {
			menuUtil.setAuthSql(user, contractInfo);
			if (Common.isNotNull(contractInfo.getAuthSql()) && contractInfo.getAuthSql().contains(CommonConstants.A_DEPT_ID)) {
				contractInfo.setAuthSql(contractInfo.getAuthSql().replace(CommonConstants.A_DEPT_ID, "a.SETTLE_DOMAIN"));
			}
		}
		tEmployeeContractInfoService.exportContractInfoHistory(contractInfo, response);
	}


	/**
	 * @Author fxj
	 * @Description  通过id查询员工合同
	 * @Date 10:41 2022/7/25
	 * @Param
	 * @return
	**/
	@Operation(summary = "通过id查询", description = "通过id查询员工合同")
	@Inner
	@PostMapping("/getContractById")
	public TEmployeeContractInfo getContractById(@RequestBody String id) {
		return tEmployeeContractInfoService.getById(id);
	}

	/**
	 * @Author fxj
	 * @Description 批量导出合同审核
	 * @Date 15:22 2022/11/4
	 * @Param
	 * @return
	**/
	@Operation(summary = "批量导出合同审核(记得传参mId)employee_export_contract", description = "批量导出合同审核 hasPermission('employee_export_contract_audit')")
	@SysLog("批量导出合同审核")
	@PostMapping("/exportAuditContractInfo")
	@PreAuthorize("@pms.hasPermission('employee_export_contract_audit')")
	public void exportAuditContractInfo(@RequestBody(required = false) TEmployeeContractSearchVo contractInfo,
										@RequestParam(required = false,name = "mId") String mId,
										HttpServletResponse response) {
		if (Common.isEmpty(contractInfo)){
			contractInfo = new TEmployeeContractSearchVo();
		}
		contractInfo.setMId(mId);
		contractInfo.setAuditStatus(CommonConstants.ONE_INT);
		contractInfo.setMId(mId);
		this.setAuth(contractInfo);
		tEmployeeContractInfoService.exportAuditContractInfo(contractInfo, response);
	}

	/**
	 * @param empIdCard 身份证
	 * @param deptId 项目id
	 * @Description: 通过身份证与项目id查询
	 * @Author: hgw
	 * @Date: 2023/3/9 14:30
	 * @return: com.yifu.cloud.plus.v1.yifu.common.core.util.R<java.lang.String>
	 **/
	@Operation(summary = "通过身份证与项目id查询", description = "通过身份证与项目id查询")
	@PostMapping("/getContractByEmpIdCardAndDeptId")
	public R<String> getContractByEmpIdCardAndDeptId(@RequestParam String empIdCard, @RequestParam String deptId) {
		return tEmployeeContractInfoService.getContractByEmpIdCardAndDeptId(empIdCard, deptId);
	}

}
