/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.archives.service.impl;

import cn.hutool.core.util.ArrayUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TCompleteMonitorSet;
import com.yifu.cloud.plus.v1.yifu.archives.mapper.TCompleteMonitorSetMapper;
import com.yifu.cloud.plus.v1.yifu.archives.service.TCompleteMonitorSetService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.TCompleteMonitorSetSearchVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.util.Common;
import com.yifu.cloud.plus.v1.yifu.common.core.util.R;
import com.yifu.cloud.plus.v1.yifu.common.core.vo.YifuUser;
import com.yifu.cloud.plus.v1.yifu.common.mybatis.base.BaseEntity;
import com.yifu.cloud.plus.v1.yifu.common.security.util.SecurityUtils;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 完整度监控配置
 *
 * @author hgw
 * @date 2025-02-26 15:18:51
 */
@Log4j2
@Service
public class TCompleteMonitorSetServiceImpl extends ServiceImpl<TCompleteMonitorSetMapper, TCompleteMonitorSet> implements TCompleteMonitorSetService {
	/**
	 * 完整度监控配置简单分页查询
	 *
	 * @param tCompleteMonitorSet 完整度监控配置
	 * @return
	 */
	@Override
	public IPage<TCompleteMonitorSet> getTCompleteMonitorSetPage(Page<TCompleteMonitorSet> page, TCompleteMonitorSetSearchVo tCompleteMonitorSet) {
		return baseMapper.getTCompleteMonitorSetPage(page, tCompleteMonitorSet);
	}

	@Override
	public List<TCompleteMonitorSet> noPageDiy(TCompleteMonitorSetSearchVo searchVo) {
		LambdaQueryWrapper<TCompleteMonitorSet> wrapper = buildQueryWrapper(searchVo);
		List<String> idList = Common.getList(searchVo.getIds());
		if (Common.isNotNull(idList)) {
			wrapper.in(TCompleteMonitorSet::getId, idList);
		}
		if (searchVo.getLimitStart() >= 0 && searchVo.getLimitEnd() > 0) {
			wrapper.last(" limit " + searchVo.getLimitStart() + "," + searchVo.getLimitEnd());
		}
		wrapper.orderByDesc(BaseEntity::getCreateTime);
		return baseMapper.selectList(wrapper);
	}

	private LambdaQueryWrapper<TCompleteMonitorSet> buildQueryWrapper(TCompleteMonitorSetSearchVo entity) {
		LambdaQueryWrapper<TCompleteMonitorSet> wrapper = Wrappers.lambdaQuery();
		if (ArrayUtil.isNotEmpty(entity.getCreateTimes())) {
			wrapper.ge(TCompleteMonitorSet::getCreateTime, entity.getCreateTimes()[0])
					.le(TCompleteMonitorSet::getCreateTime,
							entity.getCreateTimes()[1]);
		}
		if (Common.isNotNull(entity.getCreateName())) {
			wrapper.eq(TCompleteMonitorSet::getCreateName, entity.getCreateName());
		}
		return wrapper;
	}

	/**
	 * @Description: 获取在用的员工合同类型
	 * @Author: hgw
	 * @Date: 2025/2/27 16:16
	 * @return: java.lang.String
	 **/
	@Override
	public String getContractType() {
		return baseMapper.getAllUseContractType(null);
	}

	/**
	 * @Description: 新增或编辑
	 * @Author: hgw
	 * @Date: 2025/2/27 16:16
	 * @return: java.lang.String
	 **/
	@Override
	public R<String> saveOrUpdateInfo(TCompleteMonitorSet tCompleteMonitorSet) {
		if (Common.isEmpty(tCompleteMonitorSet.getContractType()) || Common.isEmpty(tCompleteMonitorSet.getContractTypeName())) {
			return R.failed("请传合同类型与合同类型名称");
		}
		if (Common.isEmpty(tCompleteMonitorSet.getIsMonitor())) {
			return R.failed("请传是否监控0否1是");
		}
		if (Common.isEmpty(tCompleteMonitorSet.getBaseId())) {
			return R.failed("请传BaseId，为空传-1");
		}
		String allUseContractType = baseMapper.getAllUseContractType(tCompleteMonitorSet.getId());
		if (Common.isNotNull(allUseContractType)) {
			String[] contractTypeArr = tCompleteMonitorSet.getContractType().split(",");
			String[] contractTypeNameArr = tCompleteMonitorSet.getContractTypeName().split(",");
			if (contractTypeArr.length != contractTypeNameArr.length) {
				return R.failed("请检查合同类型与合同类型名称的个数是否一致");
			}
			String[] allUseContractTypeArr = allUseContractType.split(",");
			Map<String, Integer> curMap = new HashMap<>();
			for (String str : allUseContractTypeArr) {
				curMap.put(str, CommonConstants.ONE_INT);
			}
			String curContractType = null;
			for (String str : contractTypeArr) {
				if (curMap.get(str) != null) {
					if (Common.isEmpty(curContractType)) {
						curContractType = str;
					} else {
						curContractType += "、" + str;
					}
				}
			}
			if (Common.isNotNull(curContractType)) {
				return R.failed("员工合同类型已存在");
			}
		}
		YifuUser user = SecurityUtils.getUser();
		if (user == null || Common.isEmpty(user.getNickname())) {
			return R.failed("请登录！");
		}
		tCompleteMonitorSet.setUpdateTime(LocalDateTime.now());
		tCompleteMonitorSet.setUpdateName(user.getNickname());
		this.saveOrUpdate(tCompleteMonitorSet);
		return R.ok(CommonConstants.SAVE_SUCCESS);
	}

}
