/*
 *    Copyright (c) 2018-2025, lengleng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the yifu4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: lengleng (wangiegie@gmail.com)
 */
package com.yifu.cloud.plus.v1.yifu.archives.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ArrayUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.read.listener.ReadListener;
import com.alibaba.excel.read.metadata.holder.ReadRowHolder;
import com.alibaba.excel.util.ListUtils;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TEmpBadRecord;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeInfo;
import com.yifu.cloud.plus.v1.yifu.archives.entity.TEmployeeProject;
import com.yifu.cloud.plus.v1.yifu.archives.mapper.TAttaInfoMapper;
import com.yifu.cloud.plus.v1.yifu.archives.mapper.TEmpBadRecordMapper;
import com.yifu.cloud.plus.v1.yifu.archives.mapper.TEmployeeInfoMapper;
import com.yifu.cloud.plus.v1.yifu.archives.mapper.TEmployeeProjectMapper;
import com.yifu.cloud.plus.v1.yifu.archives.service.TEmpBadRecordService;
import com.yifu.cloud.plus.v1.yifu.archives.vo.EmpBadRecordSearchVo;
import com.yifu.cloud.plus.v1.yifu.archives.vo.EmpBadRecordVo;
import com.yifu.cloud.plus.v1.yifu.common.core.constant.CommonConstants;
import com.yifu.cloud.plus.v1.yifu.common.core.exception.ErrorCodes;
import com.yifu.cloud.plus.v1.yifu.common.core.util.*;
import com.yifu.cloud.plus.v1.yifu.common.mybatis.base.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Service;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * 员工不良记录
 *
 * @author fxj
 * @date 2022-06-29 15:50:45
 */
@Log4j2
@RequiredArgsConstructor
@Service
public class TEmpBadRecordServiceImpl extends ServiceImpl<TEmpBadRecordMapper, TEmpBadRecord> implements TEmpBadRecordService {
	private final TAttaInfoMapper attaInfoMapper;
	private final TEmployeeInfoMapper employeeInfoMapper;
	private final TEmployeeProjectMapper projectMapper;

	@Override
	public R<List<ErrorMessage>> importEmpBadRecordDiy(InputStream inputStream) {
		List<ErrorMessage> errorMessageList = new ArrayList<>();
		ExcelUtil<EmpBadRecordVo> util1 = new ExcelUtil<>(EmpBadRecordVo.class);;
		// 写法2：
		// 匿名内部类 不用额外写一个DemoDataListener
		// 这里 需要指定读用哪个class去读，然后读取第一个sheet 文件流会自动关闭
		try {
			EasyExcel.read(inputStream, EmpBadRecordVo.class, new ReadListener<EmpBadRecordVo>() {
				/**
				 * 单次缓存的数据量
				 */
				public static final int BATCH_COUNT = CommonConstants.BATCH_COUNT;
				/**
				 *临时存储
				 */
				private List<EmpBadRecordVo> cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);

				@Override
				public void invoke(EmpBadRecordVo data, AnalysisContext context) {
					ReadRowHolder readRowHolder = context.readRowHolder();
					Integer rowIndex = readRowHolder.getRowIndex();
					data.setRowIndex(rowIndex+1);
					ErrorMessage errorMessage = util1.checkEntity(data, data.getRowIndex());
					if (Common.isNotNull(errorMessage)){
						errorMessageList.add(errorMessage);
					}else {
						cachedDataList.add(data);
					}
					if (cachedDataList.size() >= BATCH_COUNT) {
						saveData();
						// 存储完成清理 list
						cachedDataList = ListUtils.newArrayListWithExpectedSize(BATCH_COUNT);
					}
				}
				@Override
				public void doAfterAllAnalysed(AnalysisContext context) {
					saveData();
				}

				/**
				 * 加上存储数据库
				 */
				private void saveData() {
					log.info("{}条数据，开始存储数据库！", cachedDataList.size());
					importEmpBadRecord(cachedDataList,errorMessageList);
					log.info("存储数据库成功！");
				}
			}).sheet().doRead();
		}catch (Exception e){
			log.error(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR,e);
			return R.failed(CommonConstants.IMPORT_DATA_ANALYSIS_ERROR);
		}
		return  R.ok(errorMessageList);
	}

	@Override
	public R<Boolean> saveDiy(TEmpBadRecord tEmpBadRecord) {
		int  res = baseMapper.insert(tEmpBadRecord);
		if (res < 0){
			return R.failed();
		}
		//更新附件信息
		List<String> ids;
		if (Common.isNotNull(tEmpBadRecord.getAttaIds())){
			ids = Common.initStrToList(tEmpBadRecord.getAttaIds(),CommonConstants.COMMA_STRING);
			attaInfoMapper.updateDomainId(tEmpBadRecord.getId(),ids);
		}
		return R.ok();
	}

	private void importEmpBadRecord(List<EmpBadRecordVo> excelVOList, List<ErrorMessage> errorMessageList) {
		// 个性化校验逻辑
		TEmployeeInfo emp;
		TEmployeeProject exist;
		ErrorMessage errorMsg;
		// 执行数据插入操作 组装
		for (int i = 0; i < excelVOList.size(); i++) {
			// 已有验证报错直接下一个
			EmpBadRecordVo excel = excelVOList.get(i);
			// 【是否有不良记录】为是，项目编码和描述必填
			if(CommonConstants.ZERO_STRING.equals(excel.getBadRecordFlag())){
				if (Common.isEmpty(excel.getProjectCode())){
					errorMessageList.add(new ErrorMessage(excel.getRowIndex(),"产生项目编码_字段不可为空"));
					continue;
				}
				if (Common.isEmpty(excel.getRemark())){
					errorMessageList.add(new ErrorMessage(excel.getRowIndex(),"不良记录描述_字段不可为空"));
					continue;
				}
			}
			/*if (Common.isEmpty(excel.getLoseFee()) && Common.isEmpty(excel.getLoseFeeOther())){
				errorMessageList.add(new ErrorMessage(excel.getRowIndex(),MsgUtils.getMessage(ErrorCodes.ARCHIVES_PROJECT_EMP_LOSE_FEE_NOT_EMPTY)));
				continue;
			}*/
			emp = employeeInfoMapper.selectOne(Wrappers.<TEmployeeInfo>query().lambda()
					.eq(TEmployeeInfo::getEmpIdcard,excel.getEmpIdcard())
					.eq(TEmployeeInfo::getDeleteFlag,CommonConstants.ZERO_STRING)
					.last(CommonConstants.LAST_ONE_SQL));
			errorMsg = checkRes(emp, excel);
			if (Common.isNotNull(errorMsg)){
				errorMessageList.add(errorMsg);
				continue;
			}
			excel.setEmpName(emp.getEmpName());
			// mvp1.7.6-chenyuxi:有不良记录,再获取项目信息
			if(CommonConstants.ZERO_STRING.equals(excel.getBadRecordFlag())){
				if (Common.isEmpty(excel.getProjectCode())){
					errorMessageList.add(new ErrorMessage(excel.getRowIndex(),MsgUtils.getMessage(ErrorCodes.ARCHIVES_PROJECT_EMP_NOT_EXIST)));
					continue;
				}
				exist = projectMapper.selectOne(Wrappers.<TEmployeeProject>query().lambda()
						.eq(TEmployeeProject::getDeptNo,excel.getProjectCode())
						.eq(TEmployeeProject::getProjectStatus,CommonConstants.ZERO_STRING)
						.eq(TEmployeeProject::getEmpIdcard,excel.getEmpIdcard())
						.last(CommonConstants.LAST_ONE_SQL));
				if (Common.isEmpty(exist)){
					errorMessageList.add(new ErrorMessage(excel.getRowIndex(),MsgUtils.getMessage(ErrorCodes.ARCHIVES_PROJECT_EMP_NOT_EXIST)));
					continue;
				}
				excel.setProjectCode(exist.getDeptNo());
				excel.setProject(exist.getDeptName());
			} else {
				excel.setRemark("未曾经或正在追究与承担过刑事责任");
			}

			// 数据合法情况
			insertExcel(excel,emp);
			errorMessageList.add(new ErrorMessage(excel.getRowIndex(),CommonConstants.SAVE_SUCCESS));
		}
	}
	private ErrorMessage checkRes(TEmployeeInfo emp, EmpBadRecordVo excel) {
		if (Common.isEmpty(emp)){
			return new ErrorMessage(excel.getRowIndex(),MsgUtils.getMessage(ErrorCodes.ARCHIVES_EMP_NOT_EXIST));
		}
		if (CommonConstants.ONE_STRING.equals(emp.getFileStatus())){
			return new ErrorMessage(excel.getRowIndex(),MsgUtils.getMessage(ErrorCodes.ARCHIVES_EMP_REDUCED));
		}
		return null;
	}
	/**
	 * 插入excel bad record
	 */
	private void insertExcel(EmpBadRecordVo excel, TEmployeeInfo emp) {
		TEmpBadRecord insert = new TEmpBadRecord();
		BeanUtil.copyProperties(excel, insert);
		insert.setEmpIdcard(emp.getEmpIdcard());
		insert.setEmpId(emp.getId());
		insert.setEmpId(emp.getId());
		insert.setSettleDomain(emp.getDeptId());

		// mvp1.7.6-chenyuxi:无不良记录,置空部分信息
		if(CommonConstants.ONE_STRING.equals(excel.getBadRecordFlag())) {
			insert.setProjectCode(null);
			insert.setProject(null);
			insert.setHappenTime(null);
			insert.setLoseFee(null);
			insert.setLoseFeeOther(null);
		}
		this.save(insert);
	}

	@Override
	public void listExport(HttpServletResponse response, EmpBadRecordSearchVo searchVo) {
		String fileName = "不良记录批量导出" + DateUtil.getThisTime() + CommonConstants.XLSX;
		//获取要导出的列表
		List<TEmpBadRecord> list = new ArrayList<>();
		long count = noPageCountDiy(searchVo);
		ServletOutputStream out = null;
		try {
			out = response.getOutputStream();
			response.setContentType(CommonConstants.MULTIPART_FORM_DATA);
			response.setCharacterEncoding("utf-8");
			response.setHeader(CommonConstants.CONTENT_DISPOSITION, CommonConstants.ATTACHMENT_FILENAME + URLEncoder.encode(fileName  , "UTF-8"));
			// 这里 需要指定写用哪个class去写，然后写到第一个sheet，然后文件流会自动关闭
			//EasyExcel.write(out, TEmpBadRecord.class).sheet("不良记录").doWrite(list);
			ExcelWriter  excelWriter = EasyExcel.write(out, TEmpBadRecord.class).build();
			int index = 0;
			if (count >  CommonConstants.ZERO_INT){
				WriteSheet writeSheet;
				ExcelUtil<TEmpBadRecord> util;
				for (int i = 0; i <= count; ) {
					// 获取实际记录
					searchVo.setLimitStart(i);
					searchVo.setLimitEnd(CommonConstants.EXCEL_EXPORT_LIMIT);
					list = noPageDiy(searchVo);
					if (Common.isNotNull(list)){
						util = new ExcelUtil<>(TEmpBadRecord.class);
						for (TEmpBadRecord vo:list){
							util.convertEntity(vo,null,null,null);
						}
					}
					if (Common.isNotNull(list)){
						writeSheet = EasyExcel.writerSheet("不良记录"+index).build();
						excelWriter.write(list,writeSheet);
						index++;
					}
					i = i + CommonConstants.EXCEL_EXPORT_LIMIT;
					if (Common.isNotNull(list)){
						list.clear();
					}
				}
			}else {
				WriteSheet writeSheet = EasyExcel.writerSheet("不良记录"+index).build();
				excelWriter.write(list,writeSheet);
			}
			if (Common.isNotNull(list)){
				list.clear();
			}
			out.flush();
			excelWriter.finish();
		}catch (Exception e){
			log.error("执行异常" ,e);
		}finally {
			try {
				if (null != out) {
					out.close();
				}
			} catch (IOException e) {
				log.error("执行异常", e);
			}
		}
	}

	@Override
	public IPage<TEmpBadRecord> pageDiy(Page page, EmpBadRecordSearchVo tEmpBadRecord) {
		LambdaQueryWrapper<TEmpBadRecord> wrapper = buildQueryWrapper(tEmpBadRecord);
		wrapper.orderByDesc(BaseEntity::getCreateTime);
		return baseMapper.selectPage(page,wrapper);
	}

	@Override
	public TEmpBadRecord getByEmpId(String empId) {
		return baseMapper.selectOne(Wrappers.<TEmpBadRecord>query().lambda().eq(TEmpBadRecord::getEmpId, empId)
				.orderByDesc(TEmpBadRecord::getCreateTime).last(CommonConstants.LAST_ONE_SQL));
	}

	@Override
	public List<TEmpBadRecord> noPageDiy(EmpBadRecordSearchVo tEmpBadRecord) {
		LambdaQueryWrapper<TEmpBadRecord> wrapper = buildQueryWrapper(tEmpBadRecord);
		List<String> idList = Common.getList(tEmpBadRecord.getIds());
		if (Common.isNotNull(idList)){
			wrapper.in(TEmpBadRecord::getId,idList);
		}
		if (tEmpBadRecord.getLimitStart() >= 0 && tEmpBadRecord.getLimitEnd() > 0){
			wrapper.last(" limit "+ tEmpBadRecord.getLimitStart() +","+ tEmpBadRecord.getLimitEnd());
		}
		wrapper.orderByDesc(BaseEntity::getCreateTime);
		return baseMapper.selectList(wrapper);
	}

	private  Long noPageCountDiy(EmpBadRecordSearchVo tEmpBadRecord) {
		LambdaQueryWrapper<TEmpBadRecord> wrapper = buildQueryWrapper(tEmpBadRecord);
		List<String> idList = Common.getList(tEmpBadRecord.getIds());
		if (Common.isNotNull(idList)){
			wrapper.in(TEmpBadRecord::getId,idList);
		}
		return baseMapper.selectCount(wrapper);
	}

	private LambdaQueryWrapper buildQueryWrapper(EmpBadRecordSearchVo entity){
		LambdaQueryWrapper<TEmpBadRecord> wrapper = Wrappers.lambdaQuery();
		if (ArrayUtil.isNotEmpty(entity.getCreateTimes())) {
			wrapper.ge(TEmpBadRecord::getCreateTime, entity.getCreateTimes()[0])
					.le(TEmpBadRecord::getCreateTime,
							entity.getCreateTimes()[1]);
		}
		if (Common.isNotNull(entity.getEmpName())){
			wrapper.likeRight(TEmpBadRecord::getEmpName,entity.getEmpName());
		}
		if (Common.isNotNull(entity.getEmpIdcard())){
			wrapper.likeRight(TEmpBadRecord::getEmpIdcard,entity.getEmpIdcard());
		}
		if (Common.isNotNull(entity.getCreateName())){
			wrapper.eq(TEmpBadRecord::getCreateName,entity.getCreateName());
		}
		// 1.9.6 加【是否有不良记录】标识查询
		if (Common.isNotNull(entity.getBadRecordFlag())){
			wrapper.eq(TEmpBadRecord::getBadRecordFlag,entity.getBadRecordFlag());
		}
		return wrapper;
	}

	@Override
	public List<TEmpBadRecord> getBadByIdCard(List<String> idList) {
		return baseMapper.getBadByIdCard(idList);
	}

	@Override
	public R<Boolean> updateDiy(TEmpBadRecord tEmpBadRecord) {
		if(CommonConstants.ONE_STRING.equals(tEmpBadRecord.getBadRecordFlag())){
			LambdaUpdateWrapper<TEmpBadRecord> updateDiyInfoWrapper = new LambdaUpdateWrapper<>();
			updateDiyInfoWrapper.eq(TEmpBadRecord::getId, tEmpBadRecord.getId())
					.set(TEmpBadRecord::getBadRecordFlag, tEmpBadRecord.getBadRecordFlag())
					.set(TEmpBadRecord::getRemark, tEmpBadRecord.getRemark())
					.set(TEmpBadRecord::getHappenTime, null)
					.set(TEmpBadRecord::getLoseFee, null)
					.set(TEmpBadRecord::getLoseFeeOther, null)
					.set(TEmpBadRecord::getProject, null)
					.set(TEmpBadRecord::getProjectCode, null)
					.set(TEmpBadRecord::getUpdateTime, LocalDateTime.now());
			this.update(null, updateDiyInfoWrapper);

			return R.ok(true);
		}
		return R.ok(this.updateById(tEmpBadRecord));
	}
}
